# Copyright (c) 2015, Hubert Kario
#
# See the LICENSE file for legal information regarding use of this file.

# compatibility with Python 2.6, for that we need unittest2 package,
# which is not available on 3.3 or 3.4
try:
    import unittest2 as unittest
except ImportError:
    import unittest

from tlslite.utils.rsakey import RSAKey
from tlslite.utils.python_rsakey import Python_RSAKey
from tlslite.utils.cryptomath import *
from tlslite.errors import *
try:
    import mock
    from mock import call
except ImportError:
    import unittest.mock as mock
    from unittest.mock import call

class TestRSAPSS_components(unittest.TestCase):
    # component functions NOT tested from test vectors

    def setUp(self):
        self.rsa = Python_RSAKey()

    def test_unknownRSAType(self):
        message = bytearray(b'\xad\x8f\xd1\xf7\xf9' +
                                                 b'\x7fgrRS\xce}\x18\x985' +
                                                 b'\xb3')
        signed = bytearray(
                             b'\xb80\x12s\xbb\xd9j\xce&U\x08\x14\xb2\x070' +
                             b'\xc7\xc8\xa8\xa0\xc1\xc3\xf3\xd41\xad\xbe' +
                             b'\xe8\x1dN\x94\xf6sx\x02\xed\xfb\x0b\x0b\x85' +
                             b'\xc5N\xff\x04z\xec\x13\x86O\x15\xe8|\xae\xc6' +
                             b'\x1c\r\xcd\xec\xf4\xb1\xb5$\xf2\x17\xff\xf6' +
                             b'\xc2\xf5\xd2\x8a\xd2\x98\xa8\xb7\xe0;\xab\xe0' +
                             b'\xe9P\xd9\xea\x86\xb3\xeb)\xa3\x98\xb4e\xb5P' +
                             b'\x07\x14\xf1?\xa8i\xb7\xc6\x94\x1c9\x1fX>@' +
                             b'\xe3')
        with self.assertRaises(UnknownRSAType):
            self.rsa.hashAndVerify(message, signed, rsaScheme='foo',
                                   hAlg='sha1')

    def test_encodingError(self):
        mHash = secureHash(
                bytearray(b'\xc7\xf5\'\x0f\xcar_\x9b\xd1\x9fQ\x9a\x8d|\xca<' +
                          b'\xc5\xc0y\x02@)\xf3\xba\xe5\x10\xf9\xb0!@\xfe#' +
                          b'\x89\x08\xe4\xf6\xc1\x8f\x07\xa8\x9ch|\x86\x84f' +
                          b'\x9b\x1f\x1d\xb2\xba\xf9%\x1a<\x82\x9f\xac\xcbI0' +
                          b'\x84\xe1n\xc9\xe2\x8dX\x86\x80t\xa5\xd6"\x16g' +
                          b'\xddnR\x8d\x16\xfe,\x9f=\xb4\xcf\xaflM\xce\x8c' +
                          b'\x849\xaf8\xce\xaa\xaa\x9c\xe2\xec\xae{\xc8\xf4' +
                          b'\xa5\xa5^;\xf9m\xf9\xcdW\\O\x9c\xb3\'\x95\x1b' +
                          b'\x8c\xdf\xe4\x08qh'),
                'sha1')
        with self.assertRaises(EncodingError):
            self.assertEqual(self.rsa.EMSA_PSS_encode(mHash, 10, 'sha1', 10),
                bytearray(b'eA=!Fq4\xce\xef5?\xf4\xec\xd8\xa6FPX\xdc~(\xe3' +
                          b'\x92\x17z\xa5-\xcfV\xd4)\x99\x8fJ\xb2\x08\xa2<Q' +
                          b'\x02e\xb4\xe0\xecq\xa3:\xe0I\x1f\x83\x9f\xe2\xf4' +
                          b'\xb9\x89\x9b\xdbv\xb8\xb1&\r\xa8\xdfA\x13\xfd' +
                          b'\xed\xce\x85\xb5\x0e\xae\xd0t\xe5`\xbaen"' +
                          b'\xcd_=:V\'A\xe32u%7\x8b\xed\x16\xc7$\x919\xa5' +
                          b'\x18XA\xa3\xf5r\xad\x8f\xd1\xf7\xf9\x7f;\x01' +
                          b'\xccgrRS\xce}\x18\x985\xb3\xbc'))

    def test_maskTooLong(self):
        with self.assertRaises(MaskTooLongError):
            self.rsa.MGF1(bytearray(b'\xad\x8f\xd1\xf7\xf9\x7fgrRS\xce}\x18' +
                                    b'\x985\xb3'), 85899345921, 'sha1')

    def test_EMSA_PSS_verifyInvalidSignature29(self):
        with self.assertRaises(InvalidSignature):
            self.rsa.EMSA_PSS_verify(
                bytearray(b'\x6b\x9c\xfa\xc0\xba\x1c\x78\x90\xb1\x3e\x38\x1c' +
                          b'\xe7\x52\x19\x5c\xc1\x37\x52\x37\xdb\x2a\xfc\xf6' +
                          b'\xa9\xdc\xd1\xf9\x5e\xc7\x33\xa8\x0c\xc1\x70\xad' +
                          b'\x0d\xd2\x44\x61\xa3\x43\x9f\x83\xdb\xf7\xa9\x30' +
                          b'\xb7\xff\x03\x0f\x8b\x68\x50\xcd\xa8\x57\xf5\xc9' +
                          b'\x38\x59\x1c\x10\x69\xf4\xff\x8b\xce\x82\x63\x77' +
                          b'\x02\x3f\x62\x16\x8e\xa0\x6e\xa4\x0c\xe6\x93\x8d' +
                          b'\x8d\xf4\xfc\xdd\x39\x5d\xb9\xf8\x35\xc2\xc0\x8b' +
                          b'\xca\x5b\x31\xc2\xc3\xd3\xde\xbc\x16\x59\x0a\x3c' +
                          b'\xc7\xf4\x71\xa1\x0a\x16\x89\x18\xa8\x75\x08\x10' +
                          b'\xa3\x16\xa6\xee\x1d\xaa\x24\x02'),
                bytearray(b'\x56\x2d\x87\xb5\x78\x1c\x01\xd1\x66\xfe\xf3\x97' +
                          b'\x26\x69\xa0\x49\x5c\x14\x5b\x89\x8a\x17\xdf\x47' +
                          b'\x43\xfb\xef\xb0\xa1\x58\x2b\xd6\xba\x9d\xf1\xb0' +
                          b'\xf7\x0f\xa4\xb4\x48\x5a\x75\x7d\x91\x74\x97\x2e' +
                          b'\x69\x19\x29\xf4\xc1\xd3\x45\x19\xbc\xcc\x7a\x5c' +
                          b'\x4c\x8b\xca\x14\x4e\x58\xbe\x08\x39\xe8\x7e\x1e' +
                          b'\x9f\xe9\x01\xf0\xda\x40\x29\xb3\x1e\x78\x00\xdc' +
                          b'\x79\xe8\xc1\xbd\x68\xaf\x41\xc3\x93\xf8\x11\xe9' +
                          b'\x9e\xec\xdb\x52\xf6\x97\x8b\x18\x42\xe1\x70\xc0' +
                          b'\x15\x3e\xba\x5f\x18\x45\xff\xcd\x36\xff\x3f\x79' +
                          b'\x95\x53\x9f\x3c\x74\xde\x3c\xaa'),
                1023, 'sha1', 10)

    def test_EMSA_PSS_verifyInvalidSignature(self):
        with self.assertRaises(InvalidSignature):
            self.rsa.EMSA_PSS_verify(
                bytearray(b'\xc7\xf5\'\x0f\xcar_\x9b\xd1\x9fQ\x9a\x8d|\xca<' +
                          b'\xc5\xc0y\x02@)\xf3\xba\xe5\x10\xf9\xb0!@\xfe#' +
                          b'\x89\x08\xe4\xf6\xc1\x8f\x07\xa8\x9ch|\x86\x84f' +
                          b'\x9b\x1f\x1d\xb2\xba\xf9%\x1a<\x82\x9f\xac\xcbI0' +
                          b'\x84\xe1n\xc9\xe2\x8dX\x86\x80t\xa5\xd6"\x16g' +
                          b'\xddnR\x8d\x16\xfe,\x9f=\xb4\xcf\xaflM\xce\x8c' +
                          b'\x849\xaf8\xce\xaa\xaa\x9c\xe2\xec\xae{\xc8\xf4' +
                          b'\xa5\xa5^;\xf9m\xf9\xcdW\\O\x9c\xb3\'\x95\x1b' +
                          b'\x8c\xdf\xe4\x08qh'),
                bytearray(b'eA=!Fq4\xce\xef5?\xf4\xec\xd8\xa6FPX\xdc~(\xe3' +
                          b'\x92\x17z\xa5-\xcfV\xd4)\x99\x8fJ\xb2\x08\xa2<Q' +
                          b'\x02e\xb4\xe0\xecq\xa3:\xe0I\x1f\x83\x9f\xe2' +
                          b'\xf4\xb9\x89\x9b\xdbv\xb8\xb1&\r\xa8\xdfA\x13' +
                          b'\xfd\xed\xce\x85\xb5\x0e\xae\xd0t\xe5`\xbaen"' +
                          b'\xcd_=:V\'A\xe32u%7\x8b\xed\x16\xc7$\x919\xa5' +
                          b'\x18XA\xa3\xf5r\xad\x8f\xd1\xf7\xf9\x7f;\x01' +
                          b'\xccgrRS\xce}\x18\x985\xb3\xbc'),
                10, 'sha1', 10)

    def test_EMSA_PSS_verifyInvalidSignature2(self):
        with self.assertRaises(InvalidSignature):
            self.rsa.EMSA_PSS_verify(
                bytearray(b'\xc7\xf5\'\x0f\xcar_\x9b\xd1\x9fQ\x9a\x8d|\xca<' +
                          b'\xc5\xc0y\x02@)\xf3\xba\xe5\x10\xf9\xb0!@\xfe#' +
                          b'\x89\x08\xe4\xf6\xc1\x8f\x07\xa8\x9ch|\x86\x84f' +
                          b'\x9b\x1f\x1d\xb2\xba\xf9%\x1a<\x82\x9f\xac\xcbI0' +
                          b'\x84\xe1n\xc9\xe2\x8dX\x86\x80t\xa5\xd6"\x16g' +
                          b'\xddnR\x8d\x16\xfe,\x9f=\xb4\xcf\xaflM\xce\x8c' +
                          b'\x849\xaf8\xce\xaa\xaa\x9c\xe2\xec\xae{\xc8\xf4' +
                          b'\xa5\xa5^;\xf9m\xf9\xcdW\\O\x9c\xb3\'\x95\x1b' +
                          b'\x8c\xdf\xe4\x08qh'),
                bytearray(b'eA=!Fq4\xee\xef5?\xf4\xec\xd8\xa6FPX\xdc~(\xe3' +
                          b'\x92\x17z\xa5-\xcfV\xd4)\x99\x8fJ\xb2\x08\xa2<Q' +
                          b'\x02e\xb4\xe0\xecq\xa3:\xe0I\x1f\x83\x9f\xe2' +
                          b'\xf4\xb9\x89\x9b\xdbv\xb8\xb1&\r\xa8\xdfA\x13' +
                          b'\xfd\xed\xce\x85\xb5\x0e\xae\xd0t\xe5`\xbaen"' +
                          b'\xcd_=:V\'A\xe32u%7\x8b\xed\x16\xc7$\x919\xa5' +
                          b'\x18XA\xa3\xf5r\xad\x8f\xd1\xf7\xf9\x7f;\x01' +
                          b'\xccgrRS\xce}\x18\x985\xb3\xbc'),
                1023, 'sha1', 10)

    def test_EMSA_PSS_verifyInvalidSignature3(self):
        with self.assertRaises(InvalidSignature):
            self.rsa.EMSA_PSS_verify(
                bytearray(b'\xc7\xf5\'\x0f\xcar_\x9b\xd1\x9fQ\x9a\x8d|\xca<' +
                          b'\xc5\xc0y\x02@)\xf3\xba\xe5\x10\xf9\xb0!@\xfe#' +
                          b'\x89\x08\xe4\xf6\xc1\x8f\x07\xa8\x9ch|\x86\x84f' +
                          b'\x9b\x1f\x1d\xb2\xba\xf9%\x1a<\x82\x9f\xac\xcbI0' +
                          b'\x84\xe1n\xc9\xe2\x8dX\x86\x80t\xa5\xd6"\x16g' +
                          b'\xddnR\x8d\x16\xfe,\x9f=\xb4\xcf\xaflM\xce\x8c' +
                          b'\x849\xaf8\xce\xaa\xaa\x9c\xe2\xec\xae{\xc8\xf4' +
                          b'\xa5\xa5^;\xf9m\xf9\xcdW\\O\x9c\xb3\'\x95\x1b' +
                          b'\x8c\xdf\xe4\x08qh'),
                bytearray(b'\xffA=!Fq4\xee\xef5?\xf4\xec\xd8\xa6FPX\xdc~(\xe3'+
                          b'\x92\x17z\xa5-\xcfV\xd4)\x99\x8fJ\xb2\x08\xa2<Q' +
                          b'\x02e\xb4\xe0\xecq\xa3:\xe0I\x1f\x83\x9f\xe2' +
                          b'\xf4\xb9\x89\x9b\xdbv\xb8\xb1&\r\xa8\xdfA\x13' +
                          b'\xfd\xed\xce\x85\xb5\x0e\xae\xd0t\xe5`\xbaen"' +
                          b'\xcd_=:V\'A\xe32u%7\x8b\xed\x16\xc7$\x919\xa5' +
                          b'\x18XA\xa3\xf5r\xad\x8f\xd1\xf7\xf9\x7f;\x01' +
                          b'\xccgrRS\xce}\x18\x985\xb3\xbc'),
                1023, 'sha1', 10)

    def test_EMSA_PSS_verifyInvalidSignature4(self):
        def m(p, hAlg):
            r = secureHash(p, hAlg)
            r[getattr(hashlib, hAlg)().digest_size-1] = r[getattr(
                hashlib, hAlg)().digest_size-1]-1
            return r

        with mock.patch('tlslite.utils.rsakey.secureHash', m):
            with self.assertRaises(InvalidSignature):
                self.rsa.EMSA_PSS_verify(
                    bytearray(b'\xc7\xf5\'\x0f\xcar_\x9b\xd1\x9fQ\x9a\x8d|' +
                              b'\xca<\xc5\xc0y\x02@)\xf3\xba\xe5\x10\xf9' +
                              b'\xb0!@\xfe#\x89\x08\xe4\xf6\xc1\x8f\x07\xa8' +
                              b'\x9ch|\x86\x84f\x9b\x1f\x1d\xb2\xba\xf9%' +
                              b'\x1a<\x82\x9f\xac\xcbI0\x84\xe1n\xc9\xe2' +
                              b'\x8dX\x86\x80t\xa5\xd6"\x16g\xddnR\x8d\x16' +
                              b'\xfe,\x9f=\xb4\xcf\xaflM\xce\x8c\x849\xaf8' +
                              b'\xce\xaa\xaa\x9c\xe2\xec\xae{\xc8\xf4\xa5' +
                              b'\xa5^;\xf9m\xf9\xcdW\\O\x9c\xb3\'\x95\x1b' +
                              b'\x8c\xdf\xe4\x08qh'),
                    bytearray(b'eA=!Fq4\xce\xef5?\xf4\xec\xd8\xa6FPX\xdc~(' +
                              b'\xe3\x92\x17z\xa5-\xcfV\xd4)\x99\x8fJ\xb2' +
                              b'\x08\xa2<Q\x02e\xb4\xe0\xecq\xa3:\xe0I\x1f' +
                              b'\x83\x9f\xe2\xf4\xb9\x89\x9b\xdbv\xb8\xb1&' +
                              b'\r\xa8\xdfA\x13\xfd\xed\xce\x85\xb5\x0e\xae' +
                              b'\xd0t\xe5`\xbaen"\xcd_=:V\'A\xe32u%7\x8b\xed' +
                              b'\x16\xc7$\x919\xa5\x18XA\xa3\xf5r\xad\x8f' +
                              b'\xd1\xf7\xf9\x7f;\x01\xccgrRS\xce}\x18\x985' +
                              b'\xb3\xbc'),
                    1023, 'sha1', 10)

    def test_EMSA_PSS_verifyInvalidSignature5(self):
        def m(leght):
            return bytearray(b'\x11"3DUT2\x16x\x90')

        with self.assertRaises(InvalidSignature):
            with mock.patch('tlslite.utils.rsakey.getRandomBytes', m):
                self.rsa.EMSA_PSS_verify(
                    bytearray(b'\xc8\xf5\'\x0f\xcar_\x9b\xd1\x9fQ\x9a\x8d|' +
                              b'\xca<\xc5\xc0y\x02@)\xf3\xba\xe5\x10\xf9' +
                              b'\xb0!@\xfe#\x89\x08\xe4\xf6\xc1\x8f\x07\xa8' +
                              b'\x9ch|\x86\x84f\x9b\x1f\x1d\xb2\xba\xf9%' +
                              b'\x1a<\x82\x9f\xac\xcbI0\x84\xe1n\xc9\xe2' +
                              b'\x8dX\x86\x80t\xa5\xd6"\x16g\xddnR\x8d\x16' +
                              b'\xfe,\x9f=\xb4\xcf\xaflM\xce\x8c\x849\xaf8' +
                              b'\xce\xaa\xaa\x9c\xe2\xec\xae{\xc8\xf4\xa5' +
                              b'\xa5^;\xf9m\xf9\xcdW\\O\x9c\xb3\'\x95\x1b' +
                              b'\x8c\xdf\xe4\x08qh'),
                    bytearray(b'eA=!Fq4\xce\xef5?\xf4\xec\xd8\xa6FPX\xdc~(' +
                              b'\xe3\x92\x17z\xa5-\xcfV\xd4)\x99\x8fJ\xb2' +
                              b'\x08\xa2<Q\x02e\xb4\xe0\xecq\xa3:\xe0I\x1f' +
                              b'\x83\x9f\xe2\xf4\xb9\x89\x9b\xdbv\xb8\xb1&' +
                              b'\r\xa8\xdfA\x13\xfd\xed\xce\x85\xb5\x0e\xae' +
                              b'\xd0t\xe5`\xbaen"\xcd_=:V\'A\xe32u%7\x8b\xed' +
                              b'\x16\xc7$\x919\xa5\x18XA\xa3\xf5r\xad\x8f' +
                              b'\xd1\xf7\xf9\x7f;\x01\xccgrRS\xce}\x18\x985' +
                              b'\xb3\xbc'),
                    1023, 'sha1', 10)

    def test_EMSA_PSS_verifyInvalidSignature6(self):
        def m(leght):
            return bytearray(b'\x11"3DUT2\x16x\x90')

        with self.assertRaises(InvalidSignature):
            with mock.patch('tlslite.utils.rsakey.getRandomBytes', m):
                self.rsa.EMSA_PSS_verify(
                    bytearray(b'\xc7\xf5\'\x0f\xcar_\x9b\xd1\x9fQ\x9a' +
                              b'\x8d|\xca<\xc5\xc0y\x02@)\xf3\xba\xe5' +
                              b'\x10\xf9\xb0!@\xfe#\x89\x08\xe4\xf6\xc1' +
                              b'\x8f\x07\xa8\x9ch|\x86\x84f\x9b\x1f\x1d' +
                              b'\xb2\xba\xf9%\x1a<\x82\x9f\xac\xcbI0\x84' +
                              b'\xe1n\xc9\xe2\x8dX\x86\x80t\xa5\xd6"' +
                              b'\x16g\xddnR\x8d\x16\xfe,\x9f=\xb4\xcf' +
                              b'\xaflM\xce\x8c\x849\xaf8\xce\xaa\xaa\x9c' +
                              b'\xe2\xec\xae{\xc8\xf4\xa5\xa5^;\xf9m\xf9' +
                              b'\xcdW\\O\x9c\xb3\'\x95\x1b\x8c\xdf\xe4' +
                              b'\x08qh'),
                    bytearray(b'eA=!Fq4\xce\xef5?\xf4\xec\xd8\xa6FPX' +
                              b'\xdc~(\xe3\x92\x17z\xa5-\xcfV\xd4)\x99' +
                              b'\x8fJ\xb2\x08\xa2<Q\x02e\xb4\xe0\xecq' +
                              b'\xa3:\xe0I\x1f\x83\x9f\xe2\xf4\xb9\x89' +
                              b'\x9b\xdbv\xb8\xb1&\r\xa8\xdfA\x13\xfd' +
                              b'\xed\xce\x85\xb5\x0e\xae\xd0t\xe5`' +
                              b'\xbaen"\xcd_=:V\'A\xe32u%7\x8b\xed\x17' +
                              b'\xc7$\x919\xa5\x18XA\xa3\xf5r\xad\x8f' +
                              b'\xd1\xf7\xf9\x7f;\x01\xccgrRS\xce}\x18' +
                              b'\x985\xb3\xbc'),
                    1023, 'sha1', 10)

    def test_MGF1_1(self):
        self.assertEqual(self.rsa.MGF1(bytearray(b'\xad\x8f\xd1\xf7\xf9' +
                                                 b'\x7fgrRS\xce}\x18\x985' +
                                                 b'\xb3'), 107, 'sha1'),
                         bytearray(
                             b'\xb80\x12s\xbb\xd9j\xce&U\x08\x14\xb2\x070' +
                             b'\xc7\xc8\xa8\xa0\xc1\xc3\xf3\xd41\xad\xbe' +
                             b'\xe8\x1dN\x94\xf6sx\x02\xed\xfb\x0b\x0b\x85' +
                             b'\xc5N\xff\x04z\xec\x13\x86O\x15\xe8|\xae\xc6' +
                             b'\x1c\r\xcd\xec\xf4\xb1\xb5$\xf2\x17\xff\xf6' +
                             b'\xc2\xf5\xd2\x8a\xd2\x98\xa8\xb7\xe0;\xab\xe0' +
                             b'\xe9P\xd9\xea\x86\xb3\xeb)\xa3\x98\xb4e\xb5P' +
                             b'\x07\x14\xf1?\xa8i\xb7\xc6\x94\x1c9\x1fX>@' +
                             b'\xe3'))

    def test_MGF1_2(self):
        self.assertEqual(self.rsa.MGF1(bytearray(b'\xad\x8f\xd1\xf7\xf9' +
                                                 b'\x7fgrRS\xce}\x18\x985' +
                                                 b'\xb3'), 40, 'sha1'),
                         bytearray(
                             b'\xb80\x12s\xbb\xd9j\xce&U\x08\x14\xb2\x070' +
                             b'\xc7\xc8\xa8\xa0\xc1\xc3\xf3\xd41\xad\xbe\xe8' +
                             b'\x1dN\x94\xf6sx\x02\xed\xfb\x0b\x0b\x85\xc5'))

    def test_EMSA_PSS_encode(self):
        def m(leght):
            return bytearray(b'\x11"3DUT2\x16x\x90')
        with mock.patch('tlslite.utils.rsakey.getRandomBytes', m):
            mHash = SHA1(
                bytearray(b'\xc7\xf5\'\x0f\xcar_\x9b\xd1\x9fQ\x9a\x8d|\xca<' +
                          b'\xc5\xc0y\x02@)\xf3\xba\xe5\x10\xf9\xb0!@\xfe#' +
                          b'\x89\x08\xe4\xf6\xc1\x8f\x07\xa8\x9ch|\x86\x84f' +
                          b'\x9b\x1f\x1d\xb2\xba\xf9%\x1a<\x82\x9f\xac\xcbI0' +
                          b'\x84\xe1n\xc9\xe2\x8dX\x86\x80t\xa5\xd6"\x16g' +
                          b'\xddnR\x8d\x16\xfe,\x9f=\xb4\xcf\xaflM\xce\x8c' +
                          b'\x849\xaf8\xce\xaa\xaa\x9c\xe2\xec\xae{\xc8\xf4' +
                          b'\xa5\xa5^;\xf9m\xf9\xcdW\\O\x9c\xb3\'\x95\x1b' +
                          b'\x8c\xdf\xe4\x08qh'))
            self.assertEqual(self.rsa.EMSA_PSS_encode(mHash, 1023, 'sha1', 10),
                bytearray(b'eA=!Fq4\xce\xef5?\xf4\xec\xd8\xa6FPX\xdc~(\xe3' +
                          b'\x92\x17z\xa5-\xcfV\xd4)\x99\x8fJ\xb2\x08\xa2<Q' +
                          b'\x02e\xb4\xe0\xecq\xa3:\xe0I\x1f\x83\x9f\xe2\xf4' +
                          b'\xb9\x89\x9b\xdbv\xb8\xb1&\r\xa8\xdfA\x13\xfd' +
                          b'\xed\xce\x85\xb5\x0e\xae\xd0t\xe5`\xbaen"' +
                          b'\xcd_=:V\'A\xe32u%7\x8b\xed\x16\xc7$\x919\xa5' +
                          b'\x18XA\xa3\xf5r\xad\x8f\xd1\xf7\xf9\x7f;\x01' +
                          b'\xccgrRS\xce}\x18\x985\xb3\xbc'))

    def test_EMSA_PSS_verify(self):
        def m(leght):
            return bytearray(b'\x11"3DUT2\x16x\x90')
        with mock.patch('tlslite.utils.rsakey.getRandomBytes', m):
            mHash = SHA1(
                bytearray(b'\xc7\xf5\'\x0f\xcar_\x9b\xd1\x9fQ\x9a\x8d|\xca<' +
                          b'\xc5\xc0y\x02@)\xf3\xba\xe5\x10\xf9\xb0!@\xfe#' +
                          b'\x89\x08\xe4\xf6\xc1\x8f\x07\xa8\x9ch|\x86\x84f' +
                          b'\x9b\x1f\x1d\xb2\xba\xf9%\x1a<\x82\x9f\xac\xcbI0' +
                          b'\x84\xe1n\xc9\xe2\x8dX\x86\x80t\xa5\xd6"\x16g' +
                          b'\xddnR\x8d\x16\xfe,\x9f=\xb4\xcf\xaflM\xce\x8c' +
                          b'\x849\xaf8\xce\xaa\xaa\x9c\xe2\xec\xae{\xc8\xf4' +
                          b'\xa5\xa5^;\xf9m\xf9\xcdW\\O\x9c\xb3\'\x95\x1b' +
                          b'\x8c\xdf\xe4\x08qh'))
            self.assertTrue(self.rsa.EMSA_PSS_verify(mHash,
                bytearray(b'eA=!Fq4\xce\xef5?\xf4\xec\xd8\xa6FPX\xdc~(\xe3' +
                          b'\x92\x17z\xa5-\xcfV\xd4)\x99\x8fJ\xb2\x08\xa2<Q' +
                          b'\x02e\xb4\xe0\xecq\xa3:\xe0I\x1f\x83\x9f\xe2' +
                          b'\xf4\xb9\x89\x9b\xdbv\xb8\xb1&\r\xa8\xdfA\x13' +
                          b'\xfd\xed\xce\x85\xb5\x0e\xae\xd0t\xe5`\xbaen"' +
                          b'\xcd_=:V\'A\xe32u%7\x8b\xed\x16\xc7$\x919\xa5' +
                          b'\x18XA\xa3\xf5r\xad\x8f\xd1\xf7\xf9\x7f;\x01' +
                          b'\xccgrRS\xce}\x18\x985\xb3\xbc'),
                1023, 'sha1', 10))

class TestRSAPSS_mod1026(unittest.TestCase):

    n = int("2b35541ee2850f34c7727e13eed15d60fb62c0cf230c31bbd0766b6994cff4e5e"
            "6279ad00e5fedabd0ed31c77956be0d91c0d9435bf73b270a740fb792c0819ce7"
            "cdac92189d6fcc08d58640c44b69c8fea7c5301a3a62960dee0698ab49ab5f8f1"
            "c4a465c244b79c293f108a9f5ed8425bacba0eb51dcee4a3c3c4d5d81fae70bac"
            "e8ba8c79a2a9cf2be32e0ef2a23b5755743bd356bac5733796d34ebba2d420ae6"
            "e384208b9d8cdd83dced8f6f587b744f814a5d3fcd409869220bb63439cf50fa0"
            "24de4bfad2061706a31c77a8589ac218c7fd90b135f1975783b2765ce88d1570e"
            "812ebea195ae06639d88688e505582dc49ddcab767741cba145be11bf",
            16)
    e = int("65537", 10)
    d = int("024d35eceb3e11404b7b82d1c6ffea0c77779c33ac7742d2f158cd81f3465c923"
            "e7f4f94d39f3286db7b37129c190dc8a541f390cdfe4e6d56f635bc1e9a188d66"
            "1fa398a8ad023e891deea7d68cf9d6961213c3eb31befca5434fa0a4472954cec"
            "7c0011d796577d7f08f7f59a65aff960eec37e7311626af57a412aeef7490fce4"
            "1fdac6d20c4034ac68187a824d6c1a642d917450ede20578cc7812c0109094e28"
            "8b5fdd74b1dc7cb94ecc864b5867e143689cba68a0c1d044569e3cd35f1e815c4"
            "c023ebb24b21dbcdd30e0c7375db1f641cb364470a8378643fd3e6a7692708269"
            "9264cb9dee983832dfac3ee7cee32a35738b5c1dc16bf62f9dcd7bf59",
            16)
    p = int("72a5b2ff09d1ad78f1c72e9bc9c0cf1bcc9b03a1595a153886a0273f515e58cc4"
            "e27adfb6fe52dc9bd95a76056a1093350f5cd9e1a6aeecc6137540fe747f3e3f5"
            "3f2d71fdf4d7ae9b2576830c40f04a8851af78f27b2378056a80053a8a31aed27"
            "0bf757c2e1dc82857dd9a490fc1a442d22c9d7d54137483793486ded1e773",
            16)
    q = int("607b3d256cbe2c063105065825c062d2b4495da3362090820b8cf42bb2434e879"
            "b271543741b691307a75d92b6251723f4e31a79e76ab0b45928200fb67931b8b4"
            "28f89bcd10f1816adffc97bfe0f50b50e63cdcf9da57ee93be926654d5b4e1fe0"
            "33e96ace587ffb2af7ec2c37664a1dfaaec8e9810ba7c13f0f15b17f92185",
            16)
    dP = d % (p - 1)
    dQ = d % (q - 1)
    qInv = invMod(q, p)
    message = bytearray(b'\xc7\xf5\x27\x0f\xca\x72\x5f\x9b\xd1\x9f\x51' +
                        b'\x9a\x8d\x7c\xca\x3c\xc5\xc0\x79\x02\x40\x29' +
                        b'\xf3\xba\xe5\x10\xf9\xb0\x21\x40\xfe\x23\x89' +
                        b'\x08\xe4\xf6\xc1\x8f\x07\xa8\x9c\x68\x7c\x86' +
                        b'\x84\x66\x9b\x1f\x1d\xb2\xba\xf9\x25\x1a\x3c' +
                        b'\x82\x9f\xac\xcb\x49\x30\x84\xe1\x6e\xc9\xe2' +
                        b'\x8d\x58\x86\x80\x74\xa5\xd6\x22\x16\x67\xdd' +
                        b'\x6e\x52\x8d\x16\xfe\x2c\x9f\x3d\xb4\xcf\xaf' +
                        b'\x6c\x4d\xce\x8c\x84\x39\xaf\x38\xce\xaa\xaa' +
                        b'\x9c\xe2\xec\xae\x7b\xc8\xf4\xa5\xa5\x5e\x3b' +
                        b'\xf9\x6d\xf9\xcd\x57\x5c\x4f\x9c\xb3\x27\x95' +
                        b'\x1b\x8c\xdf\xe4\x08\x71\x68')

    def setUp(self):
        self.rsa = Python_RSAKey(self.n, self.e, self.d, self.p, self.q,
                                 self.dP, self.dQ, self.qInv)

    def test_RSAPSS_1026(self):
        signed = bytearray(b'\x0f\xd2\xef/\xa6\xb6+j\xce\xfc\x16.\xce\x99WH_' +
                           b'\x94\xb5\x15\xf1\x8a\xf3\xaa\x12\x82R.\xe8&\xc6' +
                           b'\xf9\xd3\xa9STTW\x0f/L`\nQ>\xeb\t{\x8c\x13\xa3' +
                           b'\xfbm\xe7\xd2\r\xbc!\x91\x07\xaf \xf9\xd3V@' +
                           b'\xb0z,\xb8q\xec$A\x0c\x13\xda\xb7/\xa4U\xa3' +
                           b'\xbb7A\x92\x9b\'\xcf\x9f,\xad\x18\x1a\xf9_\x87' +
                           b'\xf5\x0e\x02\x08\x04\x88\xa3uDnrI\xff\xd0\xa7~_' +
                           b'\xdc>"}%i\xc4\x1a"w\xfdW\x14\x91\xae\x1b\x1f' +
                           b'\xcd\xd9L\xf7(w-]f\xc7\xdc\x82\n\x00[4\xf1f\xf6' +
                           b'\x141\x0e\x12\x13\xf8\x96fc\xc1\x15\xca\x95W' +
                           b'\xd8i\x0f@\x94\x01%\x14(Z\x88\xe1\x00\xf4\xf7' +
                           b'\x81\xfd,\x899x\x9b"\xa4\x01\xef@4@\x9cY\xe2' +
                           b'\x91\x89;{\x8dh\x80Ei\x1b\xf1\x7f1\x93\xe1\xa1j' +
                           b'\xb0\xf1Z\x0c>\xcc\xbeX&$\xfd\x96\xd3\x1e\x92' +
                           b'\xf5\x9b\xbd\x1a\xaa\t\x85>\x13\xb5\xf1s\xa7YN' +
                           b'\x1f\xdb\xa1*\xcc\x93\xa2\xbf\xfd\xe0\xda>0')
        mHash = secureHash(self.message, 'sha1')
        self.assertTrue(self.rsa.RSASSA_PSS_verify(
            mHash, signed, 'sha1', 0))

class TestRSAPSS_mod1024(unittest.TestCase):
    # Test cases from http://csrc.nist.gov/groups/STM/cavp/
    # file SigVerPSS_186-3.rsp

    n = int("be499b5e7f06c83fa0293e31465c8eb6b58af920bae52a7b5b9bfeb7aa72db126"
            "4112eb3fd431d31a2a7e50941566929494a0e891ed5613918b4b51b0d1fb97783"
            "b26acf7d0f384cfb35f4d2824f5dd380623a26bf180b63961c619dcdb20cae406"
            "f22f6e276c80a37259490cfeb72c1a71a84f1846d330877ba3e3101ec9c7b",
            16)
    e = int("00000000000000000000000000000000000000000000000000000000000000000"
            "00000000000000000000000000000000000000000000000000000000000000000"
            "00000000000000000000000000000000000000000000000000000000000000000"
            "0000000000000000000000000000000000000000000000000000000000011",
            16)
    d = int("0d0f17362bdad181db4e1fe03e8de1a3208989914e14bf269558826bfa20faf4b"
            "68dba6bb989a01f03a21c44665dc5f648cb5b59b954eb1077a80263bd22cdfb88"
            "d39164b7404f4f1106ee01cf60b77695748d8fdaf9fd428963fe75144010b1934"
            "c8e26a88239672cf49b3422a07c4d834ba208d570fe408e7095c90547e68d",
            16)
    p = int("e7a80c5d211c06acb900939495f26d365fc2b4825b75e356f89003eaa5931e6be"
            "5c3f7e6a633ad59db6289d06c354c235e739a1e3f3d39fb40d1ffb9cb44288f",
            16)
    q = int("d248aa248000f720258742da67b711940c8f76e1ecd52b67a6ffe1e49354d66ff"
            "84fa601804743f5838da2ed4693a5a28658d6528cc1803bf6c8dc73c5230b55",
            16)
    dP = d % (p - 1)
    dQ = d % (q - 1)
    qInv = invMod(q, p)
    message = bytearray(b'\xc7\xf5\x27\x0f\xca\x72\x5f\x9b\xd1\x9f\x51' +
                        b'\x9a\x8d\x7c\xca\x3c\xc5\xc0\x79\x02\x40\x29' +
                        b'\xf3\xba\xe5\x10\xf9\xb0\x21\x40\xfe\x23\x89' +
                        b'\x08\xe4\xf6\xc1\x8f\x07\xa8\x9c\x68\x7c\x86' +
                        b'\x84\x66\x9b\x1f\x1d\xb2\xba\xf9\x25\x1a\x3c' +
                        b'\x82\x9f\xac\xcb\x49\x30\x84\xe1\x6e\xc9\xe2' +
                        b'\x8d\x58\x86\x80\x74\xa5\xd6\x22\x16\x67\xdd' +
                        b'\x6e\x52\x8d\x16\xfe\x2c\x9f\x3d\xb4\xcf\xaf' +
                        b'\x6c\x4d\xce\x8c\x84\x39\xaf\x38\xce\xaa\xaa' +
                        b'\x9c\xe2\xec\xae\x7b\xc8\xf4\xa5\xa5\x5e\x3b' +
                        b'\xf9\x6d\xf9\xcd\x57\x5c\x4f\x9c\xb3\x27\x95' +
                        b'\x1b\x8c\xdf\xe4\x08\x71\x68')
    salt = bytearray(b'\x11\x22\x33\x44\x55\x54\x32\x16\x78\x90')

    def setUp(self):
        self.rsa = Python_RSAKey(self.n, self.e, self.d, self.p, self.q,
                                 self.dP, self.dQ, self.qInv)

    def test_RSAPSS_sha1(self):
        intendedS = bytearray(b'\x96\xc3\xf6\x92\x70\x1d\x14\xeb\xbe\xf9' +
                              b'\x22\xa5\xc2\x25\x7f\x71\x3d\x20\xa9\x2c' +
                              b'\x69\x38\x74\xe0\x35\xb5\xb0\x65\x15\x92' +
                              b'\xab\x1b\x96\x43\xd3\x81\xd6\xb4\xa9\x70' +
                              b'\xda\xd7\xe2\x38\x00\xe4\x9d\x1a\x66\x57' +
                              b'\xc3\x33\x35\x8e\x9b\xfa\x5c\x71\x34\x93' +
                              b'\x53\x3b\x90\xb0\x23\x4a\x0d\x0d\xcf\x42' +
                              b'\xd0\xa6\x6b\x48\x03\xe4\xdb\x78\x06\x19' +
                              b'\xcc\xab\x6b\xa5\xbb\x27\xd0\x43\xf3\x2d' +
                              b'\x8e\x60\x1e\x2f\x12\xee\x08\xae\xce\x5c' +
                              b'\x47\xcc\x2e\x02\x89\xcd\xbf\x25\xc9\x77' +
                              b'\xcf\x1b\xea\xdc\x04\x74\x21\x50\xbe\xea' +
                              b'\xd6\x96\x2d\xdd\xa9\xe9\x1e\x17')
        def m(leght):
            return self.salt
        with mock.patch('tlslite.utils.rsakey.getRandomBytes', m):
            mHash = secureHash(self.message, 'sha1')
            signed = self.rsa.RSASSA_PSS_sign(mHash, 'sha1', 10)
            self.assertEqual(signed, intendedS)

    def test_RSAPSS_sha224(self):
        intendedS = bytearray(b'\x47\xbd\x25\xf8\x19\xbe\x0f\x7e\xe8\x48\xa3' +
                              b'\x3c\x19\x54\xb5\xbb\xc5\xb0\x0f\xf1\x04\xa2' +
                              b'\xab\x98\xf4\x8c\x38\xe0\x17\x6a\x74\xd7\x07' +
                              b'\xb4\x4c\x36\xdf\x8d\x8c\x12\xda\x49\xec\xec' +
                              b'\x7b\xdc\xc3\x51\x45\x39\xdb\x2b\xd8\xe0\x64' +
                              b'\xca\x62\x89\xaf\xd0\x72\xfd\x86\xc9\xf4\x2e' +
                              b'\x56\x58\xb4\x35\x5b\x34\x19\x30\x4e\x0a\xe9' +
                              b'\x28\x57\x12\x8a\x3c\x5e\xbc\x9b\xa6\x01\x38' +
                              b'\xaf\x67\x44\xec\xf7\x52\x1a\xa1\x11\x94\xac' +
                              b'\x95\x20\x6c\xf7\xa8\x0b\xe9\xca\x5f\x4e\x58' +
                              b'\x49\xae\x67\xf0\x73\xdb\x7b\x69\x2f\xd9\x39' +
                              b'\xcb\x31\xed\x6b\xf5\xe0\x66')
        def m(leght):
            return self.salt
        with mock.patch('tlslite.utils.rsakey.getRandomBytes', m):
            mHash = secureHash(self.message, 'sha224')
            signed = self.rsa.RSASSA_PSS_sign(mHash, 'sha224', 10)
            self.assertEqual(signed, intendedS)


    def test_RSAPSS_sha256(self):
        intendedS = bytearray(b'\x11\xe1\x69\xf2\xfd\x40\xb0\x76\x41\xb9\x76' +
                              b'\x8a\x2a\xb1\x99\x65\xfb\x6c\x27\xf1\x0f\xcf' +
                              b'\x03\x23\xfc\xc6\xd1\x2e\xb4\xf1\xc0\x6b\x33' +
                              b'\x0d\xda\xa1\xea\x50\x44\x07\xaf\xa2\x9d\xe9' +
                              b'\xeb\xe0\x37\x4f\xe9\xd1\xe7\xd0\xff\xbd\x5f' +
                              b'\xc1\xcf\x3a\x34\x46\xe4\x14\x54\x15\xd2\xab' +
                              b'\x24\xf7\x89\xb3\x46\x4c\x5c\x43\xa2\x56\xbb' +
                              b'\xc1\xd6\x92\xcf\x7f\x04\x80\x1d\xac\x5b\xb4' +
                              b'\x01\xa4\xa0\x3a\xb7\xd5\x72\x8a\x86\x0c\x19' +
                              b'\xe1\xa4\xdc\x79\x7c\xa5\x42\xc8\x20\x3c\xec' +
                              b'\x2e\x60\x1e\xb0\xc5\x1f\x56\x7f\x2e\xda\x02' +
                              b'\x2b\x0b\x9e\xbd\xde\xee\xfa')
        def m(leght):
            return self.salt
        with mock.patch('tlslite.utils.rsakey.getRandomBytes', m):
            mHash = secureHash(self.message, 'sha256')
            signed = self.rsa.RSASSA_PSS_sign(mHash, 'sha256', 10)
            self.assertEqual(signed, intendedS)


    def test_RSAPSS_sha384(self):
        intendedS = bytearray(b'\xb2\x81\xad\x93\x4b\x27\x75\xc0\xcb\xa5\xfb' +
                              b'\x10\xaa\x57\x4d\x2e\xd8\x5c\x7f\x99\xb9\x42' +
                              b'\xb7\x8e\x49\x70\x24\x80\x06\x93\x62\xed\x39' +
                              b'\x4b\xad\xed\x55\xe5\x6c\xfc\xbe\x7b\x0b\x8d' +
                              b'\x22\x17\xa0\x5a\x60\xe1\xac\xd7\x25\xcb\x09' +
                              b'\x06\x0d\xfa\xc5\x85\xbc\x21\x32\xb9\x9b\x41' +
                              b'\xcd\xbd\x53\x0c\x69\xd1\x7c\xdb\xc8\x4b\xc6' +
                              b'\xb9\x83\x0f\xc7\xdc\x8e\x1b\x24\x12\xcf\xe0' +
                              b'\x6d\xcf\x8c\x1a\x0c\xc3\x45\x3f\x93\xf2\x5e' +
                              b'\xbf\x10\xcb\x0c\x90\x33\x4f\xac\x57\x3f\x44' +
                              b'\x91\x38\x61\x6e\x1a\x19\x4c\x67\xf4\x4e\xfa' +
                              b'\xc3\x4c\xc0\x7a\x52\x62\x67')
        def m(leght):
            return self.salt
        with mock.patch('tlslite.utils.rsakey.getRandomBytes', m):
            mHash = secureHash(self.message, 'sha384')
            signed = self.rsa.RSASSA_PSS_sign(mHash, 'sha384', 10)
            self.assertEqual(signed, intendedS)


    def test_RSAPSS_sha512(self):
        intendedS = bytearray(b'\x8f\xfc\x38\xf9\xb8\x20\xef\x6b\x08\x0f\xd2' +
                              b'\xec\x7d\xe5\x62\x6c\x65\x8d\x79\x05\x6f\x3e' +
                              b'\xdf\x61\x0a\x29\x5b\x7b\x05\x46\xf7\x3e\x01' +
                              b'\xff\xdf\x4d\x00\x70\xeb\xf7\x9c\x33\xfd\x86' +
                              b'\xc2\xd6\x08\xbe\x94\x38\xb3\xd4\x20\xd0\x95' +
                              b'\x35\xb9\x7c\xd3\xd8\x46\xec\xaf\x8f\x65\x51' +
                              b'\xcd\xf9\x31\x97\xe9\xf8\xfb\x04\x80\x44\x47' +
                              b'\x3a\xb4\x1a\x80\x1e\x9f\x7f\xc9\x83\xc6\x2b' +
                              b'\x32\x43\x61\xda\xde\x9f\x71\xa6\x59\x52\xbd' +
                              b'\x35\xc5\x9f\xaa\xa4\xd6\xff\x46\x2f\x68\xa6' +
                              b'\xc4\xec\x0b\x42\x8a\xa4\x73\x36\xf2\x17\x8a' +
                              b'\xeb\x27\x61\x36\x56\x3b\x7d')
        def m(leght):
            return self.salt
        with mock.patch('tlslite.utils.rsakey.getRandomBytes', m):
            mHash = secureHash(self.message, 'sha512')
            signed = self.rsa.RSASSA_PSS_sign(mHash, 'sha512', 10)
            self.assertEqual(signed, intendedS)

    def test_RSASSA_PSS_verify_sha1(self):
        signed = bytearray(b'\x96\xc3\xf6\x92\x70\x1d\x14\xeb\xbe\xf9' +
                              b'\x22\xa5\xc2\x25\x7f\x71\x3d\x20\xa9\x2c' +
                              b'\x69\x38\x74\xe0\x35\xb5\xb0\x65\x15\x92' +
                              b'\xab\x1b\x96\x43\xd3\x81\xd6\xb4\xa9\x70' +
                              b'\xda\xd7\xe2\x38\x00\xe4\x9d\x1a\x66\x57' +
                              b'\xc3\x33\x35\x8e\x9b\xfa\x5c\x71\x34\x93' +
                              b'\x53\x3b\x90\xb0\x23\x4a\x0d\x0d\xcf\x42' +
                              b'\xd0\xa6\x6b\x48\x03\xe4\xdb\x78\x06\x19' +
                              b'\xcc\xab\x6b\xa5\xbb\x27\xd0\x43\xf3\x2d' +
                              b'\x8e\x60\x1e\x2f\x12\xee\x08\xae\xce\x5c' +
                              b'\x47\xcc\x2e\x02\x89\xcd\xbf\x25\xc9\x77' +
                              b'\xcf\x1b\xea\xdc\x04\x74\x21\x50\xbe\xea' +
                              b'\xd6\x96\x2d\xdd\xa9\xe9\x1e\x17')
        mHash = secureHash(self.message, 'sha1')
        self.assertTrue(self.rsa.RSASSA_PSS_verify(mHash, signed,
                                                   'sha1', 10))

    def test_RSASSA_PSS_verify_shortSign(self):
        with self.assertRaises(InvalidSignature):
            signed = bytearray(b'\x96\xc3\xf6\x92\x70\x1d\x14\xeb\xbe\xf9' +
                               b'\x22\xa5\xc2\x25\x7f\x71\x3d\x20\xa9\x2c' +
                               b'\x69\x38\x74\xe0\x35\xb5\xb0\x65\x15\x92' +
                               b'\xab\x1b\x96\x43\xd3\x81\xd6\xb4\xa9\x70')

            self.assertTrue(self.rsa.RSASSA_PSS_verify(self.message, signed,
                                                   'sha1', 10))

    def test_RSASSA_PSS_verify_sha224(self):
        signed = bytearray(b'\x47\xbd\x25\xf8\x19\xbe\x0f\x7e\xe8\x48\xa3' +
                              b'\x3c\x19\x54\xb5\xbb\xc5\xb0\x0f\xf1\x04\xa2' +
                              b'\xab\x98\xf4\x8c\x38\xe0\x17\x6a\x74\xd7\x07' +
                              b'\xb4\x4c\x36\xdf\x8d\x8c\x12\xda\x49\xec\xec' +
                              b'\x7b\xdc\xc3\x51\x45\x39\xdb\x2b\xd8\xe0\x64' +
                              b'\xca\x62\x89\xaf\xd0\x72\xfd\x86\xc9\xf4\x2e' +
                              b'\x56\x58\xb4\x35\x5b\x34\x19\x30\x4e\x0a\xe9' +
                              b'\x28\x57\x12\x8a\x3c\x5e\xbc\x9b\xa6\x01\x38' +
                              b'\xaf\x67\x44\xec\xf7\x52\x1a\xa1\x11\x94\xac' +
                              b'\x95\x20\x6c\xf7\xa8\x0b\xe9\xca\x5f\x4e\x58' +
                              b'\x49\xae\x67\xf0\x73\xdb\x7b\x69\x2f\xd9\x39' +
                              b'\xcb\x31\xed\x6b\xf5\xe0\x66')
        mHash = secureHash(self.message, 'sha224')
        self.assertTrue(self.rsa.RSASSA_PSS_verify(mHash, signed,
                                                   'sha224', 10))

    def test_RSASSA_PSS_verify_sha256(self):
        signed = bytearray(b'\x11\xe1\x69\xf2\xfd\x40\xb0\x76\x41\xb9\x76' +
                              b'\x8a\x2a\xb1\x99\x65\xfb\x6c\x27\xf1\x0f\xcf' +
                              b'\x03\x23\xfc\xc6\xd1\x2e\xb4\xf1\xc0\x6b\x33' +
                              b'\x0d\xda\xa1\xea\x50\x44\x07\xaf\xa2\x9d\xe9' +
                              b'\xeb\xe0\x37\x4f\xe9\xd1\xe7\xd0\xff\xbd\x5f' +
                              b'\xc1\xcf\x3a\x34\x46\xe4\x14\x54\x15\xd2\xab' +
                              b'\x24\xf7\x89\xb3\x46\x4c\x5c\x43\xa2\x56\xbb' +
                              b'\xc1\xd6\x92\xcf\x7f\x04\x80\x1d\xac\x5b\xb4' +
                              b'\x01\xa4\xa0\x3a\xb7\xd5\x72\x8a\x86\x0c\x19' +
                              b'\xe1\xa4\xdc\x79\x7c\xa5\x42\xc8\x20\x3c\xec' +
                              b'\x2e\x60\x1e\xb0\xc5\x1f\x56\x7f\x2e\xda\x02' +
                              b'\x2b\x0b\x9e\xbd\xde\xee\xfa')
        mHash = secureHash(self.message, 'sha256')
        self.assertTrue(self.rsa.RSASSA_PSS_verify(mHash, signed,
                                                   'sha256', 10))

    def test_RSASSA_PSS_verify_sha384(self):
        signed = bytearray(b'\xb2\x81\xad\x93\x4b\x27\x75\xc0\xcb\xa5\xfb' +
                              b'\x10\xaa\x57\x4d\x2e\xd8\x5c\x7f\x99\xb9\x42' +
                              b'\xb7\x8e\x49\x70\x24\x80\x06\x93\x62\xed\x39' +
                              b'\x4b\xad\xed\x55\xe5\x6c\xfc\xbe\x7b\x0b\x8d' +
                              b'\x22\x17\xa0\x5a\x60\xe1\xac\xd7\x25\xcb\x09' +
                              b'\x06\x0d\xfa\xc5\x85\xbc\x21\x32\xb9\x9b\x41' +
                              b'\xcd\xbd\x53\x0c\x69\xd1\x7c\xdb\xc8\x4b\xc6' +
                              b'\xb9\x83\x0f\xc7\xdc\x8e\x1b\x24\x12\xcf\xe0' +
                              b'\x6d\xcf\x8c\x1a\x0c\xc3\x45\x3f\x93\xf2\x5e' +
                              b'\xbf\x10\xcb\x0c\x90\x33\x4f\xac\x57\x3f\x44' +
                              b'\x91\x38\x61\x6e\x1a\x19\x4c\x67\xf4\x4e\xfa' +
                              b'\xc3\x4c\xc0\x7a\x52\x62\x67')
        mHash = secureHash(self.message, 'sha384')
        self.assertTrue(self.rsa.RSASSA_PSS_verify(mHash, signed,
                                                   'sha384', 10))

    def test_RSASSA_PSS_verify_sha512(self):
        signed = bytearray(b'\x8f\xfc\x38\xf9\xb8\x20\xef\x6b\x08\x0f\xd2' +
                              b'\xec\x7d\xe5\x62\x6c\x65\x8d\x79\x05\x6f\x3e' +
                              b'\xdf\x61\x0a\x29\x5b\x7b\x05\x46\xf7\x3e\x01' +
                              b'\xff\xdf\x4d\x00\x70\xeb\xf7\x9c\x33\xfd\x86' +
                              b'\xc2\xd6\x08\xbe\x94\x38\xb3\xd4\x20\xd0\x95' +
                              b'\x35\xb9\x7c\xd3\xd8\x46\xec\xaf\x8f\x65\x51' +
                              b'\xcd\xf9\x31\x97\xe9\xf8\xfb\x04\x80\x44\x47' +
                              b'\x3a\xb4\x1a\x80\x1e\x9f\x7f\xc9\x83\xc6\x2b' +
                              b'\x32\x43\x61\xda\xde\x9f\x71\xa6\x59\x52\xbd' +
                              b'\x35\xc5\x9f\xaa\xa4\xd6\xff\x46\x2f\x68\xa6' +
                              b'\xc4\xec\x0b\x42\x8a\xa4\x73\x36\xf2\x17\x8a' +
                              b'\xeb\x27\x61\x36\x56\x3b\x7d')
        mHash = secureHash(self.message, 'sha512')
        self.assertTrue(self.rsa.RSASSA_PSS_verify(mHash, signed,
                                                   'sha512', 10))

    def test_RSASSA_PSS_verify_noSalt(self):
        signed = bytearray(b'\xafe\x03\xb5\xaf5\x0b\t\xd1?9\x89\xee\x0eP\xcc' +
                           b'\x82\xef%\xc2t<\xa2\xff\xd6\x13[\x97\xbd\xac' +
                           b'\xda\x97;\xcb!\xfa"\x10\t\xb7\x81\xb9\x8f\x9a' +
                           b'\x1a\xc87\xa3,\xb4\xea\xddG7\xe8RI\xf9\x91m\x8e' +
                           b'\x91\xe3\xf8Y\xdd \x92\xd7I\xcc`czm\x01~\x85' +
                           b'\xf6\xa6\xd6_PF3\xc9\xb5\x192\xf4U\\|\xcc' +
                           b'\xcd6|7d\xca,\x8dIF\x02\xf8\xcd\x81\xdd\x88' +
                           b'\xb0\xae\xe9\x1f\x93\xf3\xfa\x90\x0f\xcd' +
                           b'\xe2|\xbc<R\xf7\xa3\x8a')
        mHash = secureHash(self.message, 'sha512')
        self.assertTrue(self.rsa.RSASSA_PSS_verify(mHash, signed,
                                                   'sha512', 0))

    def test_RSASSA_PSS_verify_wrongSign(self):
        signed = bytearray(b'\x96\xc3\xf6\x92\x70\x1d\x14\xeb\xbe\xf9' +
                           b'\x22\xa5\xc2\x25\x7f\x71\x3d\x20\xa9\x2c' +
                           b'\x69\x38\x74\xe0\x35\xb5\xb0\x65\x15\x92' +
                           b'\xab\x1b\x96\x43\xd3\x81\xd6\xb4\xa9\x70' +
                           b'\xda\xd7\xe2\x38\x00\xe4\x9d\x1a\x66\x57' +
                           b'\xc3\x33\x35\x8e\x8b\xfa\x5c\x71\x34\x93' +
                           b'\x53\x3b\x90\xb0\x23\x4a\x0d\x0d\xcf\x42' +
                           b'\xd0\xa6\x6b\x48\x03\xe4\xdb\x78\x06\x19' +
                           b'\xcc\xab\x6b\xa5\xbb\x27\xd0\x43\xf3\x2d' +
                           b'\x8e\x60\x1e\x2f\x12\xee\x08\xae\xce\x5c' +
                           b'\x47\xcc\x2e\x02\x89\xcd\xbf\x25\xc9\x77' +
                           b'\xcf\x1b\xea\xdc\x04\x74\x21\x50\xbe\xea' +
                           b'\xd6\x96\x2d\xdd\xa9\xe9\x1e\x10')
        with self.assertRaises(InvalidSignature):
            self.rsa.RSASSA_PSS_verify(self.message, signed,
                                                        'sha1', 10)

    def test_RSASSA_PSS_verify_wrongSign2(self):
        def m(x, M, EM, numBits, hAlg, sLen):
            return False
        signed = bytearray(b'\x96\xc3\xf6\x92\x70\x1d\x14\xeb\xbe\xf9' +
                           b'\x22\xa5\xc2\x25\x7f\x71\x3d\x20\xa9\x2c' +
                           b'\x69\x38\x74\xe0\x35\xb5\xb0\x65\x15\x92' +
                           b'\xab\x1b\x96\x43\xd3\x81\xd6\xb4\xa9\x70' +
                           b'\xda\xd7\xe2\x38\x00\xe4\x9d\x1a\x66\x57' +
                           b'\xc3\x33\x35\x8e\x8b\xfa\x5c\x71\x34\x93' +
                           b'\x53\x3b\x90\xb0\x23\x4a\x0d\x0d\xcf\x42' +
                           b'\xd0\xa6\x6b\x48\x03\xe4\xdb\x78\x06\x19' +
                           b'\xcc\xab\x6b\xa5\xbb\x27\xd0\x43\xf3\x2d' +
                           b'\x8e\x60\x1e\x2f\x12\xee\x08\xae\xce\x5c' +
                           b'\x47\xcc\x2e\x02\x89\xcd\xbf\x25\xc9\x77' +
                           b'\xcf\x1b\xea\xdc\x04\x74\x21\x50\xbe\xea' +
                           b'\xd6\x96\x2d\xdd\xa9\xe9\x1e\x10')
        with mock.patch('tlslite.utils.rsakey.RSAKey.EMSA_PSS_verify', m):
            with self.assertRaises(InvalidSignature):
                self.rsa.RSASSA_PSS_verify(self.message, signed,
                                           'sha1', 10)

class TestRSAPSS_mod2048(unittest.TestCase):
    # Test cases from http://csrc.nist.gov/groups/STM/cavp/
    # file SigVerPSS_186-3.rsp

    n = int("c6e0ed537a2d85cf1c4effad6419884d824ceabf5200e755691cb7328acd6a755"
            "fe85798502ccaec9e55d47afd0cf3258ebe920b50c5fd9d72897462bd0e459bbd"
            "f902b63d17195b2ef54908980be12aa7489f8af274b92c0cbc16aed2fa46f782d"
            "5517b666edfb2e5e5efeaff7e24965e26472e51980b0cfe457d297e6aa5dacb8e"
            "728dc6f58130f925a13275c3cace62f820db1e13cc5274c58ff4d7837671a1bf5"
            "f80d6ad8699c568df8d24dd0f152ded36ef4861f59b354bba96a076913a25facf"
            "4722737e6deed95b69a00fb2bced0feeedea4ff01a92605cfe26a6b39553d0c74"
            "e5650eb3779705e135c4b2fa518a8d4339c53efab4bb0058238def555",
            16)
    e = int("00000000000000000000000000000000000000000000000000000000000000000"
            "00000000000000000000000000000000000000000000000000000000000000000"
            "00000000000000000000000000000000000000000000000000000000000000000"
            "00000000000000000000000000000000000000000000000000000000000000000"
            "00000000000000000000000000000000000000000000000000000000000000000"
            "00000000000000000000000000000000000000000000000000000000000000000"
            "00000000000000000000000000000000000000000000000000000000000000000"
            "000000000000000000000000000000000000000000000000000010001",
            16)
    d = int("03ac73787e325992a96749d5ef8500e2ccf99e96214dbc22df2c6fde3538aaa85"
            "78e1b3cc871af5f940ed4a6df46438bdf240f896478fd2090fffa2af9c034a7cb"
            "684e5fc491f3940987c537d80128d6b37230ba4314c60d3580ad9aeb46ed6929d"
            "cf1629f6784667c547fec48c3112a1d9144f1802c82bb1476544e757e98538191"
            "85949352b92411adabd0f76efafe72c3b3fce88c5895b0bc4ac1ad36ec8d5be4a"
            "db89e72519850c6fc8c4076b658a2e554a37b5aa76aef7293a1ec256ccdc0c93c"
            "60aa528596a44ad72c76ed55726206d4bfd2f431745cc1c7dc399213051275fcf"
            "d2757552cef855be7bf23a5480688032bb4f322669a3e7d2fbff31c91",
            16)
    p = int("e2f7ceb13ea5385ad7659d7bbf0ad4a517c697b70c9d2af7a2193d62b14014412"
            "cc3e5fda97882341e0a370ce9f0f6c8149bb199d6f408b65d0524aecaf6e3fd7e"
            "3c35de940dc661ae17ddbdf57184e75bd2e9642401045ba48c7aee4abdc1caddc"
            "a85fd064e80ab82ce58537848d9e9b8a477b4dfd04b9be496baec79cfa4c5",
            16)
    q = int("e0515147b6e596b7e5140e81365ad698dbeddd874642510f42d357123c20ffb0e"
            "1f377afefe97f20442e1c3f3c88919c39978b78835b9c7253f6ea632ab3298667"
            "48c6dc195865ce123c8e153d03a3d731b7161205e2d83e6651152ee8181e389ad"
            "7a795dd3ce6ba44c753b4c7774fafbfa9c6606f89c08eec37632ba607b751",
            16)
    dP = d % (p - 1)
    dQ = d % (q - 1)
    qInv = invMod(q, p)
    salt = bytearray(b'\x11\x22\x33\x44\x55\x54\x32\x16\x78\x90')

    def setUp(self):
        self.rsa = Python_RSAKey(self.n, self.e, self.d, self.p, self.q,
                                 self.dP, self.dQ, self.qInv)

    def test_RSAPSS_sha1_wrongSignature(self):
        message = bytearray(b'\x13\x0f\x45\x53\x89\x78\xe3\x2f\x14\xb9\xb9' +
                            b'\x1f\x2c\xf9\xa3\xa1\x28\xc2\x56\xa6\x03\xb2' +
                            b'\x43\xe8\x5f\x73\xbe\x7e\xca\xed\x5f\xad\x41' +
                            b'\xb9\xa8\x02\xf2\xd9\xe9\x9d\x46\xa7\x61\xd0' +
                            b'\x1f\x0c\xa6\xe9\x4f\xf2\x47\x4b\xa9\xfc\xaf' +
                            b'\xc4\x6b\x74\x4c\x1a\x1c\x85\xf1\xe7\xc2\xaa' +
                            b'\x79\xa7\xb8\x66\xae\x10\xae\x36\x69\xa2\xf1' +
                            b'\xc4\xfa\x7e\xed\x5d\xc9\x7b\xf0\xa5\x3e\x77' +
                            b'\x30\x89\xdf\xeb\x10\x76\xb8\xc2\x9f\xc8\x00' +
                            b'\x6c\x61\x86\xf9\x2e\x53\x4c\x18\xbc\x29\x88' +
                            b'\x66\x09\xda\xe9\x26\x5e\x5e\x15\xb8\xaa\xb6' +
                            b'\x9b\xbd\x19\x2e\x28\x7c\xe7')
        intendedS = bytearray(b'\xc2\xc8\xb9\xd9\x32\x95\x1b\xe1\x80\xeb\x87' +
                              b'\x34\x69\xb8\x6f\x71\x17\x92\xe0\x19\x3b\x03' +
                              b'\xbf\xc3\x61\x83\x76\x61\x20\x52\x5c\xfe\xb8' +
                              b'\x03\x5c\x99\x01\xd0\x09\xd9\x7d\xea\x81\xed' +
                              b'\x0a\x43\x7d\x85\x5b\x8f\x73\x55\xec\xd1\xd7' +
                              b'\x16\x20\xdb\x91\x8b\x5d\xbc\x14\x1a\x77\xed' +
                              b'\x24\x0f\x73\x9a\x98\x98\x7a\xf8\x6f\x1f\x88' +
                              b'\x20\x6f\x96\x77\x55\x26\xe6\xa9\x79\x51\x0d' +
                              b'\x4d\xf9\x1c\xf2\xa8\x90\xf9\x08\x6f\x06\x84' +
                              b'\x2e\xe2\x8b\x95\xb1\xc5\x94\xed\xe7\xd0\xc9' +
                              b'\xe8\x59\x5d\x53\x11\xc7\x0f\x00\x3b\xd3\x2c' +
                              b'\x86\x56\x09\x3a\x23\x14\xb5\x73\x54\x75\xae' +
                              b'\xe1\x21\xd4\x84\x6e\xd4\xaa\x52\xb2\x46\xb3' +
                              b'\x37\xd4\x88\x36\x9a\xa3\x91\xb0\x51\x70\xbf' +
                              b'\x73\x15\xf6\xf1\xbc\x2a\x7b\x9b\x0f\x4f\x1c' +
                              b'\xf1\xab\x5d\xe1\x71\xe0\xdf\xb4\xe0\x10\x35' +
                              b'\xb4\x3e\xbd\xa0\xef\x13\xd3\xc8\x6b\x26\xbf' +
                              b'\xf4\xf3\x50\x9b\xe6\x83\xf5\x75\xbe\xcf\x17' +
                              b'\x46\x84\xfb\xfb\xee\x98\xe8\x69\xbb\xad\xca' +
                              b'\x8a\xbd\x02\xde\xfd\xf9\x3a\x68\xf4\x50\xba' +
                              b'\xa7\x5e\x6f\x81\xc0\xaa\x17\xaf\x59\x12\xdb' +
                              b'\x12\xa2\x4d\xf8\xf7\x2f\x63\xc2\x2d\x17\x14' +
                              b'\x13\x6e\x47\x4d\x9d\xb6\xf8\xa9\xf0\xb2\x6d' +
                              b'\x72\xa9\x1f')
        def m(leght):
            return self.salt
        with mock.patch('tlslite.utils.rsakey.getRandomBytes', m):
            signed = self.rsa.RSASSA_PSS_sign(message, 'sha1', 10)
            self.assertNotEqual(signed, intendedS)

    def test_RSAPSS_sha1_messageTooLong(self):
        message = bytearray(b'\xc2\xc8\xb9\xd9\x32\x95\x1b\xe1\x80\xeb\x87' +
                            b'\x34\x69\xb8\x6f\x71\x17\x92\xe0\x19\x3b\x03' +
                            b'\xbf\xc3\x61\x83\x76\x61\x20\x52\x5c\xfe\xb8' +
                            b'\x03\x5c\x99\x01\xd0\x09\xd9\x7d\xea\x81\xed' +
                            b'\x0a\x43\x7d\x85\x5b\x8f\x73\x55\xec\xd1\xd7' +
                            b'\x16\x20\xdb\x91\x8b\x5d\xbc\x14\x1a\x77\xed' +
                            b'\x24\x0f\x73\x9a\x98\x98\x7a\xf8\x6f\x1f\x88' +
                            b'\x20\x6f\x96\x77\x55\x26\xe6\xa9\x79\x51\x0d' +
                            b'\x4d\xf9\x1c\xf2\xa8\x90\xf9\x08\x6f\x06\x84' +
                            b'\x2e\xe2\x8b\x95\xb1\xc5\x94\xed\xe7\xd0\xc9' +
                            b'\xe8\x59\x5d\x53\x11\xc7\x0f\x00\x3b\xd3\x2c' +
                            b'\x86\x56\x09\x3a\x23\x14\xb5\x73\x54\x75\xae' +
                            b'\xe1\x21\xd4\x84\x6e\xd4\xaa\x52\xb2\x46\xb3' +
                            b'\x37\xd4\x88\x36\x9a\xa3\x91\xb0\x51\x70\xbf' +
                            b'\x73\x15\xf6\xf1\xbc\x2a\x7b\x9b\x0f\x4f\x1c' +
                            b'\xf1\xab\x5d\xe1\x71\xe0\xdf\xb4\xe0\x10\x35' +
                            b'\xb4\x3e\xbd\xa0\xef\x13\xd3\xc8\x6b\x26\xbf' +
                            b'\xf4\xf3\x50\x9b\xe6\x83\xf5\x75\xbe\xcf\x17' +
                            b'\x46\x84\xfb\xfb\xee\x98\xe8\x69\xbb\xad\xca' +
                            b'\x8a\xbd\x02\xde\xfd\xf9\x3a\x68\xf4\x50\xba' +
                            b'\xa7\x5e\x6f\x81\xc0\xaa\x17\xaf\x59\x12\xdb' +
                            b'\x12\xa2\x4d\xf8\xf7\x2f\x63\xc2\x2d\x17\x14' +
                            b'\x13\x6e\x47\x4d\x9d\xb6\xf8\xa9\xf0\xb2\x6d' +
                            b'\x72\xa9\x1f' +
                            b'\x13\x0f\x45\x53\x89\x78\xe3\x2f\x14\xb9\xb9' +
                            b'\x1f\x2c\xf9\xa3\xa1\x28\xc2\x56\xa6\x03\xb2' +
                            b'\x43\xe8\x5f\x73\xbe\x7e\xca\xed\x5f\xad\x41' +
                            b'\xb9\xa8\x02\xf2\xd9\xe9\x9d\x46\xa7\x61\xd0' +
                            b'\x1f\x0c\xa6\xe9\x4f\xf2\x47\x4b\xa9\xfc\xaf' +
                            b'\xc4\x6b\x74\x4c\x1a\x1c\x85\xf1\xe7\xc2\xaa' +
                            b'\x79\xa7\xb8\x66\xae\x10\xae\x36\x69\xa2\xf1' +
                            b'\xc4\xfa\x7e\xed\x5d\xc9\x7b\xf0\xa5\x3e\x77' +
                            b'\x30\x89\xdf\xeb\x10\x76\xb8\xc2\x9f\xc8\x00' +
                            b'\x6c\x61\x86\xf9\x2e\x53\x4c\x18\xbc\x29\x88' +
                            b'\x66\x09\xda\xe9\x26\x5e\x5e\x15\xb8\xaa\xb6' +
                            b'\x9b\xbd\x19\x2e\x28\x7c\xe7' +
                            b'\xc2\xc8\xb9\xd9\x32\x95\x1b\xe1\x80\xeb\x87' +
                            b'\x34\x69\xb8\x6f\x71\x17\x92\xe0\x19\x3b\x03' +
                            b'\xbf\xc3\x61\x83\x76\x61\x20\x52\x5c\xfe\xb8' +
                            b'\x03\x5c\x99\x01\xd0\x09\xd9\x7d\xea\x81\xed' +
                            b'\x0a\x43\x7d\x85\x5b\x8f\x73\x55\xec\xd1\xd7' +
                            b'\x16\x20\xdb\x91\x8b\x5d\xbc\x14\x1a\x77\xed' +
                            b'\x24\x0f\x73\x9a\x98\x98\x7a\xf8\x6f\x1f\x88' +
                            b'\x20\x6f\x96\x77\x55\x26\xe6\xa9\x79\x51\x0d' +
                            b'\x4d\xf9\x1c\xf2\xa8\x90\xf9\x08\x6f\x06\x84' +
                            b'\x2e\xe2\x8b\x95\xb1\xc5\x94\xed\xe7\xd0\xc9' +
                            b'\xe8\x59\x5d\x53\x11\xc7\x0f\x00\x3b\xd3\x2c' +
                            b'\x86\x56\x09\x3a\x23\x14\xb5\x73\x54\x75\xae' +
                            b'\xe1\x21\xd4\x84\x6e\xd4\xaa\x52\xb2\x46\xb3' +
                            b'\x37\xd4\x88\x36\x9a\xa3\x91\xb0\x51\x70\xbf' +
                            b'\x73\x15\xf6\xf1\xbc\x2a\x7b\x9b\x0f\x4f\x1c' +
                            b'\xf1\xab\x5d\xe1\x71\xe0\xdf\xb4\xe0\x10\x35' +
                            b'\xb4\x3e\xbd\xa0\xef\x13\xd3\xc8\x6b\x26\xbf' +
                            b'\xf4\xf3\x50\x9b\xe6\x83\xf5\x75\xbe\xcf\x17' +
                            b'\x46\x84\xfb\xfb\xee\x98\xe8\x69\xbb\xad\xca' +
                            b'\x8a\xbd\x02\xde\xfd\xf9\x3a\x68\xf4\x50\xba' +
                            b'\xa7\x5e\x6f\x81\xc0\xaa\x17\xaf\x59\x12\xdb' +
                            b'\x12\xa2\x4d\xf8\xf7\x2f\x63\xc2\x2d\x17\x14' +
                            b'\x13\x6e\x47\x4d\x9d\xb6\xf8\xa9\xf0\xb2\x6d' +
                            b'\x72\xa9\x1f' +
                            b'\x13\x0f\x45\x53\x89\x78\xe3\x2f\x14\xb9\xb9' +
                            b'\x1f\x2c\xf9\xa3\xa1\x28\xc2\x56\xa6\x03\xb2' +
                            b'\x43\xe8\x5f\x73\xbe\x7e\xca\xed\x5f\xad\x41' +
                            b'\xb9\xa8\x02\xf2\xd9\xe9\x9d\x46\xa7\x61\xd0' +
                            b'\x1f\x0c\xa6\xe9\x4f\xf2\x47\x4b\xa9\xfc\xaf' +
                            b'\xc4\x6b\x74\x4c\x1a\x1c\x85\xf1\xe7\xc2\xaa' +
                            b'\x79\xa7\xb8\x66\xae\x10\xae\x36\x69\xa2\xf1' +
                            b'\xc4\xfa\x7e\xed\x5d\xc9\x7b\xf0\xa5\x3e\x77' +
                            b'\x30\x89\xdf\xeb\x10\x76\xb8\xc2\x9f\xc8\x00' +
                            b'\x6c\x61\x86\xf9\x2e\x53\x4c\x18\xbc\x29\x88' +
                            b'\x66\x09\xda\xe9\x26\x5e\x5e\x15\xb8\xaa\xb6' +
                            b'\x9b\xbd\x19\x2e\x28\x7c\xe7' +
                            b'\x9b\xbd\x19\x2e\x28\x7c\xe7' +
                            b'\xc2\xc8\xb9\xd9\x32\x95\x1b\xe1\x80\xeb\x87' +
                            b'\x34\x69\xb8\x6f\x71\x17\x92\xe0\x19\x3b\x03' +
                            b'\xbf\xc3\x61\x83\x76\x61\x20\x52\x5c\xfe\xb8' +
                            b'\x03\x5c\x99\x01\xd0\x09\xd9\x7d\xea\x81\xed' +
                            b'\x0a\x43\x7d\x85\x5b\x8f\x73\x55\xec\xd1\xd7' +
                            b'\x16\x20\xdb\x91\x8b\x5d\xbc\x14\x1a\x77\xed' +
                            b'\x24\x0f\x73\x9a\x98\x98\x7a\xf8\x6f\x1f\x88' +
                            b'\x20\x6f\x96\x77\x55\x26\xe6\xa9\x79\x51\x0d' +
                            b'\x4d\xf9\x1c\xf2\xa8\x90\xf9\x08\x6f\x06\x84' +
                            b'\x2e\xe2\x8b\x95\xb1\xc5\x94\xed\xe7\xd0\xc9' +
                            b'\xe8\x59\x5d\x53\x11\xc7\x0f\x00\x3b\xd3\x2c' +
                            b'\x86\x56\x09\x3a\x23\x14\xb5\x73\x54\x75\xae' +
                            b'\xe1\x21\xd4\x84\x6e\xd4\xaa\x52\xb2\x46\xb3' +
                            b'\x37\xd4\x88\x36\x9a\xa3\x91\xb0\x51\x70\xbf' +
                            b'\x73\x15\xf6\xf1\xbc\x2a\x7b\x9b\x0f\x4f\x1c' +
                            b'\xf1\xab\x5d\xe1\x71\xe0\xdf\xb4\xe0\x10\x35' +
                            b'\xb4\x3e\xbd\xa0\xef\x13\xd3\xc8\x6b\x26\xbf' +
                            b'\xf4\xf3\x50\x9b\xe6\x83\xf5\x75\xbe\xcf\x17' +
                            b'\x46\x84\xfb\xfb\xee\x98\xe8\x69\xbb\xad\xca' +
                            b'\x8a\xbd\x02\xde\xfd\xf9\x3a\x68\xf4\x50\xba' +
                            b'\xa7\x5e\x6f\x81\xc0\xaa\x17\xaf\x59\x12\xdb' +
                            b'\x12\xa2\x4d\xf8\xf7\x2f\x63\xc2\x2d\x17\x14' +
                            b'\x13\x6e\x47\x4d\x9d\xb6\xf8\xa9\xf0\xb2\x6d' +
                            b'\x72\xa9\x1f' +
                            b'\x13\x0f\x45\x53\x89\x78\xe3\x2f\x14\xb9\xb9' +
                            b'\x1f\x2c\xf9\xa3\xa1\x28\xc2\x56\xa6\x03\xb2' +
                            b'\x43\xe8\x5f\x73\xbe\x7e\xca\xed\x5f\xad\x41' +
                            b'\xb9\xa8\x02\xf2\xd9\xe9\x9d\x46\xa7\x61\xd0' +
                            b'\x1f\x0c\xa6\xe9\x4f\xf2\x47\x4b\xa9\xfc\xaf' +
                            b'\xc4\x6b\x74\x4c\x1a\x1c\x85\xf1\xe7\xc2\xaa' +
                            b'\x79\xa7\xb8\x66\xae\x10\xae\x36\x69\xa2\xf1' +
                            b'\xc4\xfa\x7e\xed\x5d\xc9\x7b\xf0\xa5\x3e\x77' +
                            b'\x30\x89\xdf\xeb\x10\x76\xb8\xc2\x9f\xc8\x00' +
                            b'\x6c\x61\x86\xf9\x2e\x53\x4c\x18\xbc\x29\x88' +
                            b'\x66\x09\xda\xe9\x26\x5e\x5e\x15\xb8\xaa\xb6' +
                            b'\x9b\xbd\x19\x2e\x28\x7c\xe7' +
                            b'\xc2\xc8\xb9\xd9\x32\x95\x1b\xe1\x80\xeb\x87' +
                            b'\x34\x69\xb8\x6f\x71\x17\x92\xe0\x19\x3b\x03' +
                            b'\xbf\xc3\x61\x83\x76\x61\x20\x52\x5c\xfe\xb8' +
                            b'\x03\x5c\x99\x01\xd0\x09\xd9\x7d\xea\x81\xed' +
                            b'\x0a\x43\x7d\x85\x5b\x8f\x73\x55\xec\xd1\xd7' +
                            b'\x16\x20\xdb\x91\x8b\x5d\xbc\x14\x1a\x77\xed' +
                            b'\x24\x0f\x73\x9a\x98\x98\x7a\xf8\x6f\x1f\x88' +
                            b'\x20\x6f\x96\x77\x55\x26\xe6\xa9\x79\x51\x0d' +
                            b'\x4d\xf9\x1c\xf2\xa8\x90\xf9\x08\x6f\x06\x84' +
                            b'\x2e\xe2\x8b\x95\xb1\xc5\x94\xed\xe7\xd0\xc9' +
                            b'\xe8\x59\x5d\x53\x11\xc7\x0f\x00\x3b\xd3\x2c' +
                            b'\x86\x56\x09\x3a\x23\x14\xb5\x73\x54\x75\xae' +
                            b'\xe1\x21\xd4\x84\x6e\xd4\xaa\x52\xb2\x46\xb3' +
                            b'\x37\xd4\x88\x36\x9a\xa3\x91\xb0\x51\x70\xbf' +
                            b'\x73\x15\xf6\xf1\xbc\x2a\x7b\x9b\x0f\x4f\x1c' +
                            b'\xf1\xab\x5d\xe1\x71\xe0\xdf\xb4\xe0\x10\x35' +
                            b'\xb4\x3e\xbd\xa0\xef\x13\xd3\xc8\x6b\x26\xbf' +
                            b'\xf4\xf3\x50\x9b\xe6\x83\xf5\x75\xbe\xcf\x17' +
                            b'\x46\x84\xfb\xfb\xee\x98\xe8\x69\xbb\xad\xca' +
                            b'\x8a\xbd\x02\xde\xfd\xf9\x3a\x68\xf4\x50\xba' +
                            b'\xa7\x5e\x6f\x81\xc0\xaa\x17\xaf\x59\x12\xdb' +
                            b'\x12\xa2\x4d\xf8\xf7\x2f\x63\xc2\x2d\x17\x14' +
                            b'\x13\x6e\x47\x4d\x9d\xb6\xf8\xa9\xf0\xb2\x6d' +
                            b'\x72\xa9\x1f' +
                            b'\x13\x0f\x45\x53\x89\x78\xe3\x2f\x14\xb9\xb9' +
                            b'\x1f\x2c\xf9\xa3\xa1\x28\xc2\x56\xa6\x03\xb2' +
                            b'\x43\xe8\x5f\x73\xbe\x7e\xca\xed\x5f\xad\x41' +
                            b'\xb9\xa8\x02\xf2\xd9\xe9\x9d\x46\xa7\x61\xd0' +
                            b'\x1f\x0c\xa6\xe9\x4f\xf2\x47\x4b\xa9\xfc\xaf' +
                            b'\xc4\x6b\x74\x4c\x1a\x1c\x85\xf1\xe7\xc2\xaa' +
                            b'\x79\xa7\xb8\x66\xae\x10\xae\x36\x69\xa2\xf1' +
                            b'\xc4\xfa\x7e\xed\x5d\xc9\x7b\xf0\xa5\x3e\x77' +
                            b'\x30\x89\xdf\xeb\x10\x76\xb8\xc2\x9f\xc8\x00' +
                            b'\x6c\x61\x86\xf9\x2e\x53\x4c\x18\xbc\x29\x88' +
                            b'\x66\x09\xda\xe9\x26\x5e\x5e\x15\xb8\xaa\xb6' +
                            b'\x9b\xbd\x19\x2e\x28\x7c\xe7' +
                            b'\x9b\xbd\x19\x2e\x28\x7c\xe7')

        def m(x,M, emBits, hAlg, sLen):
            return bytearray(b"4c\xa3n\xae\xc9\x15\x84\'\xc3&\xd4*A\xb2\x04r" +
                             b"\x930\xc2\x972(cQ\xc9\xea\xcf\x94?#\x9b\xfer" +
                             b"\xff\xfe]C\x8d\xd5\x9b\xab\xe2oM`\xdc\x92!" +
                             b"\x00\x8fSp\xcf\x8b\x9c\x12s\xabH\xd6\xe4=\xac" +
                             b"\xd5\x12\xc8^\x0f\x95\xc4\xe7\xb0\x91\xa3\x1e" +
                             b"\xffb\xc0V{z\xbd\xc0P\xb7\xbd\x80gQXIh\x15:" +
                             b"\x7f\xca\xf7}\xf8\xaf\xdc\xb9d\x19\xe4\x02" +
                             b"\xc5d\xcaJ\x81\xd9`\x0fL)~Q\xfd\x08i\xc39\x03" +
                             b"\xda\xc6\xcf\xa3\xf9\x81\x8d&\xed\xaeQ\xc3" +
                             b"\xf5\x0e\x07\x8a\x8c\x14\xf9V\xbc\xf0\x13\xa3" +
                             b"\xdcw\x97\x81\xa7ym\xb3\xbd\xa3\xc0\n\x1a\xc3"
                             b"\x94\xd0\xaa\x8a\xbfV\xa6i\xa1\x86L#\xb1O\x12"+
                             b"\xce\x0b\x90\xaaU\xa1\xe8&5\x8b\xd5\xd0Ek(" +
                             b"\x10Az\x95\xc7\xc8\xe8\x1f\x05\xc7\xb7\xc3q" +
                             b"\x8f\xe5\x0cT\x05\x1a\x0f\x05\x8cCq\x1c\xbdSV" +
                             b"\x14n\x85M\x99\xdf\xff\x86\xd5\x98\xdf\x1ef" +
                             b"\xde\x1f\x07r\xa7\xd2\xfe\x16\x8e6\xcd\xfd" +
                             b"\t%\xa4\x87\'\xc61V\x0c\xbc\ff\ff\ff\ff\ff")

        with mock.patch('tlslite.utils.rsakey.RSAKey.EMSA_PSS_encode', m):
            with self.assertRaises(MessageTooLongError):
                self.rsa.RSASSA_PSS_sign(message, 'sha224', 10)

    def test_RSAPSS_sha1_noSalt(self):
        message = bytearray(b'\x13\x0f\x45\x53\x89\x78\xe3\x2f\x14\xb9\xb9' +
                            b'\x1f\x2c\xf9\xa3\xa1\x28\xc2\x56\xa6\x03\xb2' +
                            b'\x43\xe8\x5f\x73\xbe\x7e\xca\xed\x5f\xad\x41' +
                            b'\xb9\xa8\x02\xf2\xd9\xe9\x9d\x46\xa7\x61\xd0' +
                            b'\x1f\x0c\xa6\xe9\x5f\xf2\x47\x4b\xa9\xfc\xaf' +
                            b'\xc4\x6b\x74\x4c\x1a\x1c\x85\xf1\xe7\xc2\xaa' +
                            b'\x79\xa7\xb8\x66\xae\x10\xae\x36\x69\xa2\xf1' +
                            b'\xc4\xfa\x7e\xed\x5d\xc9\x7b\xf0\xa5\x3e\x77' +
                            b'\x30\x89\xdf\xeb\x10\x76\xb8\xc2\x9f\xc8\x00' +
                            b'\x6c\x61\x86\xf9\x2e\x53\x4c\x18\xbc\x29\x88' +
                            b'\x66\x09\xda\xe9\x26\x5e\x5e\x15\xb8\xaa\xb6' +
                            b'\x9b\xbd\x19\x2e\x28\x7c\xe7')
        intendedS = bytearray(b'6\xbb\x70\x06\x53\x58\x78\xca\x42\x76\x2a' +
                              b'\x55\x43\x72\x8a\xb0\xf6\x85\x70\xfa\xbc\xd4' +
                              b'\xb4\xba\xa4\x67\x0e\x5b\xdf\xfb\x41\xfa\x31' +
                              b'\x64\xeb\xac\x95\x2f\xc8\x79\xad\x58\x73\x60' +
                              b'\xac\x38\xd2\x4c\xde\xcf\xe4\x04\xfc\xc0\x7b' +
                              b'\xde\x52\x14\x3b\x79\x98\xcf\x80\xb7\x98\xe3' +
                              b'\x38\xc1\xfa\xac\xac\x59\xeb\x65\xec\xa1\xc7' +
                              b'\xa7\xba\x9a\xa5\x35\xa5\xf7\xb4\x89\x4f\x91' +
                              b'\x85\xab\x2e\x4d\xbd\x39\x56\x7a\xaa\x4b\xb9' +
                              b'\x72\xa2\x44\xa4\xa5\x55\x37\xfd\x16\x8c\x4a' +
                              b'\xa4\x91\x3d\xa5\xb2\x45\xd3\x64\x77\xdc\x5e' +
                              b'\x02\xa6\x8c\x32\xef\x54\x18\xb5\x3a\xe5\xff' +
                              b'\x9e\xcc\x6c\x6c\x39\x36\x40\x9a\x1a\x20\xe5' +
                              b'\x86\x39\x1e\x1e\xad\xd1\xff\x25\x1f\x48\x97' +
                              b'\xd3\x38\xf6\x6e\x9a\xe6\x15\x48\xc2\x83\x4c' +
                              b'\x82\xa9\x98\x35\x29\x19\xef\xaa\x22\x26\xcc' +
                              b'\x8b\xbe\x6d\x64\xfe\xa8\x4f\xd3\xe0\x43\xf8' +
                              b'\xf7\xd2\xe0\xcf\x65\x5e\x14\x96\xbf\x1a\x2d' +
                              b'\x3e\xf5\x1f\xf7\xd3\x54\x92\xf1\x4f\xd2\x44' +
                              b'\xd2\xb8\x4c\xfc\xa8\x15\x93\xaf\x85\x92\x8f' +
                              b'\x47\x28\x2c\x0f\x48\xce\x84\xaf\x1d\xb4\x54' +
                              b'\x3f\xc1\x6d\x6e\x86\xf3\xb5\x5a\xac\x95\x08' +
                              b'\x8f\x0d\x74\xbc\xe6\xfc\x4b\x83\xdd\x4c\xae' +
                              b'\xa5\xdd\xea')
        mHash = secureHash(message, 'sha1')
        signed = self.rsa.RSASSA_PSS_sign(mHash, 'sha1')
        self.assertEqual(signed, intendedS)

    def test_RSAPSS_sha1(self):
        message = bytearray(b'\x13\x0f\x45\x53\x89\x78\xe3\x2f\x14\xb9\xb9' +
                            b'\x1f\x2c\xf9\xa3\xa1\x28\xc2\x56\xa6\x03\xb2' +
                            b'\x43\xe8\x5f\x73\xbe\x7e\xca\xed\x5f\xad\x41' +
                            b'\xb9\xa8\x02\xf2\xd9\xe9\x9d\x46\xa7\x61\xd0' +
                            b'\x1f\x0c\xa6\xe9\x5f\xf2\x47\x4b\xa9\xfc\xaf' +
                            b'\xc4\x6b\x74\x4c\x1a\x1c\x85\xf1\xe7\xc2\xaa' +
                            b'\x79\xa7\xb8\x66\xae\x10\xae\x36\x69\xa2\xf1' +
                            b'\xc4\xfa\x7e\xed\x5d\xc9\x7b\xf0\xa5\x3e\x77' +
                            b'\x30\x89\xdf\xeb\x10\x76\xb8\xc2\x9f\xc8\x00' +
                            b'\x6c\x61\x86\xf9\x2e\x53\x4c\x18\xbc\x29\x88' +
                            b'\x66\x09\xda\xe9\x26\x5e\x5e\x15\xb8\xaa\xb6' +
                            b'\x9b\xbd\x19\x2e\x28\x7c\xe7')
        intendedS = bytearray(b'\xc2\xc8\xb9\xd9\x32\x95\x1b\xe1\x80\xeb\x87' +
                              b'\x34\x69\xb8\x6f\x71\x17\x92\xe0\x19\x3b\x03' +
                              b'\xbf\xc3\x61\x83\x76\x61\x20\x52\x5c\xfe\xb8' +
                              b'\x03\x5c\x99\x01\xd0\x09\xd9\x7d\xea\x81\xed' +
                              b'\x0a\x43\x7d\x85\x5b\x8f\x73\x55\xec\xd1\xd7' +
                              b'\x16\x20\xdb\x91\x8b\x5d\xbc\x14\x1a\x77\xed' +
                              b'\x24\x0f\x73\x9a\x98\x98\x7a\xf8\x6f\x1f\x88' +
                              b'\x20\x6f\x96\x77\x55\x26\xe6\xa9\x79\x51\x0d' +
                              b'\x4d\xf9\x1c\xf2\xa8\x90\xf9\x08\x6f\x06\x84' +
                              b'\x2e\xe2\x8b\x95\xb1\xc5\x94\xed\xe7\xd0\xc9' +
                              b'\xe8\x59\x5d\x53\x11\xc7\x0f\x00\x3b\xd3\x2c' +
                              b'\x86\x56\x09\x3a\x23\x14\xb5\x73\x54\x75\xae' +
                              b'\xe1\x21\xd4\x84\x6e\xd4\xaa\x52\xb2\x46\xb3' +
                              b'\x37\xd4\x88\x36\x9a\xa3\x91\xb0\x51\x70\xbf' +
                              b'\x73\x15\xf6\xf1\xbc\x2a\x7b\x9b\x0f\x4f\x1c' +
                              b'\xf1\xab\x5d\xe1\x71\xe0\xdf\xb4\xe0\x10\x35' +
                              b'\xb4\x3e\xbd\xa0\xef\x13\xd3\xc8\x6b\x26\xbf' +
                              b'\xf4\xf3\x50\x9b\xe6\x83\xf5\x75\xbe\xcf\x17' +
                              b'\x46\x84\xfb\xfb\xee\x98\xe8\x69\xbb\xad\xca' +
                              b'\x8a\xbd\x02\xde\xfd\xf9\x3a\x68\xf4\x50\xba' +
                              b'\xa7\x5e\x6f\x81\xc0\xaa\x17\xaf\x59\x12\xdb' +
                              b'\x12\xa2\x4d\xf8\xf7\x2f\x63\xc2\x2d\x17\x14' +
                              b'\x13\x6e\x47\x4d\x9d\xb6\xf8\xa9\xf0\xb2\x6d' +
                              b'\x72\xa9\x1f')
        def m(leght):
            return self.salt
        with mock.patch('tlslite.utils.rsakey.getRandomBytes', m):
            mHash = secureHash(message, 'sha1')
            signed = self.rsa.RSASSA_PSS_sign(mHash, 'sha1', 10)
            self.assertEqual(signed, intendedS)

    def test_RSAPSS_sha224(self):
        message = bytearray(b'\x40\x12\x6e\xcf\x7f\x69\x69\x1e\x10\x74\x4e' +
                            b'\xa0\x3a\x2d\xbc\xc6\xb0\x4b\x21\x9d\x66\xc4' +
                            b'\x2a\x65\xa2\x91\x7e\x7e\x56\xb1\xab\x8c\xad' +
                            b'\x70\x60\xd3\xe4\xe9\xde\xe3\x5b\xca\xa9\x7a' +
                            b'\xec\x57\x83\xa6\x7b\xb2\x9a\x2a\xce\x59\x01' +
                            b'\x23\x9f\xf0\x5d\xeb\xf0\x41\xe1\x3e\x9f\x81' +
                            b'\x7c\x9b\x3a\xd4\x50\xed\x14\x14\x67\x6b\x99' +
                            b'\xce\xa1\xbd\xde\x8e\xf1\x60\x7d\x5d\xc9\x4b' +
                            b'\x9f\x87\xd3\xd3\x5a\xa2\xe2\xcc\x3e\xcb\x28' +
                            b'\xf1\x2a\x33\xa9\x86\x64\xdb\xef\x3b\xe9\x9b' +
                            b'\x00\x54\x02\x69\x8d\x07\x18\xd8\x6c\x25\xc9' +
                            b'\xdf\x49\xfd\x3c\xdc\x98\x7e')
        intendedS = bytearray(b'\x32\xda\x2a\xd3\x9b\x0d\xc4\xd5\x66\xfe\x7d' +
                              b'\xc7\xe3\x5d\xb8\xd7\x62\x11\x6d\x6c\x83\x4b' +
                              b'\xaf\xe6\x77\xd2\x08\x62\x0d\x86\xa1\x57\x7d' +
                              b'\x52\x45\x2b\x7a\x6f\x4c\xd5\x75\x59\xfa\x04' +
                              b'\x97\x36\x5f\x6b\xb7\x4f\x11\x2d\xbf\x34\x25' +
                              b'\x19\xc2\x45\xd0\x5a\x6a\x69\xaf\xfd\xbb\xad' +
                              b'\x71\x65\x66\xf0\x62\xac\xd6\x53\x9d\xe3\x49' +
                              b'\xe9\x50\x6c\x2a\xf9\xc4\x26\x84\x66\x5b\x11' +
                              b'\xe5\x3f\x46\x2a\xa1\xa2\x8a\x09\x2e\x68\x01' +
                              b'\x32\x38\x6c\x00\xf8\x3f\xc2\x9b\x9c\xc0\xa8' +
                              b'\xf0\x38\x26\x9c\x88\xba\x02\x56\x23\x8a\xee' +
                              b'\xde\x11\x72\xd7\xbd\x6d\xff\xef\x80\x47\x34' +
                              b'\x57\x38\x17\xf0\x6f\xb7\x6e\x6f\xe0\x16\xe3' +
                              b'\x5b\x19\x6b\xa9\x6b\xc3\x99\xe1\x19\xf3\x29' +
                              b'\xc0\xd7\x34\x69\x3b\xac\x2a\xb6\xcb\x8e\xda' +
                              b'\x9e\xa3\xe9\x1a\x77\x98\xb8\x29\x64\xbc\xe2' +
                              b'\xbe\xd6\xf2\xe4\x6a\x23\x1c\x50\x01\xc9\x68' +
                              b'\x2d\x40\x3c\xc5\x29\x51\x72\xc9\x25\xd8\xbb' +
                              b'\xbf\xbc\xe6\xd1\x0d\xfa\x2d\x67\x17\xe0\x11' +
                              b'\x18\x5c\x24\x52\x60\x1d\xf8\x2c\x27\x00\x44' +
                              b'\x09\x17\xd5\x28\x43\xd3\x39\x12\x63\x07\xc9' +
                              b'\xca\xb2\x64\x47\x30\xf1\x41\x93\x33\xf2\xd7' +
                              b'\xfa\x7e\x33\x60\xf0\x30\xfd\x95\x41\x39\x1b' +
                              b'\xc3\x7a\x31')
        def m(leght):
            return self.salt
        with mock.patch('tlslite.utils.rsakey.getRandomBytes', m):
            mHash = secureHash(message, 'sha224')
            signed = self.rsa.RSASSA_PSS_sign(mHash, 'sha224', 10)
            self.assertEqual(signed, intendedS)


    def test_RSAPSS_sha256(self):
        message = bytearray(b'\x81\xea\xf4\x73\xd4\x08\x96\xdb\xf4\xde\xac' +
                            b'\x0f\x35\xc6\x3b\xd1\xe1\x29\x14\x7c\x76\xe7' +
                            b'\xaa\x8d\x0e\xf9\x21\x63\x1f\x55\xa7\x43\x64' +
                            b'\x11\x07\x9f\x1b\xcc\x7b\x98\x71\x4a\xc2\xc1' +
                            b'\x3b\x5e\x73\x26\xe6\x0d\x91\x8d\xb1\xf0\x5f' +
                            b'\xfb\x19\xda\x76\x7a\x95\xbb\x14\x1a\x84\xc4' +
                            b'\xb7\x36\x64\xcc\xeb\xf8\x44\xf3\x60\x1f\x7c' +
                            b'\x85\x3f\x00\x9b\x21\xbe\xcb\xa1\x1a\xf3\x10' +
                            b'\x6f\x1d\xe5\x82\x7b\x14\xe9\xfa\xc8\x4b\x2c' +
                            b'\xbf\x16\xd1\x8c\x04\x56\x22\xac\xb2\x60\x02' +
                            b'\x47\x68\xe8\xac\xc4\xc0\xae\x2c\x0b\xd5\xf6' +
                            b'\x0a\x98\x02\x38\x28\xcd\xec')
        intendedS = bytearray(b'\x40\xd5\x9e\xbc\x6c\xb7\xb9\x60\xcb\xda\x0d' +
                              b'\xb3\x53\xf9\xb8\x5d\x77\xe7\xc0\x3f\x84\x44' +
                              b'\x7f\xb8\xe9\x1b\x96\xa5\xa7\x37\x7a\xbc\x32' +
                              b'\x9d\x1f\x55\xc8\x5e\x0d\xbe\xdb\xc2\x88\x6c' +
                              b'\xe1\x91\xd9\xe2\xcf\x3b\xe0\x5b\x33\xd6\xbb' +
                              b'\xd2\xba\x92\xb8\x5e\xee\x2f\xf8\x9c\xd6\xee' +
                              b'\x29\xcd\x53\x1e\x42\x01\x6e\x6a\xba\x1d\x62' +
                              b'\x0f\xe5\x5e\x44\x48\x0c\x03\x3e\x8a\x59\xc0' +
                              b'\x85\x2d\xd1\xca\xff\xbc\x2c\xe8\x29\x69\xe3' +
                              b'\xa9\xf4\x4c\xef\xf7\x9f\x89\x99\x3b\x9e\xbf' +
                              b'\x37\x41\xb2\xcc\xab\x0b\x95\x16\xf2\xe1\x28' +
                              b'\x65\x6a\x5b\x2a\xd5\x25\x1e\x20\xc6\xce\x0c' +
                              b'\x26\xa1\x4e\xef\x7e\xe8\x64\x58\x94\x2d\xdb' +
                              b'\xe9\x5c\xcc\x1f\x67\xb2\x53\xe4\x3e\x72\x11' +
                              b'\x7f\x49\x59\x5d\xab\x5b\xa4\x23\x49\x6e\xce' +
                              b'\x12\x82\x54\x35\x66\x11\x12\x66\x6d\xba\xe7' +
                              b'\x1a\xaf\xfd\x5a\x8f\x1d\x58\xdb\x9d\xc0\x2e' +
                              b'\x0d\x70\xfe\x3a\xc3\x6a\x87\xb8\xee\xed\x4f' +
                              b'\x20\xc0\x0f\xd4\x30\x3f\x9f\x76\x7d\x03\xbc' +
                              b'\xa1\xa6\x19\xbb\xe4\xb0\x8e\x4e\x53\xb5\xcb' +
                              b'\x69\xd2\xba\x02\x35\x06\x3e\x04\xca\x39\x23' +
                              b'\x34\xd9\x97\x9a\x41\xc4\x2a\x66\xca\x8b\x97' +
                              b'\x21\xed\xcf\x76\x98\x9b\xa8\x9f\x3a\x17\x0b' +
                              b'\xb2\xe4\x85')
        def m(leght):
            return self.salt
        with mock.patch('tlslite.utils.rsakey.getRandomBytes', m):
            mHash = secureHash(message, 'sha256')
            signed = self.rsa.RSASSA_PSS_sign(mHash, 'sha256', 10)
            self.assertEqual(signed, intendedS)


    def test_RSAPSS_sha384(self):
        message = bytearray(b'\x32\xa7\xb1\x47\x9a\xcf\x50\x5d\xb7\x93\xf3' +
                              b'\xeb\xed\x95\x3f\x4e\x31\xc9\xec\xad\x1a\x34' +
                              b'\x79\xdf\x3a\xf3\x1e\x89\xae\x7e\x03\x87\xf4' +
                              b'\x2e\xaf\x8e\xfd\xfd\xc3\x0f\x83\x8e\xe8\x5e' +
                              b'\x9d\x6d\x06\x13\x91\x97\xb7\xb1\xe9\x3d\xfb' +
                              b'\x85\xc9\xc5\x2d\xd1\x7f\x12\x35\x2a\x5c\x05' +
                              b'\x00\x1f\xc2\x43\x2d\x1b\x7f\x39\x09\x8d\x59' +
                              b'\x5e\xbe\x45\xea\xb8\xc7\x21\xaf\xa2\xa7\xea' +
                              b'\x5b\xcc\xdb\x79\x71\x83\x0d\x1e\x11\x33\x8a' +
                              b'\x42\x12\x2a\xf6\x4a\x52\x9e\x3f\xbf\x4a\xf2' +
                              b'\xcf\xac\xe6\x35\x06\x48\x93\xec\xe7\xd5\x99' +
                              b'\x11\x11\xc8\xab\x5b\xf1\x2a')
        intendedS = bytearray(b'\x0c\xb3\x75\xec\xc3\x4a\x9f\x36\xb8\x8b\xf5' +
                              b'\x6e\xbf\x12\x35\x38\x7f\xfa\xcf\xd3\xdd\x09' +
                              b'\xc4\x8e\x87\x28\x97\xca\xca\x60\xaf\x9e\x38' +
                              b'\x64\x96\xaa\xfd\x0d\x4b\x1f\xd8\xfb\x47\x14' +
                              b'\xfa\xc9\x25\xed\xda\x6f\x34\x63\x3c\x3b\xb0' +
                              b'\x8f\x7c\xca\x3d\x9a\xd8\xb7\x64\x72\xde\x8c' +
                              b'\x9f\x91\xcb\x75\x18\x64\x8d\x36\x8f\xbe\xb3' +
                              b'\x1d\x1a\x7c\xb3\x9a\x40\xa7\xb1\x7e\xe2\xf7' +
                              b'\xba\xce\x9b\xd9\x9b\xa0\x82\x95\xaa\xdd\x85' +
                              b'\x6c\xd6\x90\x2e\xe6\xc9\x6d\x5c\x12\x91\xdc' +
                              b'\x29\x9a\x7f\x35\x28\xa8\x69\xf6\x2f\xb8\xfb' +
                              b'\xd5\x18\x17\xff\xe6\x49\x0e\xd6\xe0\x00\x7d' +
                              b'\x79\x81\xab\x12\xb8\xf4\xce\x0d\x74\x32\xe8' +
                              b'\xc3\x21\x3f\xae\x2b\x81\x00\x6f\x33\x37\x14' +
                              b'\xb5\x13\xeb\xa0\x41\x4c\x16\x1f\xab\x6e\xa2' +
                              b'\x33\x38\x56\x79\x95\xf2\x73\xe3\x26\x9c\x44' +
                              b'\xa5\x87\xad\x83\x5c\x32\x0d\x1e\x5f\xf5\x53' +
                              b'\xdb\x4c\x47\x12\x66\x80\xcd\x58\x29\x32\x31' +
                              b'\x91\x5c\xf7\xae\xfb\x80\x69\x04\x99\x24\x3e' +
                              b'\xda\x83\xf5\x34\x7a\x30\x0e\x07\x05\x68\xba' +
                              b'\xee\x27\x45\xb2\x0c\x68\x68\x8d\xad\x6e\x38' +
                              b'\x07\xaf\xcb\x34\xc7\x2c\xda\xeb\x9a\x57\x10' +
                              b'\x89\xc7\xf8\xc6\x3d\x1b\x6f\xfd\xbe\x2f\xd1' +
                              b'\x33\x30\xe6')
        def m(leght):
            return self.salt
        with mock.patch('tlslite.utils.rsakey.getRandomBytes', m):
            mHash = secureHash(message, 'sha384')
            signed = self.rsa.RSASSA_PSS_sign(mHash, 'sha384', 10)
            self.assertEqual(signed, intendedS)


    def test_RSAPSS_sha512(self):
        message = bytearray(b'\x35\xa3\x79\x46\xe5\x26\x78\xee\x37\x8f\x5f' +
                            b'\x17\x68\x38\xef\x08\xf3\xc2\x13\x92\xb1\xad' +
                            b'\x20\x46\x45\x25\x5b\xe5\xb7\x1f\xbc\x18\x5f' +
                            b'\xa5\xf1\x61\x05\x6e\xa6\x52\x46\xb2\x04\xfd' +
                            b'\x39\x3c\x77\xab\x53\xc1\xb5\xd1\x88\x70\xfc' +
                            b'\x3f\xb3\xca\x9a\x9b\x38\xb4\xb3\x0e\xe8\xcb' +
                            b'\x3f\x3d\x25\xf7\x52\x7b\x46\x43\xa0\x3c\x3d' +
                            b'\xec\x40\xcd\x76\xb7\xb0\x43\x03\x88\x1a\xb2' +
                            b'\xf7\x31\xd5\x9f\x0f\x88\x2f\xb7\x98\xbc\x6a' +
                            b'\xc1\x8c\xe9\x04\xd1\xff\xe9\x3c\xbe\xb9\x6e' +
                            b'\xd1\xd7\x25\x4d\x0d\xd2\x6a\x1d\x02\x05\xd7' +
                            b'\x01\x14\xd9\x84\xc2\xb7\x7b')
        intendedS = bytearray(b'\x56\x27\x9c\xcd\x2d\x37\xe8\x11\x36\x25\x73' +
                              b'\x2c\xd3\xf3\xb6\x1b\x4e\xf9\x32\x51\x60\xc7' +
                              b'\xf6\xaf\x70\x77\xc2\x50\x49\xd3\x27\x42\x60' +
                              b'\x7a\xe3\xf8\x45\xbd\x66\xcd\x67\x52\x81\x3c' +
                              b'\x26\x06\x7f\xdc\x23\xf0\x80\x08\xcf\x6a\x53' +
                              b'\x11\x24\xe9\xeb\xc9\x26\x4f\x7c\xfd\x6d\x6e' +
                              b'\xef\xf1\x5d\xaf\x97\xda\xd2\x25\x65\xec\x36' +
                              b'\xb6\x91\x25\xe7\xb2\x7f\xc9\x38\x92\xf6\xff' +
                              b'\x42\xce\x8f\x26\x5d\xc2\xcf\x2e\x57\x58\xba' +
                              b'\x0d\x67\x96\x8e\x80\x0e\x73\xfd\x47\x13\x10' +
                              b'\x08\xf5\xad\xc8\x63\x91\x9e\xa0\xcc\x15\x3c' +
                              b'\xf7\xef\xff\x13\x4b\x0b\xcb\xd0\xaf\x55\x05' +
                              b'\xab\x49\xaf\x7b\x75\xd6\x3a\x8a\xa7\x97\x6b' +
                              b'\x8f\xe7\x7b\xaf\x5a\x69\x9a\x7e\x38\xa6\x0e' +
                              b'\xb7\xca\x64\xe8\x34\xf3\xe0\xf8\x9d\xa5\xb6' +
                              b'\xa3\x43\xa0\x1f\x76\x57\xfd\x84\x2c\x09\x1a' +
                              b'\x62\x08\x50\x3b\xc7\x5d\xe8\xf9\x5d\xe0\xd8' +
                              b'\x71\xa6\xfc\x11\x4b\x59\x4f\xf9\x9d\x61\x58' +
                              b'\x25\xfd\x3b\x89\x69\x33\x38\x14\x52\x53\x6d' +
                              b'\x68\xd9\xe0\x34\xf6\x5a\xbf\x34\x12\xe8\xe3' +
                              b'\x20\x02\x68\x9e\x10\x2a\x8b\xb6\x99\x91\xe0' +
                              b'\x4a\x7f\xf6\x81\xb6\x2e\x48\xee\x68\x7b\xad' +
                              b'\xf8\x69\x0b\x2c\xce\xe4\xbf\x24\x5c\xd0\xa2' +
                              b'\x5d\xcf\x21')
        def m(leght):
            return self.salt
        with mock.patch('tlslite.utils.rsakey.getRandomBytes', m):
            mHash = secureHash(message, 'sha512')
            signed = self.rsa.RSASSA_PSS_sign(mHash, 'sha512', 10)
            self.assertEqual(signed, intendedS)


class TestRSAPKCS1(unittest.TestCase):
    n = int("a8d68acd413c5e195d5ef04e1b4faaf242365cb450196755e92e1215ba59802aa"
            "fbadbf2564dd550956abb54f8b1c917844e5f36195d1088c600e07cada5c080ed"
            "e679f50b3de32cf4026e514542495c54b1903768791aae9e36f082cd38e941ada"
            "89baecada61ab0dd37ad536bcb0a0946271594836e92ab5517301d45176b5",
            16)
    e = int("00000000000000000000000000000000000000000000000000000000000000000"
            "00000000000000000000000000000000000000000000000000000000000000000"
            "00000000000000000000000000000000000000000000000000000000000000000"
            "0000000000000000000000000000000000000000000000000000000000003",
            16)
    p = int("c107a2fe924b76e206cb9bc4af2ab7008547c00846bf6d0680b3eac3ebcbd0c7f"
            "d7a54c2b9899b08f80cde1d3691eaaa2816b1eb11822d6be7beaf4e30977c49",
            16)
    q = int("dfea984ce4307eafc0d140c2bb82861e5dbac4f8567cbc981d70440dd63949207"
            "9031486315e305eb83e591c4a2e96064966f7c894c3ca351925b5ce82d8ef0d",
            16)
    d = int("1c23c1cce034ba598f8fd2b7af37f1d30b090f7362aee68e5187adae49b9955c7"
            "29f24a863b7a38d6e3c748e2972f6d940b7ba89043a2d6c2100256a1cf0f56a8c"
            "d35fc6ee205244876642f6f9c3820a3d9d2c8921df7d82aaadcaf2d7334d39893"
            "1ddbba553190b3a416099f3aa07fd5b26214645a828419e122cfb857ad73b",
            16)
    dP = d % (p - 1)
    dQ = d % (q - 1)
    qInv = invMod(q, p)
    message = bytearray(
            b'\xd7\x38\x29\x49\x7c\xdd\xbe\x41\xb7\x05\xfa\xac\x50\xe7' +
            b'\x89\x9f\xdb\x5a\x38\xbf\x3a\x45\x9e\x53\x63\x57\x02\x9e' +
            b'\x64\xf8\x79\x6b\xa4\x7f\x4f\xe9\x6b\xa5\xa8\xb9\xa4\x39' +
            b'\x67\x46\xe2\x16\x4f\x55\xa2\x53\x68\xdd\xd0\xb9\xa5\x18' +
            b'\x8c\x7a\xc3\xda\x2d\x1f\x74\x22\x86\xc3\xbd\xee\x69\x7f' +
            b'\x9d\x54\x6a\x25\xef\xcf\xe5\x31\x91\xd7\x43\xfc\xc6\xb4' +
            b'\x78\x33\xd9\x93\xd0\x88\x04\xda\xec\xa7\x8f\xb9\x07\x6c' +
            b'\x3c\x01\x7f\x53\xe3\x3a\x90\x30\x5a\xf0\x62\x20\x97\x4d' +
            b'\x46\xbf\x19\xed\x3c\x9b\x84\xed\xba\xe9\x8b\x45\xa8\x77' +
            b'\x12\x58')

    def setUp(self):
        self.rsa = Python_RSAKey(self.n, self.e, self.d, self.p, self.q,
                                 self.dP, self.dQ, self.qInv)

    def test_hashAndSign_RSAPKCS1_sha1(self):
        sigBytes = self.rsa.hashAndSign(self.message,
            "PKCS1", "sha1")
        self.assertEqual(sigBytes, bytearray(
            b'\x17\x50\x15\xbd\xa5\x0a\xbe\x0f\xa7\xd3\x9a\x83\x53\x88' +
            b'\x5c\xa0\x1b\xe3\xa7\xe7\xfc\xc5\x50\x45\x74\x41\x11\x36' +
            b'\x2e\xe1\x91\x44\x73\xa4\x8d\xc5\x37\xd9\x56\x29\x4b\x9e' +
            b'\x20\xa1\xef\x66\x1d\x58\x53\x7a\xcd\xc8\xde\x90\x8f\xa0' +
            b'\x50\x63\x0f\xcc\x27\x2e\x6d\x00\x10\x45\xe6\xfd\xee\xd2' +
            b'\xd1\x05\x31\xc8\x60\x33\x34\xc2\xe8\xdb\x39\xe7\x3e\x6d' +
            b'\x96\x65\xee\x13\x43\xf9\xe4\x19\x83\x02\xd2\x20\x1b\x44' +
            b'\xe8\xe8\xd0\x6b\x3e\xf4\x9c\xee\x61\x97\x58\x21\x63\xa8' +
            b'\x49\x00\x89\xca\x65\x4c\x00\x12\xfc\xe1\xba\x65\x11\x08' +
            b'\x97\x50'))

    def test_hashAndSign_wrongRSaAlgorithm(self):
        with self.assertRaises(UnknownRSAType):
            self.rsa.hashAndSign(self.message,
                                            "PKC1", "sha1")

    def test_hashAndSign_RSAPKCS1_sha1_notSet(self):
        sigBytes = self.rsa.hashAndSign(self.message,
            "PKCS1")
        self.assertEqual(sigBytes, bytearray(
            b'\x17\x50\x15\xbd\xa5\x0a\xbe\x0f\xa7\xd3\x9a\x83\x53\x88' +
            b'\x5c\xa0\x1b\xe3\xa7\xe7\xfc\xc5\x50\x45\x74\x41\x11\x36' +
            b'\x2e\xe1\x91\x44\x73\xa4\x8d\xc5\x37\xd9\x56\x29\x4b\x9e' +
            b'\x20\xa1\xef\x66\x1d\x58\x53\x7a\xcd\xc8\xde\x90\x8f\xa0' +
            b'\x50\x63\x0f\xcc\x27\x2e\x6d\x00\x10\x45\xe6\xfd\xee\xd2' +
            b'\xd1\x05\x31\xc8\x60\x33\x34\xc2\xe8\xdb\x39\xe7\x3e\x6d' +
            b'\x96\x65\xee\x13\x43\xf9\xe4\x19\x83\x02\xd2\x20\x1b\x44' +
            b'\xe8\xe8\xd0\x6b\x3e\xf4\x9c\xee\x61\x97\x58\x21\x63\xa8' +
            b'\x49\x00\x89\xca\x65\x4c\x00\x12\xfc\xe1\xba\x65\x11\x08' +
            b'\x97\x50'))

    def test_hashAndSign_RSAPKCS1_sha224(self):
        sigBytes = self.rsa.hashAndSign(self.message,
            "PKCS1", "sha224")
        self.assertEqual(sigBytes, bytearray(
            b'\x57\x67\x7b\x08\x9e\x20\x54\x86\xdf\x4f\x56\x75\x59\x72' +
            b'\xe3\xaf\x88\xca\xbb\xc2\x3e\xfe\x29\x43\x9b\x8d\x1e\x60' +
            b'\xac\x22\x6e\x99\x0d\xa4\x87\x85\x73\x92\x85\x6d\x12\xcd' +
            b'\xce\xa3\x87\xa2\x69\xd1\xbb\xbc\x12\x85\x49\xa1\x13\x5a' +
            b'\xb0\x62\x20\x1c\xab\x8a\xc0\x88\x86\xa3\x13\xaf\x85\x54' +
            b'\x50\x6d\x7a\x93\x85\x5b\x84\x30\x86\xa1\xbf\x3d\xfb\xcb' +
            b'\x00\x4c\xcd\xe7\x79\xc0\x84\xff\xa1\x72\x4b\x41\xd1\x7e' +
            b'\x10\xc8\xdd\x67\xdc\x0d\xf2\x62\x00\x37\x65\x50\xed\xa1' +
            b'\x44\x55\xd9\xb0\xb3\x1f\x1d\x8c\x5e\x8b\xb1\xd3\xd9\x63' +
            b'\xd0\xd5'))

    def test_hashAndSign_RSAPKCS1_sha256(self):
        sigBytes = self.rsa.hashAndSign(self.message,
            "PKCS1", "sha256")
        self.assertEqual(sigBytes, bytearray(
            b'\x0b\x20\xe5\x09\x3c\x2a\x92\x62\x33\x10\x8a\xfb\xdd\x85' +
            b'\x1b\x88\xee\xb5\x54\xf4\xbe\xaa\x7b\x18\xe5\x15\x19\xf7' +
            b'\xd0\xec\x53\xb1\x81\xa3\xb0\x3e\x84\x84\xba\x8d\xe2\xaa' +
            b'\x78\x64\xa4\x02\xe2\x20\x8e\x84\xec\x99\x14\xaf\x9d\x77' +
            b'\x6e\xd1\x3c\x48\xbd\xeb\x64\x84\x25\x4d\xe1\x69\x31\x8a' +
            b'\x87\xc4\x0f\x22\x65\xff\x16\x71\x4e\xae\x8a\xee\x2b\xc9' +
            b'\xc3\xcb\x4d\xee\x04\x5e\x4f\x5d\x9d\x62\x52\x10\x12\x1b' +
            b'\xfc\xf2\xbe\xd8\xd3\xff\xa6\x02\xce\x27\xff\xf4\xe6\x1c' +
            b'\xf9\xbb\x65\x0e\x71\xa6\x92\x1a\xe6\xff\xa2\x96\xcb\x11' +
            b'\xbd\xbb'))

    def test_hashAndSign_RSAPKCS1_sha384(self):
        sigBytes = self.rsa.hashAndSign(self.message,
            "PKCS1", "sha384")
        self.assertEqual(sigBytes, bytearray(
            b'\x7e\x3c\xcb\x6a\xb0\x3b\x41\x9a\x3e\x54\xf8\x13\x37\xa3' +
            b'\xc3\xf7\x2e\x8c\x65\xbb\xd1\x9d\xdd\x50\x24\x6a\x36\xf5' +
            b'\x1f\x58\x74\x1e\xc2\x45\xd2\xd0\xf0\x76\x77\xa4\xf8\x8a' +
            b'\xa3\xb1\xca\xee\xcd\xff\xe5\xfd\x6e\xdc\xf8\xb8\xbc\xfb' +
            b'\x56\x96\x37\xad\x02\xeb\x15\x4d\x17\xb8\x7a\x8f\x00\xd0' +
            b'\xe6\x18\xa7\xf4\xa7\x0c\xe4\x07\xf2\x03\x59\x15\x3e\x5f' +
            b'\x4a\x4d\x97\x44\xf3\xf3\xff\x44\x12\x0c\x08\xa4\x60\x50' +
            b'\x0f\x03\x0f\xd3\x39\x8e\x97\xfc\xae\xf9\xd0\xa7\xe2\xac' +
            b'\xef\x19\xa8\x1f\x70\x68\x05\xbe\x5f\xc0\x03\xd7\x8e\x5b' +
            b'\x29\xc0'))

    def test_hashAndSign_RSAPKCS1_sha512(self):
        sigBytes = self.rsa.hashAndSign(self.message,
            "PKCS1", "sha512")
        self.assertEqual(sigBytes, bytearray(
            b'\x8b\x57\xa6\xf9\x16\x06\xba\x48\x13\xb8\x35\x36\x58\x1e' +
            b'\xb1\x5d\x72\x87\x5d\xcb\xb0\xa5\x14\xb4\xc0\x3b\x6d\xf8' +
            b'\xf2\x02\xfa\x85\x56\xe4\x00\x21\x22\xbe\xda\xf2\x6e\xaa' +
            b'\x10\x7e\xce\x48\x60\x75\x23\x79\xec\x8b\xaa\x64\xf4\x00' +
            b'\x98\xbe\x92\xa4\x21\x4b\x69\xe9\x8b\x24\xae\x1c\xc4\xd2' +
            b'\xf4\x57\xcf\xf4\xf4\x05\xa8\x2e\xf9\x4c\x5f\x8d\xfa\xad' +
            b'\xd3\x07\x8d\x7a\x92\x24\x88\x7d\xb8\x6c\x32\x18\xbf\x53' +
            b'\xc9\x77\x9e\xd0\x98\x95\xb2\xcf\xb8\x4f\x1f\xad\x2e\x5b' +
            b'\x1f\x8e\x4b\x20\x9c\x57\x85\xb9\xce\x33\x2c\xd4\x13\x56' +
            b'\xc1\x71'))

    def test_hashAndVerify_PKCS1_sha1_notSet(self):
        sigBytes = bytearray(
            b'\x17\x50\x15\xbd\xa5\x0a\xbe\x0f\xa7\xd3\x9a\x83\x53\x88' +
            b'\x5c\xa0\x1b\xe3\xa7\xe7\xfc\xc5\x50\x45\x74\x41\x11\x36' +
            b'\x2e\xe1\x91\x44\x73\xa4\x8d\xc5\x37\xd9\x56\x29\x4b\x9e' +
            b'\x20\xa1\xef\x66\x1d\x58\x53\x7a\xcd\xc8\xde\x90\x8f\xa0' +
            b'\x50\x63\x0f\xcc\x27\x2e\x6d\x00\x10\x45\xe6\xfd\xee\xd2' +
            b'\xd1\x05\x31\xc8\x60\x33\x34\xc2\xe8\xdb\x39\xe7\x3e\x6d' +
            b'\x96\x65\xee\x13\x43\xf9\xe4\x19\x83\x02\xd2\x20\x1b\x44' +
            b'\xe8\xe8\xd0\x6b\x3e\xf4\x9c\xee\x61\x97\x58\x21\x63\xa8' +
            b'\x49\x00\x89\xca\x65\x4c\x00\x12\xfc\xe1\xba\x65\x11\x08' +
            b'\x97\x50')

        self.assertTrue(self.rsa.hashAndVerify(sigBytes,
                                               self.message, "PKCS1"))

    def test_hashAndVerify_PKCS1_sha224(self):
        sigBytes = bytearray(
            b'\x57\x67\x7b\x08\x9e\x20\x54\x86\xdf\x4f\x56\x75\x59\x72' +
            b'\xe3\xaf\x88\xca\xbb\xc2\x3e\xfe\x29\x43\x9b\x8d\x1e\x60' +
            b'\xac\x22\x6e\x99\x0d\xa4\x87\x85\x73\x92\x85\x6d\x12\xcd' +
            b'\xce\xa3\x87\xa2\x69\xd1\xbb\xbc\x12\x85\x49\xa1\x13\x5a' +
            b'\xb0\x62\x20\x1c\xab\x8a\xc0\x88\x86\xa3\x13\xaf\x85\x54' +
            b'\x50\x6d\x7a\x93\x85\x5b\x84\x30\x86\xa1\xbf\x3d\xfb\xcb' +
            b'\x00\x4c\xcd\xe7\x79\xc0\x84\xff\xa1\x72\x4b\x41\xd1\x7e' +
            b'\x10\xc8\xdd\x67\xdc\x0d\xf2\x62\x00\x37\x65\x50\xed\xa1' +
            b'\x44\x55\xd9\xb0\xb3\x1f\x1d\x8c\x5e\x8b\xb1\xd3\xd9\x63' +
            b'\xd0\xd5')

        self.assertTrue(self.rsa.hashAndVerify(sigBytes,
                                               self.message, "PKCS1",
                                               'sha224'))

    def test_hashAndVerify_PKCS1_sha256(self):
        sigBytes = bytearray(
            b'\x0b\x20\xe5\x09\x3c\x2a\x92\x62\x33\x10\x8a\xfb\xdd\x85' +
            b'\x1b\x88\xee\xb5\x54\xf4\xbe\xaa\x7b\x18\xe5\x15\x19\xf7' +
            b'\xd0\xec\x53\xb1\x81\xa3\xb0\x3e\x84\x84\xba\x8d\xe2\xaa' +
            b'\x78\x64\xa4\x02\xe2\x20\x8e\x84\xec\x99\x14\xaf\x9d\x77' +
            b'\x6e\xd1\x3c\x48\xbd\xeb\x64\x84\x25\x4d\xe1\x69\x31\x8a' +
            b'\x87\xc4\x0f\x22\x65\xff\x16\x71\x4e\xae\x8a\xee\x2b\xc9' +
            b'\xc3\xcb\x4d\xee\x04\x5e\x4f\x5d\x9d\x62\x52\x10\x12\x1b' +
            b'\xfc\xf2\xbe\xd8\xd3\xff\xa6\x02\xce\x27\xff\xf4\xe6\x1c' +
            b'\xf9\xbb\x65\x0e\x71\xa6\x92\x1a\xe6\xff\xa2\x96\xcb\x11' +
            b'\xbd\xbb')

        self.assertTrue(self.rsa.hashAndVerify(sigBytes,
                                               self.message, "PKCS1",
                                               'sha256'))

    def test_hashAndVerify_PKCS1_sha384(self):
        sigBytes = bytearray(
            b'\x7e\x3c\xcb\x6a\xb0\x3b\x41\x9a\x3e\x54\xf8\x13\x37\xa3' +
            b'\xc3\xf7\x2e\x8c\x65\xbb\xd1\x9d\xdd\x50\x24\x6a\x36\xf5' +
            b'\x1f\x58\x74\x1e\xc2\x45\xd2\xd0\xf0\x76\x77\xa4\xf8\x8a' +
            b'\xa3\xb1\xca\xee\xcd\xff\xe5\xfd\x6e\xdc\xf8\xb8\xbc\xfb' +
            b'\x56\x96\x37\xad\x02\xeb\x15\x4d\x17\xb8\x7a\x8f\x00\xd0' +
            b'\xe6\x18\xa7\xf4\xa7\x0c\xe4\x07\xf2\x03\x59\x15\x3e\x5f' +
            b'\x4a\x4d\x97\x44\xf3\xf3\xff\x44\x12\x0c\x08\xa4\x60\x50' +
            b'\x0f\x03\x0f\xd3\x39\x8e\x97\xfc\xae\xf9\xd0\xa7\xe2\xac' +
            b'\xef\x19\xa8\x1f\x70\x68\x05\xbe\x5f\xc0\x03\xd7\x8e\x5b' +
            b'\x29\xc0')

        self.assertTrue(self.rsa.hashAndVerify(sigBytes,
                                               self.message, "PKCS1",
                                               'sha384'))

    def test_hashAndVerify_PKCS1_sha512(self):
        sigBytes = bytearray(
            b'\x8b\x57\xa6\xf9\x16\x06\xba\x48\x13\xb8\x35\x36\x58\x1e' +
            b'\xb1\x5d\x72\x87\x5d\xcb\xb0\xa5\x14\xb4\xc0\x3b\x6d\xf8' +
            b'\xf2\x02\xfa\x85\x56\xe4\x00\x21\x22\xbe\xda\xf2\x6e\xaa' +
            b'\x10\x7e\xce\x48\x60\x75\x23\x79\xec\x8b\xaa\x64\xf4\x00' +
            b'\x98\xbe\x92\xa4\x21\x4b\x69\xe9\x8b\x24\xae\x1c\xc4\xd2' +
            b'\xf4\x57\xcf\xf4\xf4\x05\xa8\x2e\xf9\x4c\x5f\x8d\xfa\xad' +
            b'\xd3\x07\x8d\x7a\x92\x24\x88\x7d\xb8\x6c\x32\x18\xbf\x53' +
            b'\xc9\x77\x9e\xd0\x98\x95\xb2\xcf\xb8\x4f\x1f\xad\x2e\x5b' +
            b'\x1f\x8e\x4b\x20\x9c\x57\x85\xb9\xce\x33\x2c\xd4\x13\x56' +
            b'\xc1\x71')

        self.assertTrue(self.rsa.hashAndVerify(sigBytes,
                                               self.message, "PKCS1",
                                               'sha512'))
    def test_verify_PKCS1_sha512(self):
        sigBytes = bytearray(
            b'\x8b\x57\xa6\xf9\x16\x06\xba\x48\x13\xb8\x35\x36\x58\x1e' +
            b'\xb1\x5d\x72\x87\x5d\xcb\xb0\xa5\x14\xb4\xc0\x3b\x6d\xf8' +
            b'\xf2\x02\xfa\x85\x56\xe4\x00\x21\x22\xbe\xda\xf2\x6e\xaa' +
            b'\x10\x7e\xce\x48\x60\x75\x23\x79\xec\x8b\xaa\x64\xf4\x00' +
            b'\x98\xbe\x92\xa4\x21\x4b\x69\xe9\x8b\x24\xae\x1c\xc4\xd2' +
            b'\xf4\x57\xcf\xf4\xf4\x05\xa8\x2e\xf9\x4c\x5f\x8d\xfa\xad' +
            b'\xd3\x07\x8d\x7a\x92\x24\x88\x7d\xb8\x6c\x32\x18\xbf\x53' +
            b'\xc9\x77\x9e\xd0\x98\x95\xb2\xcf\xb8\x4f\x1f\xad\x2e\x5b' +
            b'\x1f\x8e\x4b\x20\x9c\x57\x85\xb9\xce\x33\x2c\xd4\x13\x56' +
            b'\xc1\x71')
        self.assertTrue(self.rsa.verify(sigBytes,
                                        secureHash(self.message, "sha512"),
                                        hashAlg="sha512"))

    def test_verify_invalid_PKCS1_sha512(self):
        sigBytes = bytearray(
            b'\x0b\x57\xa6\xf9\x16\x06\xba\x48\x13\xb8\x35\x36\x58\x1e' +
            b'\xb1\x5d\x72\x87\x5d\xcb\xb0\xa5\x14\xb4\xc0\x3b\x6d\xf8' +
            b'\xf2\x02\xfa\x85\x56\xe4\x00\x21\x22\xbe\xda\xf2\x6e\xaa' +
            b'\x10\x7e\xce\x48\x60\x75\x23\x79\xec\x8b\xaa\x64\xf4\x00' +
            b'\x98\xbe\x92\xa4\x21\x4b\x69\xe9\x8b\x24\xae\x1c\xc4\xd2' +
            b'\xf4\x57\xcf\xf4\xf4\x05\xa8\x2e\xf9\x4c\x5f\x8d\xfa\xad' +
            b'\xd3\x07\x8d\x7a\x92\x24\x88\x7d\xb8\x6c\x32\x18\xbf\x53' +
            b'\xc9\x77\x9e\xd0\x98\x95\xb2\xcf\xb8\x4f\x1f\xad\x2e\x5b' +
            b'\x1f\x8e\x4b\x20\x9c\x57\x85\xb9\xce\x33\x2c\xd4\x13\x56' +
            b'\xc1\x71')
        self.assertFalse(self.rsa.verify(sigBytes,
                                         secureHash(self.message, "sha512"),
                                         hashAlg="sha512"))


# because RSAKey is an abstract class...
class TestRSAKey(unittest.TestCase):

    # random RSA parameters
    N = int("101394340507163232476731540998223559348384567842249950630680016"
            "729829651735259973644737329194901739140557378171784099933376993"
            "53519793819698299093375577631")
    e = 65537
    d = int("141745721972918790698280063566067268498148845185400775263435953"
            "111621933337897734637889622802200979017278309730638712431978569"
            "771023240787627463565420833")
    p = int("903614668974112441151570413608036278756730123846327797584414732"
            "71561046135679")
    q = int("112209710608480690748363491355148749700390327497055102381924341"
            "581861552321889")
    dP = int("37883511062045429960298073888481933556799848761465588242411735"
             "654811958185817")
    dQ = int("62620473256245674709410658602365234471246407950887183034263101"
             "286525236349249")
    qInv = int("479278327226690415958629934820002183615697717603796111150941"
               "44623120451328875")

    def test___init__(self):
        rsa = Python_RSAKey()

        self.assertIsNotNone(rsa)

    def test___init___with_values(self):
        rsa = Python_RSAKey(self.N, self.e, self.d, self.p, self.q, self.dP,
                            self.dQ, self.qInv)

        self.assertIsNotNone(rsa)

    def test_hashAndSign(self):
        rsa = Python_RSAKey(self.N, self.e, self.d, self.p, self.q, self.dP,
                            self.dQ, self.qInv)

        sigBytes = rsa.hashAndSign(bytearray(b'text to sign'))

        self.assertEqual(bytearray(
            b'K\x7f\xf2\xca\x81\xf0A1\x95\xb1\x19\xe3\xd7QTL*Q|\xb6\x04' +
            b'\xbdG\x88H\x12\xc3\xe2\xb3\x97\xd2\xcd\xd8\xe8^Zn^\x8f\x1a' +
            b'\xae\x9a\x0b)\xb5K\xe8\x98|R\xac\xdc\xdc\n\x7f\x8b\xe7\xe6' +
            b'HQ\xc3hS\x19'), sigBytes)

    def test_hashAndSign_PSS(self):
        rsa = Python_RSAKey(self.N, self.e, self.d, self.p, self.q, self.dP,
                            self.dQ, self.qInv)

        sigBytes = rsa.hashAndSign(bytearray(b'text to sign'), "PSS", "sha1")
        self.assertEqual(bytearray(b'op\xfa\x1d\xfa\xe8i\xf2zV\x9a\xf4\x8d' +
                                   b'\xf1\xaf:\x1a\xb6\xce\xae3\xd1\xc2E[EG' +
                                   b'\x8ba\xfe.\x8e\x9dJ\xc9<Q\x05\xeaO\x8c' +
                                   b'\x8b\x01\xaer\x0f\xd8R\xb1\x1f\xb0\x06' +
                                   b'\x95\\\x8c\xae\xc9\xec\xa5{\x13' +
                                   b'\xa2ms'), sigBytes)

    def test_hashAndVerify(self):
        rsa = Python_RSAKey(self.N, self.e)

        sigBytes = bytearray(
            b'K\x7f\xf2\xca\x81\xf0A1\x95\xb1\x19\xe3\xd7QTL*Q|\xb6\x04' +
            b'\xbdG\x88H\x12\xc3\xe2\xb3\x97\xd2\xcd\xd8\xe8^Zn^\x8f\x1a' +
            b'\xae\x9a\x0b)\xb5K\xe8\x98|R\xac\xdc\xdc\n\x7f\x8b\xe7\xe6' +
            b'HQ\xc3hS\x19')

        self.assertTrue(rsa.hashAndVerify(sigBytes,
                                          bytearray(b'text to sign')))

    def test_hashAndVerify_PSS(self):
        rsa = Python_RSAKey(self.N, self.e)

        sigBytes = bytearray(
            b'op\xfa\x1d\xfa\xe8i\xf2zV\x9a\xf4\x8d\xf1\xaf:\x1a\xb6\xce' +
            b'\xae3\xd1\xc2E[EG\x8ba\xfe.\x8e\x9dJ\xc9<Q\x05\xeaO\x8c\x8b' +
            b'\x01\xaer\x0f\xd8R\xb1\x1f\xb0\x06\x95\\\x8c\xae\xc9\xec' +
            b'\xa5{\x13\xa2ms')

        self.assertTrue(rsa.hashAndVerify(sigBytes, bytearray(b'text to sign'),
                                          "PSS", 'sha1'))

    def test_verify_PSS(self):
        rsa = Python_RSAKey(self.N, self.e)

        sigBytes = bytearray(
            b'op\xfa\x1d\xfa\xe8i\xf2zV\x9a\xf4\x8d\xf1\xaf:\x1a\xb6\xce' +
            b'\xae3\xd1\xc2E[EG\x8ba\xfe.\x8e\x9dJ\xc9<Q\x05\xeaO\x8c\x8b' +
            b'\x01\xaer\x0f\xd8R\xb1\x1f\xb0\x06\x95\\\x8c\xae\xc9\xec' +
            b'\xa5{\x13\xa2ms')

        self.assertTrue(rsa.verify(sigBytes,
                                   secureHash(bytearray(b'text to sign'),
                                              'sha1'),
                                   "pss", 'sha1', 0))

    def test_verify_invalid_PSS(self):
        rsa = Python_RSAKey(self.N, self.e)

        sigBytes = bytearray(
            b'Xp\xfa\x1d\xfa\xe8i\xf2zV\x9a\xf4\x8d\xf1\xaf:\x1a\xb6\xce' +
            b'\xae3\xd1\xc2E[EG\x8ba\xfe.\x8e\x9dJ\xc9<Q\x05\xeaO\x8c\x8b' +
            b'\x01\xaer\x0f\xd8R\xb1\x1f\xb0\x06\x95\\\x8c\xae\xc9\xec' +
            b'\xa5{\x13\xa2ms')

        self.assertFalse(rsa.verify(sigBytes,
                                    secureHash(bytearray(b'text to sign'),
                                               'sha1'),
                                    "pss", 'sha1', 0))

    def test_hashAndVerify_without_NULL_encoding_of_SHA1(self):
        rsa = Python_RSAKey(self.N, self.e)

        sigBytes = bytearray(
            b'F\xe7\x8a>\x8a<;Cj\xdd\xea\x7f\x9d\x0c\xfd\xa7r\xd8\xa1O' +
            b'\xe1\xf5\x174\x0bR\xad:+\xc9C\x06\xf4\x88n\tp\x14FJ=\xfa' +
            b'\x8b\xefc\xe2\xdf\x00e\xc1\x1e\xe8\xd2\x97@\x8a\x96\xe2' +
            b'\x039Y_\x9c\xc9')

        self.assertTrue(rsa.hashAndVerify(sigBytes,
                                          bytearray(b'text to sign')))

    def test_hashAndVerify_with_invalid_signature(self):
        rsa = Python_RSAKey(self.N, self.e)

        sigBytes = bytearray(64)

        self.assertFalse(rsa.hashAndVerify(sigBytes,
                                           bytearray(b'text to sign')))

    def test_hashAndVerify_with_slightly_wrong_signature(self):
        rsa = Python_RSAKey(self.N, self.e)

        sigBytes = bytearray(
            b'K\x7f\xf2\xca\x81\xf0A1\x95\xb1\x19\xe3\xd7QTL*Q|\xb6\x04' +
            b'\xbdG\x88H\x12\xc3\xe2\xb3\x97\xd2\xcd\xd8\xe8^Zn^\x8f\x1a' +
            b'\xae\x9a\x0b)\xb5K\xe8\x98|R\xac\xdc\xdc\n\x7f\x8b\xe7\xe6' +
            b'HQ\xc3hS\x19')
        sigBytes[0] = 255

        self.assertFalse(rsa.hashAndVerify(sigBytes,
                                           bytearray(b'text to sign')))

    def test_addPKCS1SHA1Prefix(self):
        data = bytearray(b' sha-1 hash of data ')

        self.assertEqual(RSAKey.addPKCS1SHA1Prefix(data), bytearray(
            b'0!0\t\x06\x05+\x0e\x03\x02\x1a\x05\x00\x04\x14' + 
            b' sha-1 hash of data '))

    def test_addPKCS1SHA1Prefix_without_NULL(self):
        data = bytearray(b' sha-1 hash of data ')

        self.assertEqual(RSAKey.addPKCS1SHA1Prefix(data, False), bytearray(
            b'0\x1f0\x07\x06\x05+\x0e\x03\x02\x1a\x04\x14' +
            b' sha-1 hash of data '))

    def test_addPKCS1Prefix(self):
        data = bytearray(b' sha-1 hash of data ')

        self.assertEqual(RSAKey.addPKCS1Prefix(data, 'sha1'), bytearray(
            b'0!0\t\x06\x05+\x0e\x03\x02\x1a\x05\x00\x04\x14' +
            b' sha-1 hash of data '))
