/**********************************************************************
 * $read_vecval example -- PLI application using ACC routines
 *
 * C source to read logic values of a net and print the net name and 
 * current logic value.
 *
 * Usage: $read_vecval(<net_name>);
 *
 * For the book, "The Verilog PLI Handbook" by Stuart Sutherland
 *  Book copyright 1999, Kluwer Academic Publishers, Norwell, MA, USA
 *   Contact: www.wkap.il
 *  Example copyright 1998, Sutherland HDL Inc, Portland, Oregon, USA
 *   Contact: www.sutherland.com or (503) 692-0898
 *
 * Routine definitions for a veriusertfs array:
 *  /* routine prototypes -/
 *   extern int PLIbook_ReadVecVal_checktf(),
 *              PLIbook_ReadVecVal_calltf();
 *  /* table entries -/
 *   {usertask,                       /* type of PLI routine -/
 *     0,                             /* user_data value -/
 *     PLIbook_ReadVecVal_checktf,    /* checktf routine -/
 *     0,                             /* sizetf routine -/
 *     PLIbook_ReadVecVal_calltf,     /* calltf routine -/
 *     0,                             /* misctf routine -/
 *     "$read_vecval",                /* system task/function name -/
 *     1                              /* forward reference = true -/
 *   },
 *********************************************************************/

#include "veriuser.h"         /* IEEE 1364 PLI TF  routine library */
#include "acc_user.h"         /* IEEE 1364 PLI ACC routine library */

/* prototypes of subroutines used by the calltf routine */
char PLIbook_get_4state_val();
int  PLIbook_getbit() ;

/**********************************************************************
 * checktf routine
 *********************************************************************/
int PLIbook_ReadVecVal_checktf()
{
  int       arg_type;
  acc_initialize();
  if (tf_nump() != 1)
    tf_error("$timescale_info must have 1 argument.");
  else if (tf_typep(1) == TF_NULLPARAM)
    tf_error("$time_info arg cannot be null.");
  else {
    arg_type = acc_fetch_type(acc_handle_tfarg(1));
    if (   (arg_type != accNet)
        && (arg_type != accReg) )
      tf_error("$read_vecval arg must be a net or reg.");
  }
  acc_close();
  return(0);
}

/**********************************************************************
 * calltf routine
 *********************************************************************/
int PLIbook_ReadVecVal_calltf() 
{
  handle       vector_h;
  s_acc_value  vector_val;     /* structure to receive vector value */
  int          i, vector_size, array_size, avalbit, bvalbit, bit_num;
  char         vlogval;

  vector_h = acc_handle_tfarg(1);

  vector_size = acc_fetch_size(vector_h);   /* determine number of...*/
  array_size  = ((vector_size-1) / 32 + 1); /* ...elements in array  */

  vector_val.value.vector = (p_acc_vecval)malloc(array_size * sizeof(p_acc_vecval));

  vector_val.format = accVectorVal;        /* set value format field */

  acc_fetch_value(vector_h,"%%",&vector_val); /* read vector's value */

  io_printf("\nVector %s encoded value:\n",
             acc_fetch_name(vector_h));
  for (i=0; i<array_size; i++) {
    /* the following loop assumes the Verilog LSB is bit 0 */
    for (bit_num=0; bit_num<=31; bit_num++) {
      avalbit=PLIbook_getbit(vector_val.value.vector[i].aval, bit_num);
      bvalbit=PLIbook_getbit(vector_val.value.vector[i].bval, bit_num);
      vlogval=PLIbook_get_4state_val(avalbit, bvalbit);
      io_printf("  bit[%2d]  aval/bval = %d/%d  4-state value = %c\n",
                 (i*32+bit_num), avalbit, bvalbit, vlogval);
      /* quit when reach last bit of Verilog vector */
      if ((i*32+bit_num) == vector_size-1) break;
    }
  }
  return(0);
}

/**********************************************************************
 * Function to determine if a specific bit is set in a 32-bit word.
 * Sets the least-significant bit of a mask value to 1 and shifts the
 * mask left to the desired bit number.
 *********************************************************************/
int PLIbook_getbit(int word, int bit_num) 
{
  int mask;
  mask = 0x00000001 << bit_num;
  return((word & mask)? TRUE: FALSE);
}

/**********************************************************************
 * Function to convert aval/bval encoding to 4-state logic represented
 * as a C character.
 *********************************************************************/
char PLIbook_get_4state_val(int aval, int bval)
{
  if      (!bval && !aval) return('0');
  else if (!bval &&  aval) return('1');
  else if ( bval && !aval) return('z');
  else                     return('x');
}

/*********************************************************************/
