/**********************************************************************
 * $display_all_nets example -- C source code using TF/ACC PLI routines
 *
 * C source to scan through a module and list the names of all nets in
 * the module with the current logic value. For run-time performance, 
 * the handles for all nets are collected once at the start of 
 * simulation, and the list of handles used in each call to the 
 * application.
 *
 * For the book, "The Verilog PLI Handbook" by Stuart Sutherland
 *  Book copyright 1999, Kluwer Academic Publishers, Norwell, MA, USA
 *   Contact: www.wkap.il
 *  Example copyright 1998, Sutherland HDL Inc, Portland, Oregon, USA
 *   Contact: www.sutherland.com or (503) 692-0898
 *
 * Usage: $display_all_nets(<module_instance>);
 *
 * Routine definitions for a veriusertfs array:
 *  /* routine prototypes -/
 *   extern int PLIbook_DisplayNets_checktf(),
 *              PLIbook_DisplayNets_calltf(),
 *              PLIbook_DisplayNets_misctf();
 *  /* table entries -/
 *   {usertask,                       /* type of PLI routine -/
 *     0,                             /* user_data value -/
 *     PLIbook_DisplayNets_checktf,   /* checktf routine -/
 *     0,                             /* sizetf routine -/
 *     PLIbook_DisplayNets_calltf,    /* calltf routine -/
 *     PLIbook_DisplayNets_misctf,    /* misctf routine -/
 *     "$display_all_nets",           /* system task/function name -/
 *     1                              /* forward reference = true -/
 *   },
 *********************************************************************/

#include "veriuser.h"         /* IEEE 1364 PLI TF  routine library */
#include "acc_user.h"         /* IEEE 1364 PLI ACC routine library */
/**********************************************************************
 * structure definition for data to be passed from misctf to calltf
 *********************************************************************/
typedef struct PLIbook_NetData {
  handle  module_h;
  handle *net_array;
  int     net_count;
} PLIbook_NetData_s, *PLIbook_NetData_p;

/**********************************************************************
 * checktf routine
 *********************************************************************/
int PLIbook_DisplayNets_checktf()
{
  acc_initialize();
  if (tf_nump() != 1)
    tf_error("$display_all_nets must have 1 argument.");
  else if (tf_typep(1) == TF_NULLPARAM)
    tf_error("$display_all_nets arg cannot be null.");
  else if (acc_fetch_type(acc_handle_tfarg(1)) != accModule)
    tf_error("$display_all_nets arg must be a module instance.");
  acc_close();
  return(0);
}

/**********************************************************************
 * misctf routine
 *********************************************************************/
int PLIbook_DisplayNets_misctf(int user_data, int reason)
{
  PLIbook_NetData_p  net_data;
  handle  mod_h;
  
  acc_initialize();
  acc_configure(accDisplayWarnings, "true");

  switch (reason) {
    case REASON_ENDOFCOMPILE:
      acc_initialize();
      net_data = (PLIbook_NetData_p)malloc(sizeof(PLIbook_NetData_s));
      net_data->module_h  = acc_handle_tfarg(1);
      net_data->net_array = acc_collect(acc_next_net,
                                        net_data->module_h,
                                        &net_data->net_count);
      io_printf("Total nets collected in misctf routine = %d\n",
                net_data->net_count);
      tf_setworkarea((char *)net_data);
      acc_close();
      break;
    case REASON_FINISH:
      net_data = (PLIbook_NetData_p)tf_getworkarea();
      acc_free(net_data->net_array);
      break;
  }
  acc_close();
  return(0);
}

/**********************************************************************
 * calltf routine
 *********************************************************************/
int PLIbook_DisplayNets_calltf()
{
  PLIbook_NetData_p  net_data;
  int i;

  acc_initialize();
  acc_configure(accDisplayWarnings, "true");

  net_data = (PLIbook_NetData_p)tf_getworkarea();

  io_printf("\nAt time %s, nets in module %s (%s):\n",
            tf_strgettime(),
            acc_fetch_fullname(net_data->module_h),
            acc_fetch_defname(net_data->module_h));
  for (i=0; i<net_data->net_count; i++) {
    io_printf("  %-13s  value is  %s (hex)\n",
              acc_fetch_name(net_data->net_array[i]),
              acc_fetch_value(net_data->net_array[i], "%h", null));
  }

  acc_close();
  return(0);
}
/*********************************************************************/

