"""
Unit Test Case
"""
#  Copyright (C) 2004  Henning Jacobs <henning@srcco.de>
#
#  This program is free software; you can redistribute it and/or modify
#  it under the terms of the GNU General Public License as published by
#  the Free Software Foundation; either version 2 of the License, or
#  (at your option) any later version.
#
#  $Id: testvcalendar.py 82 2004-07-11 13:01:44Z henning $

import unittest
from vcalendar import vEvent, vCalendar

DEMOCOMMENT = u"""This is a simple Test Case for my vCalendar Python Module.
Here comes german umlauts: \xe4\xc4\xfc\xdc\xf6\xd6\xdf (this string
is Unicode and will be written to disk as Latin-1)
Now some special vCalendar chars: \t (tab) ; : , ? (note the escaping!)
"""

RFCTESTDESCR1 = """Project xyz Review Meeting Minutes
Agenda\n1. Review of project version 1.0 requirements.
2. Definition of project processes.
3. Review of project schedule.
Participants: John Smith, Jane Doe, Jim Dandy
-It was decided that the requirements need to be signed off by product marketing.
-Project processes were accepted.
-Project schedule needs to account for scheduled holidays and employee vacation time. Check with HR for specific dates.
-New schedule will be distributed by Friday.
-Next weeks meeting is cancelled. No meeting until 3/23."""

RFCTESTCAL1 = r"""BEGIN:VCALENDAR
VERSION:2.0
PRODID:-//ABC Corporation//NONSGML My Product//EN
BEGIN:VJOURNAL
DTSTAMP:19970324T120000Z
UID:uid5@host1.com
ORGANIZER:MAILTO:jsmith@host.com
STATUS:DRAFT
CLASS:PUBLIC
CATEGORIES:Project Report, XYZ, Weekly Meeting
DESCRIPTION:Project xyz Review Meeting Minutes\n
 Agenda\n1. Review of project version 1.0 requirements.\n2.
  Definition
  of project processes.\n3. Review of project schedule.\n
 Participants: John Smith, Jane Doe, Jim Dandy\n-It was
  decided that the requirements need to be signed off by
  product marketing.\n-Project processes were accepted.\n
 -Project schedule needs to account for scheduled holidays
  and employee vacation time. Check with HR for specific
  dates.\n-New schedule will be distributed by Friday.\n-
 Next weeks meeting is cancelled. No meeting until 3/23.
END:VJOURNAL
END:VCALENDAR"""

def fillDemoJournal(jour):
    "Completely fill a vEvent with demo values"
    jour.summary.set("Summary")
    jour.description.set("Description")
    jour.categories.set("Demo, Business, Private, TestCase")
    jour.comment.set(DEMOCOMMENT)
    jour.location.set("Somewhere on terra")
    jour.contact.set("John Smith, Ocean Drive 3, Atlantis")
    jour.dtstart.set("2004-02-22")
    jour.url.set("http://www.mydomain.nowhere")
    jour.uid.set("-//TEST//TestvEvent@mydomain.nowhere//123456789")

class vCalendarTestCase(unittest.TestCase):
    def setUp(self):
        self.calendar = vCalendar()
        self.demojournal = vEvent()
        fillDemoJournal(self.demojournal)
        
    def testReadWrite(self):
        "writing and reading vCalendar from stream"
        self.calendar.add(self.demojournal)
        jourbefore = self.demojournal
        import StringIO
        stream = StringIO.StringIO()
        self.calendar.SaveToStream(stream)
        stream.seek(0)
        self.calendar.clear()
        self.calendar.LoadFromStream(stream)
        handle = self.calendar.sortedlist()[0]
        jourafter = self.calendar[handle]
        self.assertEqual(jourbefore.VCF_repr(), jourafter.VCF_repr())

    def testFieldValue(self):
        "compare journal values with our demo values"
        self.assertEqual(self.demojournal.getFieldValueStr("Comment"), DEMOCOMMENT)

    def testRFCConformity(self):
        "rfc conformity"
        self.calendar.add(vEvent(RFCTESTCAL1))
        self.assertEqual(self.calendar[1].description.get(), RFCTESTDESCR1)

    def testSort(self):
        "vCalendar sorting by Date"
        jour = vEvent()
        jour.dtstart.set("2004-02-22")
        hdl1 = self.calendar.add(jour)
        jour = vEvent()
        jour.dtstart.set("2001-09-11")
        hdl2 = self.calendar.add(jour)
        jour = vEvent()
        jour.dtstart.set("2004-08-19")
        hdl3 = self.calendar.add(jour)
        self.assertEqual(self.calendar.sortedlist(), [hdl2, hdl1, hdl3])

if __name__ == "__main__":
    unittest.main()

