/*
 * This file is part of the xTuple ERP: PostBooks Edition, a free and
 * open source Enterprise Resource Planning software suite,
 * Copyright (c) 1999-2017 by OpenMFG LLC, d/b/a xTuple.
 * It is licensed to you under the Common Public Attribution License
 * version 1.0, the full text of which (including xTuple-specific Exhibits)
 * is available at www.xtuple.com/CPAL.  By using this software, you agree
 * to be bound by its terms.
 */

#include "user.h"

#include <QMessageBox>
#include <QVariant>
#include <QRegExp>

#include <qmd5.h>
#include <metasql.h>

#include "crmaccount.h"
#include "errorReporter.h"
#include "guiErrorCheck.h"
#include "storedProcErrorLookup.h"
#include "userPreferences.h"

user::user(QWidget* parent, const char * name, Qt::WindowFlags fl)
  : XDialog(parent, name, fl)
{
  setupUi(this);

  _authCache     = false;
  _windowCache   = "";
  _cUsername     = "";
  _crmacctid     = -1;
  _inTransaction = false;
  _mode          = cView;

  connect(_close, SIGNAL(clicked()), this, SLOT(sClose()));
  connect(_crmacct, SIGNAL(clicked()),   this,     SLOT(sCrmAccount()));
  connect(_save, SIGNAL(clicked()), this, SLOT(sSave()));
  connect(_add, SIGNAL(clicked()), this, SLOT(sAdd()));
  connect(_addAll, SIGNAL(clicked()), this, SLOT(sAddAll()));
  connect(_revoke, SIGNAL(clicked()), this, SLOT(sRevoke()));
  connect(_revokeAll, SIGNAL(clicked()), this, SLOT(sRevokeAll()));
  connect(_module, SIGNAL(activated(const QString&)), this, SLOT(sModuleSelected(const QString&)));
  connect(_granted, SIGNAL(itemSelected(int)), this, SLOT(sRevoke()));
  connect(_available, SIGNAL(itemSelected(int)), this, SLOT(sAdd()));
  connect(_username, SIGNAL(editingFinished()), this, SLOT(sCheck()));
  connect(_enhancedAuth, SIGNAL(toggled(bool)), this, SLOT(sEnhancedAuthUpdate()));
  connect(_grantedGroup, SIGNAL(itemSelected(int)), this, SLOT(sRevokeGroup()));
  connect(_availableGroup, SIGNAL(itemSelected(int)), this, SLOT(sAddGroup()));
  connect(_addGroup, SIGNAL(clicked()), this, SLOT(sAddGroup()));
  connect(_revokeGroup, SIGNAL(clicked()), this, SLOT(sRevokeGroup()));
  connect(_grantedSite, SIGNAL(itemSelected(int)), this, SLOT(sRevokeSite()));
  connect(_availableSite, SIGNAL(itemSelected(int)), this, SLOT(sAddSite()));
  connect(_addSite, SIGNAL(clicked()), this, SLOT(sAddSite()));
  connect(_revokeSite, SIGNAL(clicked()), this, SLOT(sRevokeSite()));

  _available->addColumn("Available Privileges", -1, Qt::AlignLeft);
  _available->addColumn("Description", -1, Qt::AlignLeft);
  _granted->addColumn("Granted Privileges", -1, Qt::AlignLeft);
  _granted->addColumn("Description", -1, Qt::AlignLeft);

  _availableGroup->addColumn("Available Roles", -1, Qt::AlignLeft);
  _availableGroup->addColumn("Description", -1, Qt::AlignLeft);
  _grantedGroup->addColumn("Granted Roles", -1, Qt::AlignLeft);
  _grantedGroup->addColumn("Description", -1, Qt::AlignLeft);

  _availableSite->addColumn("Available Sites", -1, Qt::AlignLeft);
  _grantedSite->addColumn("Granted Sites",      -1, Qt::AlignLeft);

  _locale->setType(XComboBox::Locales);
  XSqlQuery locq;
  locq.exec("SELECT locale_id FROM locale WHERE locale_code='Default'");
  if (locq.first())
    _locale->setId(locq.value("locale_id").toInt());

  XSqlQuery modq;
  modq.exec( "SELECT DISTINCT priv_module FROM priv ORDER BY priv_module;" );
  for (int i = 0; modq.next(); i++)
    _module->append(i, modq.value("priv_module").toString());

  if(_evaluation == true)
  {
    _enhancedAuth->setEnabled(false);
    _passwd->setEnabled(false);
    _verify->setEnabled(false);
  }

  if (!_metrics->boolean("MultiWhs"))
  {
    _tab->removeTab(_tab->indexOf(_siteTab));
    _useFullApp->setVisible(false);
  }
}

user::~user()
{
  // no need to delete child widgets, Qt does it all for us
  if(_inTransaction)
    XSqlQuery rollback("ROLLBACK;");
}

void user::languageChange()
{
  retranslateUi(this);
}

enum SetResponse user::set(const ParameterList &pParams)
{
  XDialog::set(pParams);
  QVariant param;
  bool     valid;

  param = pParams.value("crmacct_id", &valid);
  if (valid)
    _crmacctid = param.toInt();

  param = pParams.value("username", &valid);
  if (valid)
  {
    _cUsername = param.toString();
    _pref = new Preferences(_cUsername);

    _inventoryMenu->setChecked(_pref->boolean("ShowIMMenu"));
    _productsMenu->setChecked(_pref->boolean("ShowPDMenu"));
    _scheduleMenu->setChecked(_pref->boolean("ShowMSMenu"));
    _manufactureMenu->setChecked(_pref->boolean("ShowWOMenu"));
    _crmMenu2->setChecked(_pref->boolean("ShowCRMMenu"));
    _purchaseMenu->setChecked(_pref->boolean("ShowPOMenu"));
    _salesMenu->setChecked(_pref->boolean("ShowSOMenu"));
    _accountingMenu->setChecked(_pref->boolean("ShowGLMenu"));

    _inventoryToolbar->setChecked(_pref->boolean("ShowIMToolbar"));
    _productsToolbar->setChecked(_pref->boolean("ShowPDToolbar"));
    _scheduleToolbar->setChecked(_pref->boolean("ShowMSToolbar"));
    _manufactureToolbar->setChecked(_pref->boolean("ShowWOToolbar"));
    _crmToolbar2->setChecked(_pref->boolean("ShowCRMToolbar"));
    _purchaseToolbar->setChecked(_pref->boolean("ShowPOToolbar"));
    _salesToolbar->setChecked(_pref->boolean("ShowSOToolbar"));
    _accountingToolbar->setChecked(_pref->boolean("ShowGLToolbar"));
  }

  if (! _cUsername.isEmpty() || _crmacctid > 0)
    if (! sPopulate())
      return UndefinedError;

  param = pParams.value("mode", &valid);
  if (valid)
  {
    if (param.toString() == "new")
    {
      _mode = cNew;
      _module->setCurrentIndex(0);
      sModuleSelected(_module->itemText(0));

      if (! _cUsername.isEmpty())
      {
        _username->setEnabled(false);
        _username->setText(_cUsername);
        sCheck();
      }
      if (_metrics->boolean("MultiWhs"))
        populateSite();
    }
    else if (param.toString() == "edit")
    {
      _mode = cEdit;

      _username->setEnabled(false);
    }
    else if (param.toString() == "view")
    {
      _mode = cView;

      _close->setText(tr("&Close"));
      _save->hide();
    }
  }

  bool canEdit = (cNew == _mode || cEdit == _mode);

  _active->setEnabled(canEdit);
  _add->setEnabled(canEdit);
  _addAll->setEnabled(canEdit);
  _addGroup->setEnabled(canEdit);
//  _addSite->setEnabled(canEdit);
  _agent->setEnabled(canEdit);
  _allSites->setEnabled(canEdit);
  _email->setEnabled(canEdit);
  _employee->setReadOnly(! canEdit);
  _enhancedAuth->setEnabled(canEdit);
  _ssosOnly->setEnabled(canEdit);
  _exportContents->setEnabled(canEdit);
  _initials->setEnabled(canEdit);
  _locale->setEnabled(canEdit);
  _passwd->setEnabled(canEdit);
  _properName->setEnabled(canEdit);
  _revoke->setEnabled(canEdit);
  _revokeAll->setEnabled(canEdit);
  _revokeGroup->setEnabled(canEdit);
//  _revokeSite->setEnabled(canEdit);
  _save->setEnabled(canEdit);
  _selectedSites->setEnabled(canEdit);
  _verify->setEnabled(canEdit);
  if (! canEdit)
  {
    _available->setSelectionMode(QAbstractItemView::NoSelection);
    _availableGroup->setSelectionMode(QAbstractItemView::NoSelection);
    _availableSite->setSelectionMode(QAbstractItemView::NoSelection);
    _granted->setSelectionMode(QAbstractItemView::NoSelection);
    _grantedGroup->setSelectionMode(QAbstractItemView::NoSelection);
    _grantedSite->setSelectionMode(QAbstractItemView::NoSelection);
  }

  if(canEdit)
  {
    XSqlQuery begin;
    _inTransaction = begin.exec("BEGIN;");
  }

  return NoError;
}

void user::sClose()
{
  if (_mode == cNew)
  {
    XSqlQuery delq;
    delq.prepare( "DELETE FROM usrpriv WHERE (usrpriv_username=:username);" );
    delq.bindValue(":username", _cUsername);
    delq.exec();
    ErrorReporter::error(QtCriticalMsg, this, tr("Deleting Privileges"),
                         delq, __FILE__, __LINE__);
  }

  reject();
}

void user::sSave()
{
  if(save())
    accept();
}

bool user::save()
{
  QString username = _username->text().trimmed().toLower();

  QList<GuiErrorCheck> errors;
  errors << GuiErrorCheck(! username.contains(QRegExp("[A-Za-z]")), _username,
                          tr("You must enter a valid Username before you can save this User Account."))
         << GuiErrorCheck(_username->text().contains(QRegExp("\\s")), _username,
                          tr("The Username cannot include any spaces."))
         << GuiErrorCheck(_passwd->text().isEmpty(), _passwd,
                          tr("You must enter a valid Password before you can save this User Account."))
         << GuiErrorCheck(_passwd->text() != _verify->text(), _passwd,
                          tr("The entered password and verify do not match. "
                             "Please enter both again carefully."))
   ;

  if (GuiErrorCheck::reportErrors(this, tr("Cannot save User Account"), errors))
  {
     if (_passwd->text() != _verify->text())
     {
      _passwd->clear();
      _verify->clear();
     }
    return false;
  }

  QString passwd = _passwd->text();
  if(_enhancedAuth->isChecked())
  {
      QRegExp xtuplecloud(".*\\.xtuplecloud\\.com.*");
      QRegExp xtuple(".*\\.xtuple\\.com.*");

      bool isCloud = xtuplecloud.exactMatch(omfgThis->databaseURL());
      bool isXtuple = xtuple.exactMatch(omfgThis->databaseURL());
      QString salt;

      if(isCloud || isXtuple)
      {
        salt = "j3H44uadEI#8#kSmkh#H%JSLAKDOHImklhdfsn3#432?%^kjasdjla3uy989apa3uipoweurw-03235##+=-lhkhdNOHA?%@mxncvbwoiwerNKLJHwe278NH28shNeGc";
      }
      else
      {
        salt = "xTuple";
      }
      passwd = passwd + salt + username;
    passwd = QMd5(passwd);
  }

  XSqlQuery usrq;
  if (_mode == cNew)
  {
    usrq.prepare("SELECT usesysid"
                 "  FROM pg_user"
                 " WHERE (usename=:username);" );
    usrq.bindValue(":username", username);
    usrq.exec();
    if (!usrq.first())
    {
      usrq.prepare("SELECT createUser(:username, :createUsers);");
      usrq.bindValue(":username", username);
      usrq.bindValue(":createUsers", QVariant(_createUsers->isChecked()));
      usrq.exec();
      if (ErrorReporter::error(QtCriticalMsg, this, tr("Creating User"),
                               usrq, __FILE__, __LINE__))
        return false;
    }
  }
  else if (_mode == cEdit)
  {
    if(_createUsers->isEnabled())
    {
      usrq.prepare("SELECT setUserCanCreateUsers(:username, :createUsers);");
      usrq.bindValue(":username", username);
      usrq.bindValue(":createUsers", QVariant(_createUsers->isChecked()));
      usrq.exec();
      if (ErrorReporter::error(QtCriticalMsg, this, tr("Saving User Account"),
                               usrq, __FILE__, __LINE__))
        return false;
    }
  }

  if(_createUsers->isEnabled())
  {
    usrq.prepare("SELECT pg_has_role(:username,'xtrole','member') AS result;");
    usrq.bindValue(":username", username);
    usrq.exec();
    if(usrq.first() && !usrq.value("result").toBool())
    {
      usrq.exec( QString("ALTER GROUP xtrole ADD USER %1;")
              .arg(username) );
    }
    if (ErrorReporter::error(QtCriticalMsg, this, tr("Saving User Account"),
                             usrq, __FILE__, __LINE__))
      return false;
  }

  if (_passwd->text() != "        ")
  {
  usrq.prepare( QString( "ALTER USER \"%1\" WITH PASSWORD :password;")
               .arg(username) );
    usrq.bindValue(":password", passwd);
    usrq.exec();
    if (ErrorReporter::error(QtCriticalMsg, this, tr("Setting Password"),
                             usrq, __FILE__, __LINE__))
      return false;
    usrq.prepare("SELECT setUserPreference(:username, 'PasswordResetDate', :passdate);");
    usrq.bindValue(":username", username);
    usrq.bindValue(":passdate", QDate::currentDate());
    usrq.exec();
    if (ErrorReporter::error(QtCriticalMsg, this, tr("Saving User Account"),
                             usrq, __FILE__, __LINE__))
      return false;
  }

  usrq.prepare("SELECT setUserPreference(:username, 'DisableExportContents', :export),"
            "       setUserPreference(:username, 'UseEnhancedAuthentication', :enhanced),"
            "       setUserPreference(:username, 'selectedSites', :sites),"
            "       setUserPreference(:username, 'propername', :propername),"
            "       setUserPreference(:username, 'email', :email),"
            "       setUserPreference(:username, 'initials', :initials),"
            "       setUserPreference(:username, 'locale_id', text(:locale_id)),"
            "       setUserPreference(:username, 'agent', :agent),"
            "       setUserPreference(:username, 'active', :active),"
            "       setUserPreference(:username, 'ShowIMMenu', :ShowIMMenu),"
            "       setUserPreference(:username, 'ShowPDMenu', :ShowPDMenu),"
            "       setUserPreference(:username, 'ShowMSMenu', :ShowMSMenu),"
            "       setUserPreference(:username, 'ShowWOMenu', :ShowWOMenu),"
            "       setUserPreference(:username, 'ShowCRMMenu', :ShowCRMMenu),"
            "       setUserPreference(:username, 'ShowPOMenu', :ShowPOMenu),"
            "       setUserPreference(:username, 'ShowSOMenu', :ShowSOMenu),"
            "       setUserPreference(:username, 'ShowGLMenu', :ShowGLMenu),"
            "       setUserPreference(:username, 'ShowIMToolbar', :ShowIMToolbar),"
            "       setUserPreference(:username, 'ShowPDToolbar', :ShowPDToolbar),"
            "       setUserPreference(:username, 'ShowMSToolbar', :ShowMSToolbar),"
            "       setUserPreference(:username, 'ShowWOToolbar', :ShowWOToolbar),"
            "       setUserPreference(:username, 'ShowCRMToolbar', :ShowCRMToolbar),"
            "       setUserPreference(:username, 'ShowPOToolbar', :ShowPOToolbar),"
            "       setUserPreference(:username, 'ShowSOToolbar', :ShowSOToolbar),"
            "       setUserPreference(:username, 'ShowGLToolbar', :ShowGLToolbar)");
  usrq.bindValue(":username", username);
  usrq.bindValue(":export", (_exportContents->isChecked() ? "t" : "f"));
  usrq.bindValue(":enhanced", (_enhancedAuth->isChecked() ? "t" : "f"));
  usrq.bindValue(":sites", (_selectedSites->isChecked() ? "t" : "f"));
  usrq.bindValue(":propername", _properName->text());
  usrq.bindValue(":email", _email->text());
  usrq.bindValue(":initials", _initials->text());
  usrq.bindValue(":locale_id", _locale->id());
  usrq.bindValue(":agent", (_agent->isChecked() ? "t" : "f"));
  usrq.bindValue(":active", (_active->isChecked() ? "t" : "f"));

  usrq.bindValue(":ShowIMMenu",  (_inventoryMenu->isChecked() ? "t" : "f"));
  usrq.bindValue(":ShowPDMenu",  (_productsMenu->isChecked() ? "t" : "f"));
  usrq.bindValue(":ShowMSMenu",  (_scheduleMenu->isChecked() ? "t" : "f"));
  usrq.bindValue(":ShowWOMenu",  (_manufactureMenu->isChecked() ? "t" : "f"));
  usrq.bindValue(":ShowCRMMenu", (_crmMenu2->isChecked() ? "t" : "f"));
  usrq.bindValue(":ShowPOMenu",  (_purchaseMenu->isChecked() ? "t" : "f"));
  usrq.bindValue(":ShowSOMenu",  (_salesMenu->isChecked() ? "t" : "f"));
  usrq.bindValue(":ShowGLMenu",  (_accountingMenu->isChecked() ? "t" : "f"));
 
  usrq.bindValue(":ShowIMToolbar",  (_inventoryToolbar->isChecked() ? "t" : "f"));
  usrq.bindValue(":ShowPDToolbar",  (_productsToolbar->isChecked() ? "t" : "f"));
  usrq.bindValue(":ShowMSToolbar",  (_scheduleToolbar->isChecked() ? "t" : "f"));
  usrq.bindValue(":ShowWOToolbar",  (_manufactureToolbar->isChecked() ? "t" : "f"));
  usrq.bindValue(":ShowCRMToolbar", (_crmToolbar2->isChecked() ? "t" : "f"));
  usrq.bindValue(":ShowPOToolbar",  (_purchaseToolbar->isChecked() ? "t" : "f"));
  usrq.bindValue(":ShowSOToolbar",  (_salesToolbar->isChecked() ? "t" : "f"));
  usrq.bindValue(":ShowGLToolbar",  (_accountingToolbar->isChecked() ? "t" : "f"));

  usrq.exec();
  if (ErrorReporter::error(QtCriticalMsg, this, tr("Saving User Account"),
                           usrq, __FILE__, __LINE__))
    return false;

  if (_ssosOnly->isChecked() || _windowCache == "salesOrderSimple")
  {
    usrq.prepare("SELECT setUserPreference(:username, 'window', :ssosonly);");
    usrq.bindValue(":username", username);
    usrq.bindValue(":ssosonly", (_ssosOnly->isChecked() ? "salesOrderSimple" : ""));
    usrq.exec();
    if (ErrorReporter::error(QtCriticalMsg, this, tr("Saving User Account"),
                             usrq, __FILE__, __LINE__))
      return false;
  }

  omfgThis->sUserUpdated(username);
  return true;
}

void user::sModuleSelected(const QString &pModule)
{
  XTreeWidgetItem *granted = NULL;
  XTreeWidgetItem *available = NULL;

  _availableGroup->clear();
  _grantedGroup->clear();
  XSqlQuery groups;
  groups.prepare("SELECT grp_id, grp_name, grp_descrip, usrgrp_id"
                 "  FROM grp LEFT OUTER JOIN usrgrp"
                 "    ON (usrgrp_grp_id=grp_id AND usrgrp_username=:username);");
  groups.bindValue(":username", _cUsername);
  groups.exec();
  while(groups.next())
  {
    if (groups.value("usrgrp_id").toInt() == 0)
      available = new XTreeWidgetItem(_availableGroup, available, groups.value("grp_id").toInt(), groups.value("grp_name"), groups.value("grp_descrip"));
    else
      granted = new XTreeWidgetItem(_grantedGroup, granted, groups.value("grp_id").toInt(), groups.value("grp_name"), groups.value("grp_descrip"));
  }
  if (ErrorReporter::error(QtCriticalMsg, this, tr("Getting Groups"),
                           groups, __FILE__, __LINE__))
    return;

  _available->clear();
  _granted->clear();

  XSqlQuery privs;
  privs.prepare( "SELECT priv_id, priv_name, priv_descrip "
                 "FROM priv "
                 "WHERE (priv_module=:priv_module) "
                 "ORDER BY priv_name;" );
  privs.bindValue(":priv_module", pModule);
  privs.exec();
  if (privs.first())
  {
    granted = NULL;
    available = NULL;

//  Insert each priv into either the available or granted list
    XSqlQuery usrpriv;
    usrpriv.prepare( "SELECT priv_id "
                     "FROM priv, usrpriv "
                     "WHERE ( (usrpriv_priv_id=priv_id)"
                     " AND (usrpriv_username=:username)"
                     " AND (priv_module=:priv_module) );" );
    usrpriv.bindValue(":username", _cUsername);
    usrpriv.bindValue(":priv_module", _module->currentText());
    usrpriv.exec();

    XSqlQuery grppriv;
    grppriv.prepare("SELECT priv_id"
                    "  FROM priv, grppriv, usrgrp"
                    " WHERE((usrgrp_grp_id=grppriv_grp_id)"
                    "   AND (grppriv_priv_id=priv_id)"
                    "   AND (usrgrp_username=:username)"
                    "   AND (priv_module=:priv_module));");
    grppriv.bindValue(":username", _cUsername);
    grppriv.bindValue(":priv_module", _module->currentText());
    grppriv.exec();

    do
    {
      if (usrpriv.findFirst("priv_id", privs.value("priv_id").toInt()) == -1 && grppriv.findFirst("priv_id", privs.value("priv_id").toInt()) == -1)
        available = new XTreeWidgetItem(_available, available, privs.value("priv_id").toInt(), privs.value("priv_name"), privs.value("priv_descrip"));
      else
      {
        granted = new XTreeWidgetItem(_granted, granted, privs.value("priv_id").toInt(), privs.value("priv_name"), privs.value("priv_descrip"));
        if(usrpriv.findFirst("priv_id", privs.value("priv_id").toInt()) == -1)
          granted->setTextColor(Qt::gray);
      }
    }
    while (privs.next());
  }
}

void user::sAdd()
{
  if (_available->id() == -1)
  {
      QMessageBox::critical(this, tr("Error"), tr("Please select an Available Privilege."));
      return;
  }

  XSqlQuery privq;
  privq.prepare("SELECT grantPriv(:username, :priv_id) AS result;");
  privq.bindValue(":username", _cUsername);
  privq.bindValue(":priv_id", _available->id());
  privq.exec();
  if (ErrorReporter::error(QtCriticalMsg, this, tr("Granting Privilege"),
                           privq, __FILE__, __LINE__))
    return;

  sModuleSelected(_module->currentText());
}

void user::sAddAll()
{
  XSqlQuery privq;
  privq.prepare("SELECT grantAllModulePriv(:username, :module) AS result;");
  privq.bindValue(":username", _cUsername);
  privq.bindValue(":module", _module->currentText());
  privq.exec();
  if (privq.first())
  {
    int result = privq.value("result").toInt();
    if (result < 0)
    {
      ErrorReporter::error(QtCriticalMsg, this, tr("Granting Privileges"),
                           storedProcErrorLookup("grantAllModulePriv", result),
                           __FILE__, __LINE__);
      return;
    }
  }
  else if (ErrorReporter::error(QtCriticalMsg, this, tr("Granting Privileges"),
                           privq, __FILE__, __LINE__))
    return;

  sModuleSelected(_module->currentText());
}

void user::sRevoke()
{
  if (_granted->id() == -1)
  {
      QMessageBox::critical(this, tr("Error"), tr("Please select a Granted Privilege."));
      return;
  }

  XSqlQuery privq;
  privq.prepare("SELECT revokePriv(:username, :priv_id) AS result;");
  privq.bindValue(":username", _cUsername);
  privq.bindValue(":priv_id", _granted->id());
  privq.exec();
  // no storedProcErrorLookup because the function returns bool, not int
  if (ErrorReporter::error(QtCriticalMsg, this, tr("Revoking Privileges"),
                           privq, __FILE__, __LINE__))
    return;

  sModuleSelected(_module->currentText());
}

void user::sRevokeAll()
{
  XSqlQuery privq;
  privq.prepare("SELECT revokeAllModulePriv(:username, :module) AS result;");
  privq.bindValue(":username", _cUsername);
  privq.bindValue(":module", _module->currentText());
  privq.exec();
  if (privq.first())
  {
    int result = privq.value("result").toInt();
    if (result < 0)
    {
      ErrorReporter::error(QtCriticalMsg, this, tr("Revoking Privileges"),
                           storedProcErrorLookup("revokeAllModulePriv", result),
                           __FILE__, __LINE__);
      return;
    }
  }
  else if (ErrorReporter::error(QtCriticalMsg, this, tr("Revoking Privileges"),
                                privq, __FILE__, __LINE__))
    return;

  sModuleSelected(_module->currentText());
}

void user::sAddGroup()
{
  if (_availableGroup->id() == -1)
  {
      QMessageBox::critical(this, tr("Error"), tr("Please select an Available Role."));
      return;
  }

  XSqlQuery grpq;
  grpq.prepare("SELECT grantGroup(:username, :grp_id) AS result;");
  grpq.bindValue(":username", _cUsername);
  grpq.bindValue(":grp_id", _availableGroup->id());
  grpq.exec();
  // no storedProcErrorLookup because the function returns bool, not int
  if (ErrorReporter::error(QtCriticalMsg, this, tr("Granting Group Privileges"),
                                grpq, __FILE__, __LINE__))
    return;

  // Get default role menus and add to user menu preferences
  grpq.prepare("SELECT grp_showimmenu, grp_showpdmenu, grp_showmsmenu, "
               "grp_showwomenu, grp_showcrmmenu, grp_showpomenu, "
               "grp_showsomenu, grp_showglmenu, "
               "grp_showimtoolbar, grp_showpdtoolbar, grp_showmstoolbar, "
               "grp_showwotoolbar, grp_showcrmtoolbar, grp_showpotoolbar, "
               "grp_showsotoolbar, grp_showgltoolbar "
               "FROM grp WHERE grp_id=:grp_id;");
  grpq.bindValue(":grp_id", _availableGroup->id());
  grpq.exec();
  if (grpq.first())
  {
    // Only switch menus on - don't switch off
    if (grpq.value("grp_showimmenu").toBool())
      _inventoryMenu->setChecked(true);
    if (grpq.value("grp_showpdmenu").toBool())
      _productsMenu->setChecked(true);
    if (grpq.value("grp_showmsmenu").toBool())
      _scheduleMenu->setChecked(true);
    if (grpq.value("grp_showwomenu").toBool())
      _manufactureMenu->setChecked(true);
    if (grpq.value("grp_showcrmmenu").toBool())
      _crmMenu2->setChecked(true);
    if (grpq.value("grp_showpomenu").toBool())
      _purchaseMenu->setChecked(true);
    if (grpq.value("grp_showsomenu").toBool())
      _salesMenu->setChecked(true);
    if (grpq.value("grp_showglmenu").toBool())
      _accountingMenu->setChecked(true);

    if (grpq.value("grp_showimtoolbar").toBool()) 
      _inventoryToolbar->setChecked(true);
    if (grpq.value("grp_showpdtoolbar").toBool())
      _productsToolbar->setChecked(true);
    if (grpq.value("grp_showmstoolbar").toBool())
      _scheduleToolbar->setChecked(true);
    if (grpq.value("grp_showwotoolbar").toBool())
      _manufactureToolbar->setChecked(true);
    if (grpq.value("grp_showcrmtoolbar").toBool())
      _crmToolbar2->setChecked(true);
    if (grpq.value("grp_showpotoolbar").toBool())
      _purchaseToolbar->setChecked(true);
    if (grpq.value("grp_showsotoolbar").toBool())
      _salesToolbar->setChecked(true);
    if (grpq.value("grp_showgltoolbar").toBool())
      _accountingToolbar->setChecked(true);
  }
  if (ErrorReporter::error(QtCriticalMsg, this, tr("Updating Group Default Menus"),
                                grpq, __FILE__, __LINE__))
    return;
    
  sModuleSelected(_module->currentText());
}

void user::sRevokeGroup()
{
  if (_grantedGroup->id() == -1)
  {
      QMessageBox::critical(this, tr("Error"), tr("Please select a Granted Role."));
      return;
  }

  XSqlQuery grpq;
  grpq.prepare("SELECT revokeGroup(:username, :grp_id) AS result;");
  grpq.bindValue(":username", _cUsername);
  grpq.bindValue(":grp_id", _grantedGroup->id());
  grpq.exec();
  // no storedProcErrorLookup because the function returns bool, not int
  if (ErrorReporter::error(QtCriticalMsg, this, tr("Revoking Group Privileges"),
                                grpq, __FILE__, __LINE__))
    return;

  sModuleSelected(_module->currentText());
}

void user::sCheck()
{
  //This regexp checks to make sure the user name starts with a letter.
  QRegExp re("^\\d"); // just digits
  QRegExp re2("^\\w"); // this includes all letters & numbers of all alphabets
  _cUsername = _username->text().trimmed().toLower();
  if (((re.indexIn(_cUsername) != -1) || (re2.indexIn(_cUsername) == -1)) && _username->text() != "")
  {
      QMessageBox::critical(this, tr("Error"), tr("User names must begin with a letter."));
      _username->clear();
      _username->setFocus();
      return;
  }

  if (_cUsername.length() > 0)
  {
    XSqlQuery usrq;
    usrq.prepare("SELECT * FROM usr WHERE (usr_username=:username);");
    usrq.bindValue(":username", _cUsername);
    usrq.exec();
    if (usrq.first())
    {
      sPopulate();
      _mode = cEdit;
      _username->setEnabled(false);
      _properName->setFocus();
    }
    else if (ErrorReporter::error(QtCriticalMsg, this, tr("Getting User Account"),
                                  usrq, __FILE__, __LINE__))
      return;
    
    XSqlQuery dupq;
    dupq.prepare("SELECT crmacct_id"
                 "  FROM crmacct "
                 " WHERE (UPPER(crmacct_number)=UPPER(:username))"
                 "   AND crmacct_usr_username IS NULL;");
    dupq.bindValue(":username", _cUsername);
    dupq.exec();
    if (dupq.first())
    {
      if (QMessageBox::question(this, tr("Convert"),
                                tr("<p>This number is currently assigned to CRM Account. "
                                   "Do you want to convert the CRM Account to a User Account?"),
                                QMessageBox::Yes,
                                QMessageBox::No | QMessageBox::Default) == QMessageBox::No)
      {
        _username->clear();
        _username->setFocus();
        return;
      }
    }
    else if (ErrorReporter::error(QtCriticalMsg, this, tr("Getting User CRM"),
                                  dupq, __FILE__, __LINE__))
      return;
  }
}

bool user::sPopulate()
{
  XSqlQuery usrq;
  if (! _cUsername.isEmpty())
  {
    usrq.prepare("SELECT usr_username, usr_active, usr_propername, usr_initials,"
                 "       usr_email, usr_locale_id, usr_agent,"
                 "       userCanCreateUsers(usr_username) AS createusers,"
                 "       userCanCreateUsers(getEffectiveXtUser()) AS enablecreateusers,"
                 "       crmacct_id, crmacct_emp_id, crmacct_owner_username"
                 "  FROM usr"
                 "  LEFT OUTER JOIN crmacct ON (usr_username=crmacct_usr_username) "
                 "WHERE (usr_username=:usr_username);" );
    usrq.bindValue(":usr_username", _cUsername);
  }
  else if (_crmacctid > 0)
  {
    usrq.prepare("SELECT LOWER(crmacct_number) AS usr_username,"
                 "       crmacct_name          AS usr_propername,"
                 "       (SELECT locale_id"
                 "          FROM locale"
                 "         WHERE locale_code='Default') AS usr_locale_id,"
                 "       NULL  AS usr_passwd,  cntct_initials AS usr_initials,"
                 "       false AS usr_agent,   crmacct_active AS usr_active,"
                 "       NULL  AS usr_window,  cntct_email AS usr_email,"
                 "       false AS createusers,"
                 "       userCanCreateUsers(getEffectiveXtUser()) AS enablecreateusers,"
                 "       crmacct_id, crmacct_emp_id, crmacct_owner_username"
                 "  FROM crmacct"
                 "  LEFT OUTER JOIN cntct ON (crmacct_cntct_id_1=cntct_id)"
                 " WHERE (crmacct_id=:id);");
    usrq.bindValue(":id", _crmacctid);
  }

  usrq.exec();
  if (usrq.first())
  {
    _username->setText(usrq.value("usr_username"));

    if (_crmacctid > 0)
    {
      if (_username->text().contains(QRegExp("\\s")))
      {
        QMessageBox::warning(this, tr("No Spaces Allowed"),
                             tr("<p>Usernames cannot include space characters "
                                "but must also match the associated Account "
                                "numbers. Please Cancel the User Account window and "
                                "remove the spaces from the Account number "
                                "before trying to create this User Account."));
        return false;
      }
      _username->setEnabled(false);
    }
    _active->setChecked(usrq.value("usr_active").toBool());
    _properName->setText(usrq.value("usr_propername"));
    _initials->setText(usrq.value("usr_initials"));
    _email->setText(usrq.value("usr_email"));
    _locale->setId(usrq.value("usr_locale_id").toInt());
    _agent->setChecked(usrq.value("usr_agent").toBool());
    _createUsers->setChecked(usrq.value("createusers").toBool());
    _createUsers->setEnabled(usrq.value("enablecreateusers").toBool());
    _employee->setId(usrq.value("crmacct_emp_id").toInt());
    _crmacctid = usrq.value("crmacct_id").toInt();
    _crmowner = usrq.value("crmacct_owner_username").toString();
    _cUsername = _username->text().trimmed().toLower();

    _passwd->setText("        ");
    _verify->setText("        ");

    usrq.prepare( "SELECT usrpref_value "
               "  FROM usrpref "
               " WHERE ( (usrpref_name = 'DisableExportContents') "
               "   AND (usrpref_username=:username) ); ");
    usrq.bindValue(":username", _cUsername);
    usrq.exec();
    if(usrq.first())
      _exportContents->setChecked(usrq.value("usrpref_value").toString()=="t");
    else
      _exportContents->setChecked(false);

    usrq.prepare( "SELECT usrpref_value "
               "  FROM usrpref "
               " WHERE ( (usrpref_name = 'UseEnhancedAuthentication') "
               "   AND (usrpref_username=:username) ); ");
    usrq.bindValue(":username", _cUsername);
    usrq.exec();
    _authCache = false;
    if(usrq.first())
      _authCache = (usrq.value("usrpref_value").toString()=="t");
    _enhancedAuth->setChecked(_authCache);

    if (_metrics->boolean("SSOSEnabled"))
    {
      usrq.prepare( "SELECT usrpref_value "
                   "  FROM usrpref "
                   " WHERE ( (usrpref_name = 'window') "
                   "   AND (usrpref_username=:username) ); ");
      usrq.bindValue(":username", _cUsername);
      usrq.exec();
      _windowCache = "";
      if(usrq.first())
        _windowCache = (usrq.value("usrpref_value").toString());
      _ssosOnly->setChecked(_windowCache=="salesOrderSimple");
    }
    else
    {
      _ssosOnly->setChecked(false);
      _ssosOnly->hide();
    }
    
    _module->setCurrentIndex(0);
    sModuleSelected(_module->itemText(0));
  }
  else if (ErrorReporter::error(QtCriticalMsg, this, tr("Getting User Account"),
                                usrq, __FILE__, __LINE__))
    return false;

  usrq.prepare( "SELECT usrpref_value "
             "  FROM usrpref "
             " WHERE ( (usrpref_name = 'selectedSites') "
             "   AND (usrpref_username=:username) "
             "   AND (usrpref_value='t') ); ");
  usrq.bindValue(":username", _cUsername);
  usrq.exec();
  if(usrq.first())
    _selectedSites->setChecked(true);
  else if (ErrorReporter::error(QtCriticalMsg, this, tr("Getting User Sites"),
                                usrq, __FILE__, __LINE__))
    return false;

  if (_metrics->boolean("MultiWhs"))
    populateSite();

  _crmacct->setEnabled(_crmacctid > 0 &&
                       (_privileges->check("MaintainAllCRMAccounts") ||
                        _privileges->check("ViewAllCRMAccounts") ||
                        (omfgThis->username() == _crmowner && _privileges->check("MaintainPersonalCRMAccounts")) ||
                        (omfgThis->username() == _crmowner && _privileges->check("ViewPersonalCRMAccounts"))));


  return true;
}

void user::sEnhancedAuthUpdate()
{
  if((_mode == cEdit) && (_authCache != _enhancedAuth->isChecked()) && (_passwd->text() == "        "))
    QMessageBox::information( this, tr("Enhanced Authentication"),
      tr("<p>You have changed this User Account's Enhanced Authentication option. "
         "The password must be updated in order for this change to take "
         "full effect.") );
}

void user::sAddSite()
{
  if(_mode == cNew)
  {
    if(!save()) return;
    else
    {
      _mode = cEdit;
      _username->setEnabled(false);
    }
  }

  XSqlQuery siteq;
  siteq.prepare("SELECT grantSite(:username, :warehous_id) AS result;");
  siteq.bindValue(":username", _cUsername);
  siteq.bindValue(":warehous_id", _availableSite->id());
  siteq.exec();
  // no storedProcErrorLookup because the function returns bool, not int
  if (ErrorReporter::error(QtCriticalMsg, this, tr("Granting Site Privilege"),
                           siteq, __FILE__, __LINE__))
    return;

  populateSite();
}

void user::sRevokeSite()
{
  XSqlQuery siteq;
  siteq.prepare("SELECT revokeSite(:username, :warehous_id) AS result;");
  siteq.bindValue(":username", _cUsername);
  siteq.bindValue(":warehous_id", _grantedSite->id());
  siteq.exec();
  // no storedProcErrorLookup because the function returns bool, not int
  if (ErrorReporter::error(QtCriticalMsg, this, tr("Revoking Site Privilege"),
                           siteq, __FILE__, __LINE__))
    return;

  populateSite();
}

void user::populateSite()
{
  XSqlQuery siteq;
  if (_mode == cNew)
  {
    siteq.prepare("SELECT warehous_id, warehous_code"
                  "  FROM whsinfo"
                  " ORDER BY warehous_code;");
    siteq.exec();
    _availableSite->populate(siteq);
    if (ErrorReporter::error(QtCriticalMsg, this, tr("Getting Sites"),
                             siteq, __FILE__, __LINE__))
      return;
  }
  else
  {
    QString sql;
    MetaSQLQuery mql;
    ParameterList params;

    params.append("username", _username->text().trimmed().toLower());

    sql = "SELECT warehous_id, warehous_code "
          " FROM whsinfo "
          " WHERE warehous_id NOT IN ( "
          "     SELECT warehous_id "
          "     FROM whsinfo, usrsite "
          "     WHERE ( (usrsite_warehous_id=warehous_id) "
          "     AND (usrsite_username=<? value('username') ?>))) ";

    mql.setQuery(sql);
    siteq = mql.toQuery(params);
    _availableSite->populate(siteq);
    if (ErrorReporter::error(QtCriticalMsg, this, tr("Getting Ungranted Sites"),
                             siteq, __FILE__, __LINE__))
      return;

    sql = "SELECT warehous_id,warehous_code,0 AS warehous_level "
          "FROM whsinfo, usrsite "
          "WHERE ( (usrsite_warehous_id=warehous_id) "
          " AND (usrsite_username=<? value('username') ?>)) ";

    mql.setQuery(sql);
    siteq = mql.toQuery(params);
    _grantedSite->populate(siteq);
    if (ErrorReporter::error(QtCriticalMsg, this, tr("Getting Granted Sites"),
                             siteq, __FILE__, __LINE__))
      return;
  }
}

void user::done(int result)
{
  if(_inTransaction)
  {
    if(result == QDialog::Accepted)
      XSqlQuery commit("COMMIT;");
    else
      XSqlQuery rollback("ROLLBACK;");
    _inTransaction = false;
  }
  XDialog::done(result);
}

void user::sCrmAccount()
{
  ParameterList params;
  params.append("crmacct_id", _crmacctid);
  if ((cView == _mode && _privileges->check("ViewAllCRMAccounts")) ||
      (cView == _mode && _privileges->check("ViewPersonalCRMAccounts")
                      && omfgThis->username() == _crmowner) ||
      (cEdit == _mode && _privileges->check("ViewAllCRMAccounts")
                      && ! _privileges->check("MaintainAllCRMAccounts")) ||
      (cEdit == _mode && _privileges->check("ViewPersonalCRMAccounts")
                      && ! _privileges->check("MaintainPersonalCRMAccounts")
                      && omfgThis->username() == _crmowner))
    params.append("mode", "view");
  else if ((cEdit == _mode && _privileges->check("MaintainAllCRMAccounts")) ||
           (cEdit == _mode && _privileges->check("MaintainPersonalCRMAccounts")
                           && omfgThis->username() == _crmowner))
    params.append("mode", "edit");
  else if ((cNew == _mode && _privileges->check("MaintainAllCRMAccounts")) ||
           (cNew == _mode && _privileges->check("MaintainPersonalCRMAccounts")
                          && omfgThis->username() == _crmowner))
    params.append("mode", "edit");
  else
  {
    qWarning("tried to open Account window without privilege");
    return;
  }

  crmaccount *newdlg = new crmaccount();
  newdlg->set(params);
  omfgThis->handleNewWindow(newdlg, Qt::WindowModal);
}
