/*	Memory_Panel

PIRL CVS ID: Memory_Panel.java,v 1.5 2012/04/16 06:22:59 castalia Exp

Copyright (C) 2006-2012  Arizona Board of Regents on behalf of the
Planetary Image Research Laboratory, Lunar and Planetary Laboratory at
the University of Arizona.

This file is part of the PIRL Java Packages.

The PIRL Java Packages are free software; you can redistribute them
and/or modify them under the terms of the GNU Lesser General Public
License as published by the Free Software Foundation, either version 3 of
the License, or (at your option) any later version.

The PIRL Java Packages are distributed in the hope that they will be
useful, but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU Lesser
General Public License for more details.

You should have received a copy of the GNU Lesser General Public License
along with this program. If not, see <http://www.gnu.org/licenses/>.

*******************************************************************************/

package	PIRL.Viewers;

import java.awt.GridBagConstraints;
import java.awt.GridBagLayout;
import java.awt.Insets;
import java.text.DecimalFormat;
import javax.swing.JLabel;
import javax.swing.JPanel;

/**	A {@link JPanel} that holds information about available Virtual Machine
	memory. Three values are displayed: free memory, total memory, and maximum
	memory.
<p>
	<i>Free memory</i> is the amount of memory not yet in use by the Virtual
	Machine. This memory <i>has</i> been claimed from the available pool,
	however.
</p>
<p>
	<i>Total memory</i> is the total amount of memory the Virtual Machine has
	claimed from the available pool of memory. Free memory is a subset of this
	quantity; when subtracted from this quantity, the difference is the amount
	of memory currently in use. This value will change over time as objects are
	created and destroyed.
</p>
<p>
	<i>Maximum memory</i> is the maximum available pool of memory for the
	Virtual Machine. Its value is static once the Virtual Machine is
	instantiated, although runtime switches may be employed to allocate amounts
	other than the default; see the documentation for the Java Virtual Machine.
	Total memory is a subset of this quantity.
</p>

	@author		Christian Schaller - UA/PIRL
	@version	1.5
*/
public class Memory_Panel
extends JPanel
{

/*==============================================================================
Constants
*/

/**	Class identification name with source code version and date.
*/
public static final String
	ID = "PIRL.Viewers.Memory_Panel (1.5 2012/04/16 06:22:59)";

/**	The default unit  divisor. Memory quantities will, by default, be divided by
	this value for display in the panel. This value is the number of bytes in a
	megabyte (1024 * 1024).
*/
public static final int
	DEFAULT_UNIT_DIVISOR				= 1024 * 1024;

/**	The default unit. Memory quantities will be displayed with this text in the
	panel.
*/
public static final String
	DEFAULT_UNITS						= "MB";

/*==============================================================================
Fields
*/

/**	The unit divisor for all memory quantities.
*/
private int
	unit_divisor;

/**	The units for all memory quantities.
*/
private String
	units;

/*==============================================================================
Constructors
*/

/*..............................................................................
Memory_Panel (int, String)
*/
/**	Creates a new <tt>Memory_Panel</tt> with the specified <tt>unit_divisor</tt>
	and <tt>units</tt> label. The divisor is used to scale the memory values,
	which are in bytes; the units label should correctly identify the scale.
<p>
	This panel uses a {@link GridBagLayout}.
</p>

	@param	unit_divisor	the divisor to be used in scaling the displayed
		memory values.
	@param	units	the label that identifies the units used.
*/
public Memory_Panel
(
	int			unit_divisor,
	String		units
)
{
super (new GridBagLayout ());
this.unit_divisor = unit_divisor;
this.units = units;
build_ui ();
}

/*..............................................................................
Memory_Panel ()
*/
/**	Creates a new <tt>Memory_Panel</tt>. The panel uses the {@link
	#DEFAULT_UNIT_DIVISOR default divisor} and {@link #DEFAULT_UNITS default
	units} for its information display.

	@see	#Memory_Panel(int,String)
*/
public Memory_Panel ()
{ this (DEFAULT_UNIT_DIVISOR, DEFAULT_UNITS); }

/*==============================================================================
User interface
*/

private JLabel
	free_label,
	total_label,
	max_label;

private DecimalFormat
	formatter;

/*..............................................................................
build_ui ()
*/
/**	Builds the user interface. The panel is initialized and its layout set (to
	a {@link GridBagLayout}). The labels are instantiated and placed into the
	layout.
*/
private void build_ui ()
{
this.formatter = new DecimalFormat ("#,##0.00");
this.free_label = new JLabel ("", JLabel.TRAILING);
this.total_label = new JLabel ("", JLabel.TRAILING);
this.max_label = new JLabel ("", JLabel.TRAILING);
update_labels ();
GridBagConstraints
	constraints = new GridBagConstraints ();
constraints.gridx = 0;
constraints.gridy = 0;
constraints.fill = GridBagConstraints.HORIZONTAL;
constraints.insets = new Insets (0,0,0,5);
this.add (new JLabel ("Free Memory:"), constraints);
constraints.gridy++;
this.add (new JLabel ("Total Memory:"), constraints);
constraints.gridy++;
this.add (new JLabel ("Maximum Memory:"), constraints);
constraints.gridx++;
constraints.gridy = 0;
constraints.insets = new Insets (0,5,0,0);
this.add (free_label, constraints);
constraints.gridy++;
this.add (total_label, constraints);
constraints.gridy++;
this.add (max_label, constraints);
}

/*..............................................................................
update_labels ()
*/
/**	Updates the memory labels. This update takes place automatically whenever
	the units or unit divisors are changed, but one might want to update the
	labels periodically, polling the system from time to time.
*/
public void update_labels ()
{
free_label.setText (
	formatter.format (free_memory () / unit_divisor) + " " + units
);
total_label.setText (
	formatter.format (total_memory () / unit_divisor) + " " + units
);
max_label.setText (
	formatter.format (max_memory () / unit_divisor) + " " + units
);
}

/*==============================================================================
Accessors
*/

/*..............................................................................
unit_divisor ()
*/
/**	Gets the common divisor used for each of the three memory quantities. Each
	quantity is divided by this value for display.

	@return	the unit divisor for this <tt>Memory_Panel</tt>.
*/
public int unit_divisor () { return this.unit_divisor; }

/*..............................................................................
unit_divisor (int)
*/
/**	Sets the common divisor used for each of the three memory quantities. Each
	quantity is divided by this value or display. The displayed labels are
	updated when this value is changed.

	@param	unit_divisor	the new divisor for this <tt>Memory_Panel</tt>.
*/
public void unit_divisor
(
	int			unit_divisor
)
{ this.unit_divisor = unit_divisor; update_labels (); }

/*..............................................................................
units ()
*/
/**	Gets the units displayed for each of the three memory quantities.

	@return	the units for this <tt>Memory_Panel</tt>.
*/
public String units () { return this.units; }

/*..............................................................................
units (String)
*/
/**	Sets the units displayued for each of the three memory quantities. The
	displayed labels are updated when this value is changed.

	@param	units	the new units for this <tt>Memory_Panel</tt>.
*/
public void units
(
	String		units
)
{ this.units = units; update_labels (); }

/*==============================================================================
Static convenience methods
*/

/*..............................................................................
free_memory ()
*/
/**	Returns the amount of free memory currently available to the Java Virtual
	Machine.

	@return	the free memory in bytes.
	@see	Runtime#freeMemory
*/
public static long free_memory ()
{
return Runtime.getRuntime ().freeMemory ();
}

/*..............................................................................
total_memory ()
*/
/**	Returns the total amount of memory in the Java Virtual Machine. This value
	may vary over time as the Virtual Machine claims more memory from the
	maximum available and releases it.

	@return	the total memory in bytes.
	@see	Runtime#totalMemory
*/
public static long total_memory ()
{
return Runtime.getRuntime ().totalMemory ();
}

/*..............................................................................
max_memory ()
*/
/**	Returns the maximum amount of memory the Java Virtual Machine will attempt
	to use from the host system.

	@return	the maximum memory in bytes.
	@see	Runtime#maxMemory
*/
public static long max_memory ()
{
return Runtime.getRuntime ().maxMemory ();
}

}	//	End of Memory_Panel class.
