/*	Suspendable_Styled_Writer

PIRL CVS ID: Suspendable_Styled_Writer.java,v 1.7 2012/04/16 06:18:24 castalia Exp

Copyright (C) 2008-2012  Arizona Board of Regents on behalf of the
Planetary Image Research Laboratory, Lunar and Planetary Laboratory at
the University of Arizona.

This file is part of the PIRL Java Packages.

The PIRL Java Packages are free software; you can redistribute them
and/or modify them under the terms of the GNU Lesser General Public
License as published by the Free Software Foundation, either version 3 of
the License, or (at your option) any later version.

The PIRL Java Packages are distributed in the hope that they will be
useful, but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU Lesser
General Public License for more details.

You should have received a copy of the GNU Lesser General Public License
along with this program. If not, see <http://www.gnu.org/licenses/>.

*******************************************************************************/
package	PIRL.Utilities;

import	java.io.Writer;
import	java.io.IOException;
import	javax.swing.text.AttributeSet;


/**	A <i>Suspendable_Styled_Writer</i> is a Writer that can have its
	output operations suspended.
<p>
	All output methods are forwarded to the backing Writer. However, output
	may be suspended, in which case no output will occur until the
	suspension has been lifted.
<p>
	The <i>Styled_Writer</i> interface is also implemented such that if
	the backing Writer implements Styled_Writer that interface is used,
	otherwise the usual Writer API is used instead in which case no style
	information will be forwarded.
<p>
	@author		Bradford Castalia - UA/PIRL
	@version	1.7
	@see		Writer
	@see		Styled_Writer
*/
public class Suspendable_Styled_Writer
	extends Writer
	implements Styled_Writer
{
/**	Class identification name with source code version and date.
*/
public static final String
	ID = "PIRL.Utilities.Suspendable_Styled_Writer (1.7 2012/04/16 06:18:24)";

private Writer
	The_Writer;

/**	The suspension state of the Writer.
<p>
	When Active is true all output methods will forward to the {@link
	#Writer() Writer} on which the object was constructed.
*/
public volatile boolean
	Active		= true;

private static final String
	NL			= System.getProperty ("line.separator");


//  DEBUG control.
private static final int
	DEBUG_OFF			= 0,
	DEBUG_CONSTRUCTORS	= 1 << 0,
	DEBUG_WRITE			= 1 << 4,
	DEBUG_ALL			= -1,

	DEBUG		    	= DEBUG_OFF;

/*==============================================================================
	Constructors
*/
/**	Construct a Suspendable_Styled_Writer.
<p>
	The intial state of the Writer is determined by the {@link #Active
	flag}.
<p>
	@param	writer	The Writer to use for all write operations.
	@throws	IllegalArgumentException	If the writer is null.
*/
public Suspendable_Styled_Writer
	(
	Writer	writer
	)
{this (writer, false);}

/**	Construct a Suspendable_Styled_Writer with suspension specified.
<p>
	@param	writer	The Writer to use for all write operations.
	@param	active	The initial state of the Writer. If true writes
		will occure; if false writes are suspended.
	@throws	IllegalArgumentException	If the writer is null.
*/
public Suspendable_Styled_Writer
	(
	Writer	writer,
	boolean	active
	)
{
if ((DEBUG & DEBUG_CONSTRUCTORS) != 0)
	System.out.println
		(">-< Suspendable_Styled_Writer: "
			+ (active ? "" : "in") + "active " + writer);
Active = active;

if (writer == null)
	throw new IllegalArgumentException (ID + NL
		+ "Null Writer specified.");
The_Writer = writer;
}

private Suspendable_Styled_Writer ()
{}

/*==============================================================================
	Accessors
*/
/**	Get the Writer to which all Write methods are forwarded.
<p>
	@return	A Writer.
*/
public Writer Writer ()
{return The_Writer;}

/**	Test if output has been suspended.
<p>
	@return	true if output has been suspended; false otherwise.
	@see	#Suspend(boolean)
*/
public boolean Suspended ()
{return ! Active;}

/**	Turn output suspension on or off.
<p>
	@param	suspend	If true subsequent output will be suspended; if
		false output will occur.
*/
public void Suspend
	(
	boolean	suspend
	)
{Active = ! suspend;}

/*==============================================================================
	Writer
*/
/**	Write a portion of an array of characters.
<p>
	Nothing is done if the Writer is {@link #Suspended() suspended}.
<p>
	@param	characters	The char array containing the characters to be
		written. If null or empty nothing is done.
	@param	offset	Array offset from which to start writing characters.
	@param	amount	The number of characters to write.
	@throws	IOException	If the {@link #Writer() Writer} throws an exception.
*/
public void write
	(
	char[]	characters,
	int		offset,
	int		amount
	)
	throws IOException
{
if ((DEBUG & DEBUG_WRITE) != 0)
	System.out.println
		(">>> Suspendable_Styled_Writer.write: "
			+ Active + ' ' + offset + '/' + amount + " -" + NL
		+ characters);
if (Active) The_Writer.write (characters, offset, amount);
}

/**	Write an array of characters.
<p>
	Nothing is done if the Writer is {@link #Suspended() suspended}.
<p>
	@param	characters	The char array containing the characters to be
		written.
	@throws	IOException	If the {@link #Writer() Writer} throws an exception.
*/
public void write
	(
	char[]	characters
	)
	throws IOException
{if (Active) The_Writer.write (characters);}

/**	Write a character.
<p>
	Nothing is done if the Writer is {@link #Suspended() suspended}.
<p>
	@param	character	The character to be written in the 16 low-order
		bits..
	@throws	IOException	If the {@link #Writer() Writer} throws an exception.
*/
public void write
	(
	int		character
	)
	throws IOException
{if (Active) The_Writer.write (character);}

/**	Write a portion of a String.
<p>
	Nothing is done if the Writer is {@link #Suspended() suspended}.
<p>
	@param	string	The String containing the characters to be written.
	@param	offset	String offset from which to start writing characters.
	@param	amount	The number of characters to write.
	@throws	IOException	If the {@link #Writer() Writer} throws an exception.
*/
public void write
	(
	String			string,
	int				offset,
	int				amount
	)
	throws IOException
{if (Active) The_Writer.write (string, offset, amount);}

/**	Write a String.
<p>
	Nothing is done if the Writer is {@link #Suspended() suspended}.
<p>
	@param	string	The String containing the characters to be written.
		If null or empty nothing is done.
	@throws	IOException	If the {@link #Writer() Writer} throws an exception.
*/
public void write
	(
	String			string
	)
	throws IOException
{if (Active) The_Writer.write (string);}

/**	Flush the writer.
<p>
	Nothing is done if the Writer is {@link #Suspended() suspended}.
<p>
	@throws	IOException	If the {@link #Writer() Writer} throws an exception.
*/
public void flush ()
	throws IOException
{if (Active) The_Writer.flush ();}

/**	Close the writer.
<p>
	<b>N.B.</b>: The {@link #Writer() Writer} will be closed regardless
	of whether writing has been {@link #Suspended() suspended} or not.
<p>
	@throws	IOException	If the {@link #Writer() Writer} throws an exception.
*/
public void close ()
	throws IOException
{The_Writer.close ();}

/*==============================================================================
	Styled_Writer
*/
/**	Write styled text.
<p>
	@param	text	The text String to be written.
	@param	style	The AttributeSet to be applied to the text. This
		may be null if plain text is to be displayed.
	@return	This Suspendable_Styled_Writer
	@throws	IOException	If the {@link #Writer() Writer} throws an exception.
	@see	Styled_Writer#Write(String, AttributeSet)
*/
public Styled_Writer Write
	(
	String			text,
	AttributeSet	style
	)
	throws IOException
{
if ((DEBUG & DEBUG_WRITE) != 0)
	System.out.println
		(">>> Suspendable_Styled_Writer.Write: " + NL
		+"    Style: " + style + NL
		+"    Styled_Writer: " + (The_Writer instanceof Styled_Writer) + NL
		+ text);
if (The_Writer instanceof Styled_Writer)
		((Styled_Writer)The_Writer).Write (text, style);
else
	The_Writer.write (text);
return this;
}

/**	Write plain text.
<p>
	This is the same as {@link #Write(String, AttributeSet) writing}
	text with a null style.
<p>
	@param	text	The text String to be written.
	@return	This Suspendable_Styled_Writer
	@throws	IOException	If the {@link #Writer() Writer} throws an exception.
*/
public Styled_Writer Write
	(
	String			text
	)
	throws IOException
{
if (The_Writer instanceof Styled_Writer)
		((Styled_Writer)The_Writer).Write (text);
else
	The_Writer.write (text);
return this;
}

}
