/*	Parameter

PIRL CVS ID: Parameter.java,v 1.44 2012/04/16 06:14:23 castalia Exp

Copyright (C) 2001-2012  Arizona Board of Regents on behalf of the
Planetary Image Research Laboratory, Lunar and Planetary Laboratory at
the University of Arizona.

This file is part of the PIRL Java Packages.

The PIRL Java Packages are free software; you can redistribute them
and/or modify them under the terms of the GNU Lesser General Public
License as published by the Free Software Foundation, either version 3 of
the License, or (at your option) any later version.

The PIRL Java Packages are distributed in the hope that they will be
useful, but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU Lesser
General Public License for more details.

You should have received a copy of the GNU Lesser General Public License
along with this program. If not, see <http://www.gnu.org/licenses/>.

*******************************************************************************/
package PIRL.PVL;

//	The base class
import	javax.swing.tree.DefaultMutableTreeNode;
import	javax.swing.tree.MutableTreeNode;

import	java.util.List;
import	java.util.Vector;
import	java.util.Iterator;
import	java.util.ListIterator;
import	java.util.NoSuchElementException;
import	java.io.OutputStream;
import	java.io.StringWriter;
import	java.io.IOException;

import	PIRL.Strings.String_Buffer;

/**	A <I>Parameter</I> is a general-purpose named entity containing
	an arbitrary value. It is derived from the Parameter Value Language
	syntax:
<P>
	<BLOCKQUOTE>
	<B><I>Name</I> = <I>Value</I></B>
	</BLOCKQUOTE>
<P>
	Every Parameter has a <I>Name</I> which is a String of zero or more
	characters.
<P>
	The kind of <I>Value</I> that a Parameter contains is indicated by its
	<I>Classification</I> code. There are three basic classifications:
<DL>
<DT><CODE>TOKEN</CODE>
	<DD>A Parameter with a name but no value.
<DT><CODE>ASSIGNMENT</CODE>
	<DD>The fundamental Parameter containing a <I>Value</I> object.
<DT><CODE>AGGREGATE</CODE>
	<DD>A Parameter that contains a List of zero or more Parameters. An
	aggregate may have a <CODE>GROUP</CODE> or <CODE>OBJECT</CODE>
	specific classification; a distinction without a difference for
	this package (but which may be useful to the application).
</DL>
<P>
	There is also an <CODE>UNKNOWN</CODE> classification for a
	completely vacant Parameter (though its name is the empty string,
	not null). In addition, <CODE>END</CODE> classification Parameters
	may be used to mark the end of an aggregate's Parameter List without
	removing the Parameters beyond the end marker.
<P>
	A Parameter may have <I>Comments</I> associated with it that contain
	any descriptive text.
<P>
	@see		Value
	@see		Parser

	@author		Bradford Castalia, UA/PIRL
	@version	1.44 
*/

public class Parameter
	extends DefaultMutableTreeNode
	implements Cloneable
{
/**	Class name and version identification.
*/
public static final String
	ID = "PIRL.PVL.Parameter (1.44 2012/04/16 06:14:23)";

/*	The Parameter's name.

	It is the userObject of the base class and is always a String.
*/
/*
	When a Parameter is an ASSIGNMENT, _Value_ holds its Value.

	However, when a Parameter is an AGGREGATE, then the children
	variable in the super class holds a Vector of Parameters.
*/
private Value	_Value_				= null;

private int		_Classification_	= UNKNOWN;

/*	Parameter _Classification_ codes:

	There are three basic Parameter classifications:

		TOKEN      - _Value_ is null (no value),
		             children is null.
		ASSIGNMENT - _Value_ is a Value object,
		             children is null.
		AGGREGATE  - _Value_ is null,
		             children is a Vector of Parameter objects.

	In addition there is the UNKNOWN classification for empty
	Parameters.

	The _Classification_ for an AGGREGATE Parameter provides a
	distinction between a GROUP and an OBJECT: with the former having
	"GROUP" (or "BEGIN_GROUP") as its Parameter name in the PVL syntax,
	and the latter having "OBJECT" (or "BEGIN_OBJECT"). They are both
	treated identically in this package (the distinction is application
	dependent). The value assigned to these special Parameters is
	expected to be a string which becomes the Name of the corresponding
	Parameter.

	Note that while a Parameter classified as an END_XXX Parameter may
	be used, they are not required except in the external PVL syntax.
	The appropriate END_XXX Parameter is implicit, and will be provided
	automatically by the Write method, at the end of each AGGREGATE. An
	END_XXX Parameter inserted in the Parameter list of an AGGREGATE
	will functionally truncate the list at that point; the additional
	Parameters of the list remain and can be functionally recovered by
	removing the END_XXX Parameter. An END_PVL Parameter, will truncate
	any remaining PVL Parameters in the current list and any parent
	aggregates lists.

	Note that Parameter _Classification_ codes are expected to be
	distinct from Value _Type_ codes.
*/
/**	The lowest bit used by the classification code bit field.
*/
public static final int
	LOWEST_BIT		= Value.HIGHEST_BIT + 1;
/**	The highest bit used by the classification code bit field.
*/
public static final int
	HIGHEST_BIT		= LOWEST_BIT + 5;

/** Classification for a Parameter with only a name.
*/
public static final int
	TOKEN			= 1 << (LOWEST_BIT + 1);

/** Classification for a Parameter with a Value.
*/
public static final int
	ASSIGNMENT		= 1 << (LOWEST_BIT + 2);

/** Non-specific classification for a Parameter associated with
	an aggregate list.
<P>
	This classification is a place-holder for all Parameters
	associated with a Parameter aggregate list. If it is used
	with the Classification method, the default GROUP classification
	will be applied.
<P>
	@see	#GROUP
*/
public static final int
	AGGREGATE		= 1 << (LOWEST_BIT + 3);

/**	Used to mark the end of an aggregate Parameter's List.
<P>
	Do not set the classification of a Parameter to this value,
	but logically OR it with a specific classification. For example,
	to temporarily mark a Parameter as the end of an aggregate:
<P>
<PRE>Parameter parameter = aggregate.Find ("Parameter_Name");
if (parameter != null)
&nbsp;&nbsp;&nbsp;&nbsp;{
&nbsp;&nbsp;&nbsp;&nbsp;// Mark the end of the list here.
&nbsp;&nbsp;&nbsp;&nbsp;parameter.Classification (parameter.Classifcation () | END);
&nbsp;&nbsp;&nbsp;&nbsp;// The list stops with the parameter before "Parameter_Name".
&nbsp;&nbsp;&nbsp;&nbsp;parameter.Parent ().Write ();
&nbsp;&nbsp;&nbsp;&nbsp;// Restore the previous classification.
&nbsp;&nbsp;&nbsp;&nbsp;parameter.Classification (parameter.Classifcation () & ~END);
&nbsp;&nbsp;&nbsp;&nbsp;}
</PRE>
<P>
	Using the <CODE>END_AGGREGATE</CODE> classification may be easier.
<P>
	@see #END_AGGREGATE
	@see #END_PVL
*/
public static final int
	END				= 1 << LOWEST_BIT;

/** A special classification that marks the end of all Parameters.
<P>
	An <CODE>END_PVL</CODE> Parameter will truncate any remaining
	Parameters in the current List and the Lists of any Parent
	Parameters. However, the truncated Parameters remain in place and
	can be restored by restoring the classification of the marked
	Parameter.
<P>
	@see #END
	@see #END_AGGREGATE
*/
public static final int
	END_PVL			= TOKEN | END;

/**	Special classification for the end of an aggregate Parameter List.
<P>
	The <CODE>END_GROUP</CODE> and <CODE>END_OBJECT</CODE> are specific
	versions of this classification.
<P>
	Note that while a Parameter classified as an <CODE>END_XXX</CODE>
	Parameter may be used, they are not required (and typically not
	used) except in the PVL syntax seen by the Parser. The appropriate
	<CODE>END_XXX</CODE> Parameter is implicit, and will be provided
	automatically by the Write method, at the end of each aggregate. An
	<CODE>END_XXX</CODE> Parameter in a Parameter List will
	functionally truncate the List at that point; the additional
	Parameters of the List remain and can be functionally recovered by
	restoring the original classification. For example,
<P>
<PRE>Parameter parameter = aggregate.Find ("Parameter_Name");
if (parameter != null)
&nbsp;&nbsp;&nbsp;&nbsp;{
&nbsp;&nbsp;&nbsp;&nbsp;// Save the current classification.
&nbsp;&nbsp;&nbsp;&nbsp;int classification = parameter.Classification ();
&nbsp;&nbsp;&nbsp;&nbsp;// Mark the end of the list here.
&nbsp;&nbsp;&nbsp;&nbsp;parameter.Classification (Parameter.END_AGGREGATE);
&nbsp;&nbsp;&nbsp;&nbsp;// The list stops with the parameter before "Parameter_Name".
&nbsp;&nbsp;&nbsp;&nbsp;parameter.Parent ().Write ();
&nbsp;&nbsp;&nbsp;&nbsp;// Restore the previous classification.
&nbsp;&nbsp;&nbsp;&nbsp;parameter.Classification (classification);
&nbsp;&nbsp;&nbsp;&nbsp;}
</PRE>
<P>
	@see #END
	@see #END_PVL
*/
public static final int
	END_AGGREGATE	= AGGREGATE | END;

/**	Classification for a Parameter that contains a List of Parameters.
<P>
	This Parameter has no Value, but instead contains a List (Vector)
	of children Parameters. This classification is functionally
	identical to <CODE>OBJECT</CODE>. The distinction is based on the
	PVL keyword syntax recognized by the Parser when an aggregate
	Parameter is constructed.
<P>
	@see Parser
*/
public static final int
	GROUP			= AGGREGATE | (AGGREGATE << 1);
/**	Alias for <CODE>GROUP</CODE>.
<P>
	@see	#GROUP
*/
public static final int
	BEGIN_GROUP		= GROUP;
/**	A specific form of <CODE>END_AGGREGATE</CODE>.	
<P>
	@see	#END_AGGREGATE
*/
public static final int
	END_GROUP		= GROUP | END;

/**	Classification for an aggregate Parameter; and is functionally
	identical to the <CODE>GROUP</CODE> classification.
<P>
	Don't confuse this classification with the Java Object class. The
	name is a formal keyword in the PVL syntax.
<P>
	@see	#GROUP
*/
public static final int
	OBJECT			= AGGREGATE | (AGGREGATE << 2);
/**	Alias for <CODE>OBJECT</CODE>.
<P>
	@see	#OBJECT
*/
public static final int
	BEGIN_OBJECT	= OBJECT;
/**	A specific form of <CODE>END_AGGREGATE</CODE>.
<P>
	@see	#END_AGGREGATE
*/
public static final int
	END_OBJECT		= OBJECT | END;

/**	The classification of an empty Parameter.
*/
public static final int
	UNKNOWN			= 0;

/**	Selects only the classification code bit fields recognized by
	the Parameter class.
<P>
	Logical AND with another int variable will ensure that only
	Parameter classification bits remain.
*/
public static final int
	MASK			= 0x3F << LOWEST_BIT;

/*	Any, and all, comment sequences preceeding the Parameter
	definition in the PVL syntax are collected together.
*/
private String
	_Comments_		= null;

/*	The delimiter for Parameter pathnames
	used when finding a nested Parameter by name.
*/
private static final char
	PATH_DELIMITER	= '/';
private static char
	_Path_Delimiter_	= PATH_DELIMITER;


//	When throwing an exception is inappropriate.
private PVL_Exception
	_First_Warning_			= null,
	_Last_Warning_			= null;	
private boolean
	_Use_First_Warning_		= true;


//	DEBUG control.
private static final int
	DEBUG_OFF		= 0,
	DEBUG_SETUP		= 1 << 0,
	DEBUG_NAME		= 1 << 1,
	DEBUG_DATA		= 1 << 2,
	DEBUG_WRITE		= 1 << 3,
	DEBUG_FIND		= 1 << 4,
	DEBUG_MATCH		= 1 << 5,
	DEBUG_ALL		= -1,

	DEBUG			= DEBUG_OFF;

/*==============================================================================
	Constructors
*/
/**	Creates an empty Parameter, with classification <CODE>UNKNOWN</CODE>
	and an empty string name (not null).
*/
public Parameter ()
{
/*	Don't allow a null Parameter name.

	A null Parameter name is used as a special flag in the find method.
	Parameters are referenced by their names which are used in other
	contexts. A Parameter with an empty ("") name would be very
	unlikely (though possible); but a null name string should not be
	possible from the external view.
*/
super ("");
}

/**	Creates a token Parameter with only a name.
<P>
	@param	name	The name of this token. If null, the name will
		be set to the empty String.
*/
public Parameter
	(
	String		name
	)
{
super ((name == null) ? "" : name);
_Classification_ = TOKEN;
allowsChildren = false;
}

/**	Create a Parameter by getting all Parameters from a Parser.
<P>
	When the source for the Parser is a String, and the String's PVL
	defines more than one Parameter (excluding any Parameters that may
	be contained within aggregate Parameters), then an aggregate
	Parameter will be provided to contain all of the Parameters, and
	will be given the name {@link Parser#CONTAINER_NAME
	<CODE>Parser.CONTAINER_NAME</CODE>}. However, if the String only
	defines a single, top-level Parameter, then that is what will be
	provided.
<P>
	When the source for the Parser is not a String, then a container
	Parameter will always be used to hold the results from the Parser,
	regardless of the number of Parameters that are defined.
<P>
	@param	parser	The Parser source for PVL syntax.
	@throws	PVL_Exception	When the Parser encounters an
		unrecoverable error. The Parser may also generate a
		Warning which will associated with the new Parameter.
	@see	Parser
*/
public Parameter
	(
	Parser		parser
	)
	throws PVL_Exception
{
super ("");	//	In case the parser is null.
if ((DEBUG & DEBUG_SETUP) != 0)
	System.out.println
		(">>> Parameter construction from a Parser");
if (parser != null)
	{
	//	Graft the Parameter from the parser into this Parameter.
	graft (parser.Get ());
	_First_Warning_ = _Last_Warning_ = parser.Warning ();
	if ((DEBUG & DEBUG_SETUP) != 0 && _First_Warning_ != null)
		System.out.println
			("Parameter construction warning:\n" + Warning ());
	}
if ((DEBUG & DEBUG_SETUP) != 0)
	System.out.println
		("<<< Parameter construction from Parser completed.");
}

/**	A Parser is used to get a new Parameter which is given the
	specified name.
<P>
	This constructor is typically used when the source for the Parser
	is a File and the resultant aggregate Parameter is to be given the
	name of the File, rather than the default {@link
	Parser#CONTAINER_NAME <CODE>Parser.CONTAINER_NAME</CODE>}. Note
	that if the Parser source is a String that defines a single
	Parameter, then the name of that Parameter will be overriden by the
	new name from this constructor.
<P>
	@param	parser	The Parser source for PVL syntax. If null this
		is the same as using the no argument constructor.
	@param	name	The name of the new Parameter. If null, the name will
		be set to the empty String.
	@throws	PVL_Exception	When the Parser encounters an
		unrecoverable error. The Parser may also generate a
		Warning which will be associated with the new Parameter.
	@see	Parser
*/
public Parameter
	(
	String		name,
	Parser		parser
	)
	throws PVL_Exception
{
this (parser);
Name (name);
}

/**	A Parameter is constructed from a copy of another Parameter.
<P>
	The contents of the original Parameter are copied into the new
	Parameter. The warning status from the original is not copied.
<P>
	When the original Parameter is an assignment, the Value is
	completely copied.
<P>
	When the original is an aggregate, a new Parameter list is created
	in which each element is a recursive copy of the elements from the
	original list in the same order. <B>Note</B>: List copying stops at
	END_AGGREGATE parameters, and all copying stops at END_PVL
	parameters; the new Parameter is a logical copy of the original.
	Thus the new Parameter will be equivalent ({@link #equalsIgnoreCase
	<CODE>equalsIgnoreCase</CODE>}) to the original, but may not be
	identical ({@link #equals <CODE>equals</CODE>}).
<P>
	@param	parameter	The original Parameter to be copied. If null this
		is the same as using the no argument constructor.
	@throws PVL_Exception
		<DL>
		<DT>BAD_ARGUMENT
			<DD>An inappropriate object was found as the element of an
			aggregate Parameter list or Value array during copying.
		</DL>
	@see #set_classification(int)
	@see #listIterator ()
	@see #Add(Parameter)
	@see Value#Value(Value)
*/
public Parameter
	(
	Parameter	parameter
	)
	throws PVL_Exception
{
super ("");	//	In case the parameter is null.
if (parameter != null)
	{
	boolean
		end[] = {false};
	graft (replicate (parameter, end));
	}
}


private Parameter replicate
	(
	Parameter	parameter,
	boolean[]	end
	)
throws PVL_Exception
{
if ((DEBUG & DEBUG_SETUP) != 0)
	System.out.println (">>> replicate");
Parameter
	replica = new Parameter (parameter.Name ());

replica._Comments_ = parameter._Comments_;
//	Don't copy the warning status.

replica.set_classification (parameter.Classification ());

Replicate_List:
if (parameter.Is_Aggregate ())
	{
	if (parameter.children != null)
		{
		/*	Use the Parameter iterator.

			END_XXX parameters will end the list.

			END_PVL parameters will be specially noticed and the end flag
			will be set so that no further list copying will be done up the
			recursion line. 
		*/
		ListIterator
			list = parameter.listIterator ();
		while (list.hasNext ())
			{
			Object
				data = list.next ();
			if (! Is_Parameter (data))
				throw new PVL_Exception
					(
					ID, PVL_Exception.BAD_ARGUMENT,
					"While constructing a new Parameter\n"
					+ "  from the " + parameter.Classification_Name ()
						+ " Parameter named \"" + parameter.Name () + "\"\n"
					+ "  element " + list.previousIndex ()
						+ " is an object of class "
						+ data.getClass ().getName () + "."
					);
			/*	Add a replica of the Parameter from the list.
				This will recursively replicate any aggregate Parameters.
			*/
			if ((DEBUG & DEBUG_SETUP) != 0)
				System.out.println ("replicate: parameter " + (Parameter)data);
			replica.Add (replicate ((Parameter)data, end));
			if (end[0])
				{
				if ((DEBUG & DEBUG_SETUP) != 0)
					System.out.println ("replicate: END_PVL in sub-parameter");
				break Replicate_List;
				}
			}
		//	Test for END_PVL.
		try {list.next ();}
		catch (NoSuchElementException exception)
			{
			if (exception.getMessage ()
					.startsWith (Classification_Name (END_PVL) + "\n"))
				{
				end[0] = true;
				if ((DEBUG & DEBUG_SETUP) != 0)
					System.out.println ("replicate: END_PVL in list");
				}
			}
		}
	}
else if (parameter.Is_Assignment ())
	{
	if (parameter._Value_ != null)
		{
		replica._Value_ = new Value (parameter._Value_);
		if ((DEBUG & DEBUG_SETUP) != 0)
			System.out.println ("replicate: Value " + _Value_);
		}
	}
if ((DEBUG & DEBUG_SETUP) != 0)
	System.out.println ("<<< Parameter: replicate");
return replica;
}

/*==============================================================================
	Accessors
*/
/*------------------------------------------------------------------------------
	Name:
*/
/**	Gets the name of the Parameter.
<P>
	@return	The name of the Parameter. This will not be null.
*/
public String Name ()
{return (String)userObject;}

/**	Sets the Parameter's name.
<P>
	@param	name	The name String. If null the name will be set
		to the empty String.
	@return	This Parameter.
*/
public Parameter Name
	(
	String		name
	)
{
if ((DEBUG & DEBUG_NAME) != 0)
	System.out.println ("Parameter.Name = " + name);
userObject = (name == null) ? "" : name;
return this;
}


private Parameter set_name
	(
	String		name
	)
{
//	Allows setting the name to null.
userObject = name;
return this;
}

/**	Gets the name of the Parameter.
<P>
	@return	The name String of the Parameter.
*/
public String toString ()
{return (String)userObject;}

/**	Gets the fully qualified pathname of the Parameter.
<P>
	The pathname is the name of the Parameter preceeded by the current
	{@link #Path_Delimiter() <CODE>Path_Delimiter</CODE>} character. If
	this Parameter has a {@link #Parent() <CODE>Parent</CODE>} then the
	delimiter is preceeded by the parent's pathname. This recursion
	produces a pathname that starts with the delimiter character
	followed by the name of the root Parameter (no parent), the names
	of all containing aggregate Parameters leading to this Parameter
	separated by the delimiter, and ending with the name of this
	Parameter. If this Parameter has no parent, then the pathname is
	just its name preceeded by the delimiter.
<P>
	@return	The pathname String for the Parameter.
	@see	#Path_Delimiter()
	@see	#Parent()
*/
public String Path_Name ()
{
//	The full pathname including this Parameter name.
if (parent != null)
	return ((Parameter)parent).Path_Name ()
		+ _Path_Delimiter_ + (String)userObject;
return _Path_Delimiter_ + (String)userObject;
}

/**	Gets the pathname of the Parameter without the Parameter's name.
<P>
	The trailing {@link #Path_Delimiter() <CODE>Path_Delimiter</CODE>}
	character is included. If this Parameter has no parent, then only
	the delimiter will be returned.
<P>
	@return	The pathname String up to, but not including, the Parameter.
	@see	#Path_Name()
*/
public String Path_to_Name ()
{
//	The pathname without this Parameter name.
String
	pathname = Path_Name ();
return pathname.substring (0, pathname.lastIndexOf (_Path_Delimiter_) + 1);
}

/**	Gets the pathname of the Parameter from some parent Parameter.
<P>
	The pathname includes the names of each Parameter from, but not
	including, the root_parent up to, and including, this Parameter.
	The pathname will begin with the {@link #Path_Delimiter()
	<CODE>Path_Delimiter</CODE>} character and each Parameter name will
	be separated from the next by the same delimiter.
<P>
	<B>Note</B>: The pathname returned by this method is the absolute
	pathname for the Parameter in the root_parent, as used by the
	<CODE>Find</CODE> method.
<P>
	@param	root_parent	A Parent of this Parameter.
	@return	The absolute pathname to this Parameter from the
		root_parent, or null if the root_parent is not a parent of this
		Parameter.
*/
public String Path_from
	(
	Parameter	root_parent
	)
{
if (parent != null &&
	root_parent != null)
	{
	if (parent == root_parent)
		return _Path_Delimiter_ + (String)userObject;
	String
		pathname = ((Parameter)parent).Path_from (root_parent);
	if (pathname != null)
		return pathname + _Path_Delimiter_ + (String)userObject;
	}
return null;
}

/**	Gets the current Path_Name delimiter.
<P>
	The default delimiter is the slash ('/') character.
<P>
	@return	The delimiter char used for constructing pathnames.
*/
public static char Path_Delimiter ()
{return _Path_Delimiter_;}

/**	Sets the Path_Name delimiter.
<P>
	<B>Note</B>: The delimiter is a static class variable that applies
	to all Parameters.
<P>
	@param	delimiter	The delimiter char.
	@return	The previous delimiter character.
*/
public static char Path_Delimiter
	(
	char		delimiter
	)
{
char
	previous_delimiter = _Path_Delimiter_;
_Path_Delimiter_ = delimiter;
return previous_delimiter;
}

/**	Combines two pathnames to produce an absolute pathname.
<P>
	The base_path is taken to be relative to the parent_path. Typically
	the base_path is a simple name (e.g. a {@link #Basename Basename}),
	and the parent_path is a path from the root (e.g. a {@link
	#Parent_Pathname(String) Parent_Pathname}). However, any pathnames
	will do. If the parent_path is not {@link
	#Is_Absolute_Pathname(String) absolute} it is made absolute. The
	base_path, if not null or empty, is appended to the parent_path with
	a single separating delimiter.
<P>
	@param	parent_path	The pathname of the parent parameter. If null or
		empty, the root pathname is used.
	@param	base_path	The pathname of the parameter relative to the
		parent_path.
	@return	The absolute pathname.
*/
public static String Absolute_Pathname
	(
	String	parent_path,
	String	base_path
	)
{
StringBuffer
	absolute_path = new StringBuffer ();
char
	delimiter = Path_Delimiter ();
absolute_path.append (delimiter);

//	Find the first and last non-delimiter characters.
int
	first = 0,
	last = 0;
if (parent_path != null)
	{
	last = parent_path.length ();
	for (first = 0;
		 first < last &&
		 	parent_path.charAt (first) == delimiter;
		 first++);
	for (last--;
		 last >= 0 &&
		 	parent_path.charAt (last) == delimiter;
		 last--);
	last++;
	}
if (first < last)
	absolute_path.append (parent_path.substring (first, last));

//	Find the first non-delimiter character.
first = -1;
if (base_path != null)
	{
	last = base_path.length ();
	for (first = 0;
		 first < last &&
		 	base_path.charAt (first) == delimiter;
		 first++);
	if (first == last)
		first = -1;
	}
if (first >= 0)
	{
	if (absolute_path.length () > 1)
		absolute_path.append (delimiter);
	absolute_path.append (base_path.substring (first));
	}
return absolute_path.toString ();
}

/**	Produce an absolute pathname.
<P>
	The specified pathname is converted, if needed, to a top level
	absolute pathname.
<P>
	@param	pathname	The pathname of a parameter.
	@return	The absolute pathname.
*/
public static String Absolute_Pathname
	(
	String	pathname
	)
{return Absolute_Pathname (null, pathname);}


/**	Test if a pathname is absolute.
<p>
	An absolute pathname begins with the current {@link #Path_Delimiter ()
	path delimiter}.
<p>
	@param	pathname	The pathname of a parameter.
	@return	true if the pathname is absolute; false otherwise. A null or
		empty pathname is not absolute.
*/
public static boolean Is_Absolute_Pathname
	(
	String	pathname
	)
{
return
	pathname != null &&
	pathname.length () > 0 &&
	pathname.charAt (0) == Path_Delimiter ();
}

/**	Gets the portion of a pathname to the parent parameter.
<P>
	The parent pathname is the substring of the pathname preceding the
	last {@link Parameter#Path_Delimiter() path delimiter} (usually the
	'/' character). If there is no delimiter in the pathname, or
	nothing preceding the last delimiter, then there is no parent
	pathname.
<P>
	<B>N.B.</B>: This function only considers the specified pathname.
	It does not provide the name of a Parameter's {@link #Parent()
	parent}.
<P>
	@param	pathname	The String naming a pathname to a parameter.
	@return	The String naming the pathname to the parent parameter, or
		null if the parameter has no parent (or the specified pathname
		is null).
*/
public static String Parent_Pathname
	(
	String	pathname
	)
{
if (pathname != null)
	{
	int
		index = pathname.lastIndexOf (Path_Delimiter ());
	if (index > 0)
		return pathname.substring (0, index);
	}
return null;
}

/**	Gets the basename portion of the pathname.
<P>
	The basename is the substring of the pathname following the last
	path {@link Parameter#Path_Delimiter() delimiter} (usually the '/'
	character). If the pathname does not contain a delimiter, then the
	entire pathname is the basename.
<P>
	@param	pathname	The String naming a pathname to a parameter.
	@return	The basename substring of the pathname. This will be null
		if the specified pathname is null. It will be the empty String
		if the pathname ends with the path delimiter.
*/
public static String Basename
	(
	String	pathname
	)
{
if (pathname == null)
	return null;
int
	index = pathname.lastIndexOf (Path_Delimiter ());
if (index < 0)
	return pathname;
return pathname.substring (++index);
}

/**	Gets a PVL description of the Parameter using a specified Lister.
<p>
	If no Lister is specified (the argument is null) then a default
	Lister will be provided.
<p>
	The Lister will have its Writer {@link Lister#Set_Writer(Writer) set}
	to a StringWriter used to write the PVL syntax. The Parameter is
	{@link #Write(OutputStream) Write}n using the Lister and the
	resultant String, without any final end-of-line sequence, is
	returned. If the Write method produces an exception, the exception
	message is returned after a {@link Lister#NEW_LINE} (this should be a
	unique flag that producing the PVL description failed).
<P>
	@param	lister	The Lister to use when writing the PVL syntax.
	@return	The String describing the Parameter in PVL syntax.
*/
public String Description
	(
	Lister	lister
	)
{
String
	description = null;
StringWriter
	writer = new StringWriter ();
if (lister == null)
	lister = new Lister (writer);
else
	lister.Set_Writer (writer);
try
	{
	lister.Write (this);
	description = writer.toString ();
	if (description.length () != 0)
		{
		String
			EOL = lister.Strict () ? Parser.LINE_BREAK : Lister.NEW_LINE;
		if (description.endsWith (EOL))
			description = description.substring
							(0, description.length () - EOL.length ());
		}
	}
catch (Exception exception)
	{description = Lister.NEW_LINE + exception.getMessage ();}
return description;
}

/**	Gets a PVL description of the Parameter.
<p>
	The Parameter is {@link #Write(OutputStream) Write}n to a
	StringWriter on a default Lister and the resultant String is
	returned. If the Write method produces an exception, the exception
	message is returned.
<P>
	@return	The String describing the Parameter in PVL syntax.
	@see	#Description(Lister)
*/
public String Description ()
{return Description (null);}

/*------------------------------------------------------------------------------
	Value:
*/
/*..............................................................................
	Get
*/
/**	Gets the Value assigned to the Parameter.
<P>
	@return	The Parameter's Value. If the Parameter has no Value, null
		is returned.
	@throws	PVL_Exception
		<DL>
		<DT>ILLEGAL_SYNTAX
			<DD>If the Parameter is not an assignment or token.
		</DL>
*/
public Value Value ()
	throws PVL_Exception
{
if (Is_Assignment ())
	return _Value_;
if (Is_Token ())
	return null;
throw new PVL_Exception
	(
	ID, PVL_Exception.ILLEGAL_SYNTAX,
	"Can't get a Value from the " + Classification_Name ()
		+ " Parameter named \"" + Name () + "\"."
	);
}

/**	Gets the Parameter list aggregated under this Parameter.
<P>
	@return	The Parameter's list as a Vector object.
		If the Parameter has no Parameter list, null is returned. If
		the Parameter has no list because it is not classified as an
		aggregate, then a <CODE>PVL_Exception.ILLEGAL_SYNTAX</CODE>
		warning status is set.
	@throws	PVL_Exception
		<DL>
		<DT>ILLEGAL_SYNTAX
			<DD>If the Parameter is not an aggregate.
		</DL>
*/
public Vector List ()
	throws PVL_Exception
{
if (Is_Aggregate ())
	return children;
throw new PVL_Exception
	(
	ID, PVL_Exception.ILLEGAL_SYNTAX,
	"Can't get a Parameter List from the " + Classification_Name ()
		+ " Parameter named \"" + Name () + "\"."
	);
}

/**	Determines if the Parameter list is empty.
<P>
	@return	true if the Parameter list is empty, or the Parameter has
		no list (a Parameter that is not classified as an aggregate
		will have no list). Otherwise, false.
	@see	Vector#isEmpty()
*/
public boolean Is_Empty ()
{
if (Is_Aggregate () &&
	children != null)
	return children.isEmpty ();
return true;
}

/**	Get the Parameter's data object.
<P>
	@return	The Parameter's data object.
		If the Parameter is classified as an aggregate, this will be
		the Parameter list (Vector). Otherwise it will be the
		Parameter's Value. In either case, the object may be null.
*/
public Object Data ()
{
if (Is_Aggregate ())
	return children;
return _Value_;
}

/*..............................................................................
	Set
*/
/**	Sets the Value for the Parameter.
<P>
	The Parameter is classified as an assignment and the specified
	Value is assigned. If the Parameter previously had data, it is
	replaced. In the case where previous data was an aggregate
	Parameter list, all elements of the list are orphaned (they
	are marked as having no parent). <B>Note</B>: The Value is
	not copied.
<P>
	@param	value	The Value object to be assigned.
	@return	This Parameter.
	@see	#set_classification(int)
	@see	#set_data(Object)
*/
public Parameter Value
	(
	Value		value
	)
{
set_classification (ASSIGNMENT);
try {set_data (value);}
catch (PVL_Exception exception) {/* Shouldn't happen */}
return this;
}

/**	Assigns a new Value constructed from a primitive byte value.
<P>
	@param	value	The primitive datum for the new Value.
	@return	This Parameter.
	@throws	PVL_Exception	If a new Value could not be constructed from
		the value.
	@see	#Value(Value)
*/
public Parameter Value (byte   value) throws PVL_Exception {return Value (new Value (value));}

/**	Assigns a new Value constructed from a primitive short value.
<P>
	@param	value	The primitive datum for the new Value.
	@return	This Parameter.
	@throws	PVL_Exception	If a new Value could not be constructed from
		the value.
	@see	#Value(Value)
*/
public Parameter Value (short  value) throws PVL_Exception {return Value (new Value (value));}

/**	Assigns a new Value constructed from a primitive int value.
<P>
	@param	value	The primitive datum for the new Value.
	@return	This Parameter.
	@throws	PVL_Exception	If a new Value could not be constructed from
		the value.
	@see	#Value(Value)
*/
public Parameter Value (int    value) throws PVL_Exception {return Value (new Value (value));}

/**	Assigns a new Value constructed from a primitive long value.
<P>
	@param	value	The primitive datum for the new Value.
	@return	This Parameter.
	@throws	PVL_Exception	If a new Value could not be constructed from
		the value.
	@see	#Value(Value)
*/
public Parameter Value (long   value) throws PVL_Exception {return Value (new Value (value));}

/**	Assigns a new Value constructed from a primitive float value.
<P>
	@param	value	The primitive datum for the new Value.
	@return	This Parameter.
	@throws	PVL_Exception	If a new Value could not be constructed from
		the value.
	@see	#Value(Value)
*/
public Parameter Value (float  value) throws PVL_Exception {return Value (new Value (value));}

/**	Assigns a new Value constructed from a primitive double value.
<P>
	@param	value	The primitive datum for the new Value.
	@return	This Parameter.
	@throws	PVL_Exception	If a new Value could not be constructed from
		the value.
	@see	#Value(Value)
*/
public Parameter Value (double value) throws PVL_Exception {return Value (new Value (value));}

/**	Assigns a new Value constructed from an Object.
<P>
	@param	value	The Object from which to construct the new Value.
	@return	This Parameter.
	@throws	PVL_Exception	If a new Value could not be constructed from
		the value.
	@see	#Value(Value)
*/
public Parameter Value
	(
	Object	value
	)
	throws PVL_Exception
{
if (value == null)
	return Data (value);
return Value (new Value (value));
}

/**	Sets the Parameter list for the Parameter.
<P>
	The Parameter is classified as an <CODE>AGGREGATE</CODE> (i.e.
	<CODE>GROUP</CODE>) and the contents of the Vector are adopted. The
	new list must contain only Parameter elements, each of which has
	its parent set to this Parameter. If the Parameter previously had
	data, it is replaced. In the case where previous data was an
	aggregate Parameter list, all elements of the list are orphaned
	(they are marked as having no parent). <B>Note</B>: The list is not
	copied.
<P>
	@param	list	The Vector of Parameters to be adopted.
	@return	This Parameter.
	@throws	PVL_Exception	From set_data.
	@see	#set_classification(int)
	@see	#set_data(Object)
*/
public Parameter List
	(
	Vector		list
	)
	throws PVL_Exception
{
set_classification (GROUP);
set_data (list);
return this;
}

/**	Gets the size of an aggregate's list; the number of Parameters
	it contains.
<P>
	@return	The number of Parameters in the aggregate list. If the
		Parameter is not an aggregate or has no list, zero will be
		returned; i.e. returning zero is not determinate that the
		Parameter is an aggregate with an empty list. <b>N.B.</b>: This
		value does not include the number of Parameters in any
		aggregates in this Parameters list.
*/
public int List_Size ()
{
if (Is_Aggregate () && children != null)
	return children.size ();
return 0;
}

/**	Copies the contents of a Parameter into this Parameter.
<P>
	This Parameter is made equivalent ({@link
	#equalsIgnoreCase(Parameter) <CODE>equalsIgnoreCase</CODE>} not
	necessarily {@link #equals(Object) <CODE>equals</CODE>}) to the
	specified Parameter.
<P>
	@param	parameter	The Parameter to be copied into this Parameter.
		If null, and empty parameter will be used.
	@return	This Parameter.
	@throws	PVL_Exception	From graft.
	@see	#Parameter(Parameter)
	@see	#graft(Parameter)
*/
public Parameter Data
	(
	Parameter	parameter
	)
	throws PVL_Exception
{
//	Graft a deep copy of the original Parameter.
graft (new Parameter (parameter));
return this;
}

/**	Sets the data for the Parameter.
<P>
	If the data is a Parameter object then its contents are copied into
	this Parameter; i.e. this Parameter is made equivalent ({@link
	#equalsIgnoreCase(Parameter) <CODE>equalsIgnoreCase</CODE>} not
	necessarily {@link #equals(Object) <CODE>equals</CODE>}) to the
	specified Parameter. Otherwise the Parameter is classified based on
	the {@link #Classification <CODE>Classification</CODE>} of the data
	Object specified. The classification will be an
	<CODE>ASSIGNMENT</CODE> if the data is a Value, an
	<CODE>AGGREGATE</CODE> (i.e. <CODE>GROUP</CODE>) for Parameter list
	data, or a <CODE>TOKEN</CODE> if the data is null (unless the
	Parameter has no name, in which case the classification is
	<CODE>UNKNOWN</CODE>). The data is applied to the Parameter
	appropriately as if either the {@link #Value(Value) <CODE>Value
	(Value)</CODE>} or {@link #List(Vector) <CODE>List (Vector)</CODE>}
	method were used.
<P>
	@param	data	The data Object to be applied.
	@return	This Parameter.
	@throws	PVL_Exception	From set_data.
	@see	#Data(Parameter)
	@see	#Classification(Object)
	@see	#set_classification(int)
	@see	#set_data(Object)
	@see	#Value(Value)
	@see	#List(Vector)
*/
public Parameter Data
	(
	Object		data
	)
	throws PVL_Exception
{
if (Is_Parameter (data))
	return Data ((Parameter)data);
int
	classification = Classification (data);
if (classification == AGGREGATE)
	classification = GROUP;		//	Be specific.
set_classification (classification);
if (data == null && Name () != null && Name ().length () != 0)
	set_classification (TOKEN);	//	Instead of UNKNOWN.
set_data (data);
return this;
}

/*..............................................................................
	Modify
*/
/**	Inserts a Parameter into the aggregate list at the specified index.
<P>
	If the Parameter is not already classified as an aggregate, and
	does not have a Value assigned, it is given the <CODE>GROUP</CODE>
	classification. The specified Parameter is inserted at the
	specified index in the aggregate Parameter list. Any Parameters in
	the list at, or beyond the index are moved to the next index.
<P>
	After the specified Parameter is inserted its parent is set to
	this Parameter.
<P>
	<B>Warning</B>: Inserted parameters are not removed from any
	Aggregate list they may already be in.
<P>
	@param	parameter	The Parameter object to be inserted.
	@param	index		The location in the list for the insertion.
	@return	This Parameter.
	@throws	PVL_Exception
		<DL>
		<DT><CODE>ILLEGAL_SYNTAX</CODE>
			<DD>The Parameter is an assignment with a Value.
		</DL>
	@see	#Classification(int)
	@see	#Remove(Parameter)
	@see	Vector#insertElementAt(Object, int)
*/
public Parameter Insert
	(
	Parameter	parameter,
	int			index
	)
	throws PVL_Exception
{
if (parameter == null)
	return this;

if ((DEBUG & DEBUG_DATA) != 0)
	System.out.println
		(">>> Parameter.Insert: " + parameter.Name () + " into " + Name () + " at index " + index);
if (Is_Assignment () && _Value_ != null)
	throw new PVL_Exception
		(
		ID, PVL_Exception.ILLEGAL_SYNTAX,
		"Can't Insert the Parameter named \"" + parameter.Name () + "\"\n"
		+ "  into the " + Classification_Name ()
			+ " Parameter named \"" + Name () + "\"."
		);

if (! Is_Aggregate ())
	Classification (GROUP);

//	Adopt it as one of the children.
if (children == null)
	children = new Vector ();
try {children.insertElementAt (parameter, index);}
catch (RuntimeException exception)
	{
	throw new PVL_Exception
		(
		ID, PVL_Exception.SYSTEM_ERROR,
		"During Insert at index " + index
			+ " of the Parameter named \"" + parameter.Name () + "\"\n"
		+ exception.getMessage ()
		);
	}
parameter.parent = this;
if ((DEBUG & DEBUG_DATA) != 0)
	System.out.println ("<<< Parameter.Insert");
return this;
}

/**	Inserts a List of Parameters into the aggregate list starting at
	the specified index.
<P>
	The elements of the List are <CODE>Insert</CODE>ed in the order
	they occur from the List's listIterator at sequential locations
	starting with the specified index.
<P>
	@param	list	The List of Parameters to Insert.
	@param	index	The first of sequential Insert locations.
	@return	This Parameter.
	@throws	PVL_Exception
		<DL>
		<DT><CODE>BAD_ARGUMENT</CODE>
			<DD>An element of the List is not a Parameter.
		</DL>
	@see	#Insert(Parameter, int)
	@see	List#listIterator()
*/
public Parameter Insert
	(
	List		list,
	int			index
	)
	throws PVL_Exception
{
if (list != null)
	{
	ListIterator
		parameter_list = list.listIterator ();
	while (parameter_list.hasNext ())
		{
		Object object = parameter_list.next ();
		if (! Is_Parameter (object))
			throw new PVL_Exception
				(
				ID, PVL_Exception.BAD_ARGUMENT,
				"During Insert at index " + index + " of a Parameter List\n"
				+ "  to the " + Classification_Name ()
					+ " Parameter named \"" + Name () + "\"\n"
				+ "  List element " + parameter_list.previousIndex ()
					+ " is an object of class "
					+ object.getClass ().getName () + "."
				);
		Insert ((Parameter)object, index++);
		}
	}
return this;
}

/**	Adds a Parameter to the end of the aggregate list.
<P>
	Uses the <CODE>Insert</CODE> method where the index is the end of
	the list.
<P>
	@param	parameter	The Parameter to be added.
	@return	This Parameter.
	@throws	PVL_Exception	From Insert.
	@see	#Insert(Parameter, int)
*/
public Parameter Add
	(
	Parameter	parameter
	)
	throws PVL_Exception
{return Insert (parameter, getChildCount ());}

/**	Adds a List of Parameters to the end of the aggregate list.
<P>
	Performs an {@link #Insert(List, int) Insert} where the
	index is the end of the list.
<P>
	@param	list	The List of Parameters to be added.
	@return	This Parameter.
	@throws	PVL_Exception	From Insert.
	@see	#Insert(List, int)
*/
public Parameter Add
	(
	List		list
	)
	throws PVL_Exception
{return Insert (list, getChildCount ());}

/**	Adds the Parameters obtained from a Parser to this Parameter's
	aggregate list.
<P>
	@param	parser	The Parser to use as the source of PVL statements.
	@return	This Parameter.
	@throws	PVL_Exception	From the Parser.
		<B>Note</B>: The warning status for the Parser is cleared before
		it is called, and then applied to this Parameter afterwards.
	@see	Parser#Add_To(Parameter)
	@see	Parser#Warning()
*/
public Parameter Add
	(
	Parser	parser
	)
	throws PVL_Exception
{
parser.Reset_Warning ();	//	Start clean.
parser.Add_To (this);
Warning (parser.Warning ());
return this;
}

/**	Removes the Parameter at the specified index from the aggregate list.
<P>
	After being removed from the aggregate list the Parameter is marked
	as having no parent.
<P>
	@param	index	The aggregate list index of the Parameter to be removed.
	@return	The Parameter removed. 
		If this Parameter is not an aggregate, or has no list, null is
		returned. If this Parameter is not an aggregate a
		<CODE>PVL_Exception.ILLEGAL_SYNTAX</CODE> Warning status is
		set.
	@see	Vector#remove(int)
*/
public Parameter Remove
	(
	int			index
	)
{
Parameter
	removed = null;
if (! Is_Aggregate ())
	Warning
		(
		PVL_Exception.ILLEGAL_SYNTAX,
		"Can't Remove from the " + Classification_Name ()
			+ " Parameter named \"" + Name () + "\"."
		);
if (children != null)
	{
	//	Remove the entry from the Parameter list.
	removed = (Parameter)children.remove (index);
	//	Make it an orphan.
	removed.parent = null;
	}
return removed;
}

/**	Removes the specified Parameter from the aggregate list.
<P>
	The first occurance of the Parameter in the list is removed. The
	aggregate list, if there is one, is searched regardless of the
	classification of this Parameter. Only this Parameter list is
	searched, not the lists of any aggregate Parameters in the list.
	The Parameter is found in the aggregate list by equality of
	Parameter object reference, not by the <CODE>equals</CODE> method.
	To remove a Parameter that equals another Parameter, use
	<CODE>Find</CODE>:
<P>
<PRE>// Use Path_from only if the specific location matters.
parameter = aggregate.Find (parameter.Name (parameter.Path_from (aggregate)));
if (parameter != null)
&nbsp;&nbsp;&nbsp;&nbsp;parameter.Parent ().Remove (parameter);
</PRE>
<P>
	@param	parameter	The Parameter object to be removed. If null
		nothing is done and null is returned.
	@return	The Parameter removed. 
		If this Parameter is not an aggregate, or has no list, null is
		returned. If this Parameter is not an aggregate an
		<CODE>ILLEGAL_SYNTAX</CODE> Warning status is produced.
	@see	#Remove(int)
*/
public Parameter Remove
	(
	Parameter	parameter
	)
{
Parameter
	removed = null;
if (parameter != null &&
	children != null)
	{
	for (int index = 0;
		 index < children.size ();
		 index++)
		{
		if ((removed = (Parameter)children.get (index)) == parameter)
			{
			//	Remove the entry from the Parameter list.
			children.remove (index);
			//	Make it an orphan.
			removed.parent = null;
			}
		}
	}
return removed;
}

/**	Empties the Parameter aggregate list.
<P>
	The aggregate list, if it exists, is removed regardless of the
	classification of this Parameter. All Parameters in the list (but
	not sub-lists) are marked as not having a parent.
<P>
	@return	This Parameter.
*/
public Parameter Remove_All ()
{
orphan (children);
children = null;
return this;
}

/*..............................................................................
*/
/**	Set the appropriate Data variables.
<P>
	The <CODE>Classification</CODE> of the data object determines how
	the internal data variables are set. Consistency of the Parameter's
	data is ensured by always setting it with this method.
<P>
	For Vector data, the data becomes the base class aggregate list
	(children) Vector. Its contents are adopted which confirms that
	each element is a Parameter and its parent is set to this
	Parameter. This Parameter's Value is set to null.
<P>
	For Value data, the data becomes the Value of the Parameter. The
	aggregate list (children) contents are orphaned and then the
	children Vector is set to null.
<P>
	<B>>>> WARNING <<<</B> The classification of the Parameter is
	not updated. Accordingly, this method should only be used by
	methods that will also set the classification appropriately.
<P>
	@param	data	The Object to become the Parameter's data.
	@throws	PVL_Exception
		<DL>
		<DT><CODE>BAD_ARGUMENT</CODE>
			<DD>If the data object is not a valid class for a Parameter
				(null is acceptable), or the data object is a Vector and
				one of its elements is not a Parameter.
		</DL>
	@see	#Classification(Object)
	@see	#adopt(Vector)
	@see	#orphan(List)
*/
protected void set_data
	(
	Object		data
	)
	throws PVL_Exception
{
switch (Classification (data))
	{
	case UNKNOWN:	//	The data is null.
	case ASSIGNMENT:
		orphan (children);
		children = null;
		_Value_ = (Value)data;
		break;
	case AGGREGATE:
		/*
			N.B.: Each DefaultMutableTreeNode contains a parent
			reference which must be updated when it is moved into a new
			parent. So transfering a Parameter list into this Parameter
			requires setting the parent of each Parameter in the list.
		*/
		children = (Vector)data;
		adopt (children);
		_Value_ = null;
		break;
	default:
		throw new PVL_Exception
			(
			ID, PVL_Exception.BAD_ARGUMENT,
			"Can't set_data of a Parameter to an object of class "
				+ data.getClass ().getName () + "."
			);
	}
}

/**	Grafts the contents of a Parameter into this Parameter.
<P>
	In effect, this Parameter becomes the specified Parameter.
<P>
	The Value reference is set (not copied) to the specified
	Parameter's Value reference along with the name, comments, and
	warning status. The aggregate list (children) Vector reference is
	set to the specified Parameter's Vector and if it has contents they
	are adopted by this Parameter (the specified Parameter's children
	Vector is set to null to avoid any possible confusion). The
	classification is set to be the same as the specified Parameter.
	<B>Note</B>: The parent of this Parameter remains unchanged; i.e.
	the <I>contents</I> of are replaced but the <I>context</I> stays
	the same.
<P>
	@param	parameter	The Parameter object to graft into this Parameter.
		If null, nothing is done.
	@throws	PVL_Exception	From adopt.
	@see	#adopt(Vector)
	@see	#set_classification(int)
*/
protected void graft
	(
	Parameter	parameter
	)
	throws PVL_Exception
{
if (parameter != null)
	{
	//	Graft the contents of the Parameter into this Parameter.
	userObject = parameter.userObject;
	_Value_ = parameter._Value_;
	if ((children = parameter.children) != null)
		{
		//	Become the parent of the children.
		parameter.children = null;
		adopt (children);
		}
	set_classification (parameter._Classification_);
	_Comments_ = parameter._Comments_;
	_First_Warning_ = parameter._First_Warning_;
	_Last_Warning_ = parameter._Last_Warning_;
	_Use_First_Warning_ = parameter._Use_First_Warning_;
	}
}

/**	Sets the parents of a Vector of Parameters to this Parameter.
<P>
	<B>>>> WARNING <<<</B>
	Only the parent references for the Parameters in the Vector are set
	to this Parameter. The Vector itself may still be referenced by some
	other Parameter. In this case the adoption of the Vector will not be
	complete until the reference in the former parent is changed (e.g.
	to null).
<P>
	@param	list	The Vector of Parameters to adopt.
	@throws	PVL_Exception
		<DL>
		<DT><CODE>BAD_ARGUMENT</CODE>
			<DD>If an entry in the list is not a Parameter.
		</DL>
*/
protected void adopt
	(
	Vector		list
	)
	throws PVL_Exception
{
Object
	parameter;
if (list != null)
	{
	for (int index = list.size ();
		   --index >= 0;)
		{
		if (Is_Parameter (parameter = list.get (index)))
			//	Set the child's parent to this Parameter.
			((Parameter)parameter).parent = this;
		else
			throw new PVL_Exception
				(
				ID, PVL_Exception.BAD_ARGUMENT,
				"Can't adopt an object from a List\n"
				+ "  into the Parameter named \"" + Name () + "\"\n"
				+ "  element " + index + " is an object of class "
					+ parameter.getClass ().getName () + "."
				);
		}
	}
}

/**	Sets a list of Parameters to have no parent.
<P>
	<B>>>> WARNING <<<</B>
	Only the parent references for the Parameters in the list are set
	to null. The list itself may still be referenced by some other
	Parameter. In this case the orphaning of the Vector will not be
	complete until the reference in the former parent is changed (e.g.
	to null).
<P>
	@param	list	The List of Parameters to orphan.
*/
static protected void orphan
	(
	List		list
	)
{
Object
	parameter;
if (list != null)
	for (int index = list.size ();
		   --index >= 0;)
		if (Is_Parameter (parameter = list.get (index)))
			((Parameter)parameter).parent = null;
}

/*------------------------------------------------------------------------------
	Classification:
*/
/**	Gets the Parameter's classification code.
<P>
	@return	The classification code.
*/
public int Classification ()
{return _Classification_;}

/**	Determine the general classification of an Object.
<P>
	The Object is considered to be potential Parameter data and is
	tested to see if it is an instance of an acceptable Java Class for
	Parameter data. Note that only a general classification can be
	determined by examining an Object.
<P>
	@param	object	The Object to be tested.
	@return	The basic classification code category int that would be
		associated with a Parameter having the object for its data. If
		the object is a Value, then the <CODE>ASSIGNMENT</CODE>
		classification is returned; if a Vector, then the
		<CODE>AGGREGATE</CODE> code is returned; and if the argument is
		null then the <CODE>UNKNOWN</CODE> classification is returned.
		If an invalid object is specified, then the return value is -1.
*/
public static int Classification
	(
	Object	object
	)
{
if (object instanceof Value)
	return ASSIGNMENT;
else if (object instanceof Vector)
	return AGGREGATE;
else if (object == null)
	return UNKNOWN;
return -1;
}

/**	Sets the classification of the Parameter.
<P>
	<B>Note</B>: Changing to a new classification may result in loss of
	data if the Parameter's previous data is invalid under the new
	classification. If the new classification is not for assignment,
	then any Value is dropped; if the new classification is not for an
	aggregate, then any existing list is removed. However,
	<CODE>END_XXX</CODE> classifications are exempt:
	<CODE>END_XXX</CODE> classification Parameters are normally
	implicit at the end of aggregate Parameter lists. They are used
	explicitly to mark, probably temporarily, the end of a list or an
	entire hierarchy (<CODE>END_PVL</CODE>). Therefore, when these
	classifications are applied to a Parameter its underlying data is
	left intact.
<P>
	@param	class_code	The new Parameter classification int.
	@return	This Parameter.
	@see	#END_AGGREGATE
	@see	#END_PVL
	@see	#set_classification(int)
	@see	#Remove_All()
*/
public Parameter Classification
	(
	int		class_code
	)
{
if (class_code == AGGREGATE)
	class_code = GROUP;

set_classification (class_code);

if (! Is_End (class_code))
	{
	if (! Is_Assignment (class_code))
		_Value_ = null;
	if (! Is_Aggregate (class_code))
		Remove_All ();
	}
return this;
}

/**	Sets the appropriate classification variables.
<P>
	Consistency of the Parameter's classification is ensured by always
	setting it with this method. If the classification code is for an
	<CODE>AGGREGATE</CODE>, then the allowsChildren variable of the
	DefaultMutableTreeNode base class will be set to true; otherwise
	it will be set to false.
<P>
	<B>>>> WARNING <<<</B> The data of the Parameter is not checked
	for consistency with the classification. Accordingly, this method
	should only be used by methods that will also set the data
	appropriately.
<P>
	@param	class_code	The int classification code for the Parameter.
*/
protected void set_classification
	(
	int		class_code
	)
{
if ((DEBUG & DEBUG_DATA) != 0)
	System.out.println
		("Parameter.set_classification: For "
		+ Name () + " to " + Classification_Name(class_code));
_Classification_ = class_code & MASK;
if (Is_Aggregate ())
	{
	allowsChildren = true;
	if (_Classification_ == AGGREGATE)
		//	Force a specific classification.
		_Classification_ = GROUP;
	}
else
	allowsChildren = false;
if ((DEBUG & DEBUG_DATA) != 0)
	System.out.println
		("Parameter.set_classification: allowsChildren = " + allowsChildren);
}

/**	Gets the name String associated with the classification code.
<P>
	Classification names are identical to the names of the
	classification code constants, except that the
	<CODE>BEGIN_XXX</CODE> aliases are not used.
<P>
	@param	class_code	The classification code int.
	@return	The appropriate classification name String. An invalid
		classification code will return a null.
*/
public static String Classification_Name
	(
	int		class_code
	)
{
switch (class_code)
	{
	case ASSIGNMENT:	return "ASSIGNMENT";
	case AGGREGATE:		return "AGGREGATE";
	case END_AGGREGATE:	return "END_AGGREGATE";
	case GROUP:			return "GROUP";
	case END_GROUP:		return "END_GROUP";
	case OBJECT:		return "OBJECT";
	case END_OBJECT:	return "END_OBJECT";
	case UNKNOWN:		return "UNKNOWN";
	case TOKEN:			return "TOKEN";
	case END:
	case END_PVL:		return "END";
	default:			return null;
	}
}

/**	Gets the name String associated with the Parameter's classification.
<P>
	@return	The appropriate classification name String.
*/
public String Classification_Name ()
{return Classification_Name (_Classification_);}

/*..............................................................................
	Parameter checkers:
*/
/**	Tests if an Object is a Parameter.
<P>
	@param	object	The Object to be tested.
	@return	true if the object is an instance of the Parameter class;
		otherwise false (also false if the object is null).
*/
public static boolean Is_Parameter (Object object)
{return (object instanceof Parameter);}

/**	Tests if an Object is a Value suitable for a Parameter.
<P>
	@param	object	The Object to be tested.
	@return	true if the object is an instance of the Value class;
		otherwise false (also false if the object is null).
	@see	Value#Is_Value(Object)
*/
public static boolean Is_Parameter_Value (Object object)
{return Value.Is_Value (object);}

/**	Tests if an Object is a List (Vector) suitable for an aggregate
	Parameter.
<P>
	<B>Note</B>: The contents of the list are recursively checked to
	all be Parameters. Aggregates without lists are acceptable
	Parameters.
<P>
	@param	object	The Object to be tested.
	@return	true if the object is an instance of the Vector class and
		all its elements are Parameters; otherwise false (also false if
		the object is null, though a null list is acceptable [hmmm]).
	@see	#Is_Parameter(Object)
	@see	#Is_Begin_Aggregate()
*/
public static boolean Is_Parameter_List (Object object)
{
if (object instanceof Vector)
	{
	Object
		parameter, list;
	for (int index = ((Vector)object).size ();
		   --index >= 0;)
		{
		if (! Is_Parameter (parameter = ((Vector)object).get (index)))
			return false;
		if (((Parameter)parameter).Is_Begin_Aggregate () &&
			(((Parameter)parameter).children != null) &&
			! Is_Parameter_List (((Parameter)parameter).children))
			return false;
		}
	return true;
	}
return false;
}

/*..............................................................................
	Classification checkers:
*/
/**	Tests if the Parameter is classified as a token.
<P>
	@return	true if the Parameter is a token.
	@see	#TOKEN
*/
public boolean Is_Token ()
{return ((_Classification_ & TOKEN) != 0);}

/**	Tests if the Parameter is one of the <CODE>END</CODE> classifications.
<P>
	<CODE>END</CODE> classifcations include both the end of aggregates
	({@link #END_GROUP <CODE>END_GROUP</CODE>} and {@link #END_OBJECT
	<CODE>END_OBJECT</CODE>}) and the end the entire hierarchy ({@link
	#END_PVL <CODE>END_PVL</CODE>})
<P>
	@return	true if the the Parameter has an <CODE>END</CODE> classification.
	@see	#END
*/
public boolean Is_End ()
{return ((_Classification_ & END) != 0);}

/**	Tests if the Parameter is the <CODE>END_PVL</CODE> classification.
<P>
	@return	true if the Parameter has the <CODE>END_PVL</CODE> classification.
	@see	#END_PVL
*/
public boolean Is_End_PVL ()
{return (_Classification_ == END_PVL);}

/**	Tests if the Parameter is classified as an assignment.
<P>
	An assignment Parameter does not necessarily have a Value (it may
	be null).
<P>
	@return	true if the Parameter is a Value assignment.
	@see	#ASSIGNMENT
*/
public boolean Is_Assignment ()
{return (_Classification_ == ASSIGNMENT);}

/**	Tests if the Parameter has a valid Value assigment.
<P>
	@return	true if the Parameter has a Value assignment.
	@see	#Is_Assignment()
*/
public boolean Has_Value ()
{return (Is_Assignment () && _Value_ != null && Value.Is_Value (_Value_));}

/**	Tests if the Parameter is classified as an aggregate.
<P>
	An aggregate Parameter does not necessarily have a aggregate list
	(it may be null). Aggregate classifications include
	{@link #GROUP <CODE>GROUP</CODE>}
		(and its alias <CODE>BEGIN_GROUP</CODE>),
	{@link #OBJECT <CODE>OBJECT</CODE>}
		(and its alias <CODE>BEGIN_OBJECT</CODE>),
	and the <CODE>END_GROUP</CODE> and <CODE>END_OBJECT</CODE> forms.
<P>
	@return	true if the Parameter has an aggregate classification.
	@see	#AGGREGATE
*/
public boolean Is_Aggregate ()
{return ((_Classification_ & AGGREGATE) != 0);}

/**	Tests if the Parameter has an aggregate list.
<P>
	The contents of the list are not checked (use {@link
	#Is_Parameter_List(Object) <CODE>Is_Parameter_List</CODE>} to check
	the list contents).
<P>
	@return	true if the Parameter has an aggregate list.
	@see	#Is_Aggregate()
*/
public boolean Has_List ()
{return (Is_Aggregate () && children != null);}

/**	Tests if the Parameter is classified as an aggregate beginning.
<P>
	An aggregate beginning is the Parameter that contains a list of
	Parameters (the aggregate list, which may be empty or null); i.e.
	it "begins" an aggregate list. Begin aggregate classifications include
	{@link #GROUP <CODE>GROUP</CODE>}
		(and its alias <CODE>BEGIN_GROUP</CODE>) and
	{@link #OBJECT <CODE>OBJECT</CODE>}
		(and its alias <CODE>BEGIN_OBJECT</CODE>).
<P>
	@return	true if the Parameter begins (contains) an aggregate.
	@see	#AGGREGATE
*/
public boolean Is_Begin_Aggregate ()
{return (((_Classification_ & AGGREGATE) != 0) &&
	((_Classification_ & END) == 0));}

/**	Tests if the Parameter is classified as an aggregate ending.
<P>
	An aggregate ending is the Parameter that ends a list of
	Parameters (the aggregate list). End aggregate classifications include
	{@link #GROUP <CODE>END_GROUP</CODE>} and
	{@link #OBJECT <CODE>END_OBJECT</CODE>}.
<P>
	@return	true if the Parameter ends an aggregate list.
	@see	#END_AGGREGATE
*/
public boolean Is_End_Aggregate ()
{return (((_Classification_ & AGGREGATE) != 0) &&
	((_Classification_ & END) != 0));}

/**	Tests if the Parameter is classified as an object aggregate.
<P>
	The {@link #OBJECT <CODE>OBJECT</CODE>} classification is
	identical to the {@link #OBJECT <CODE>BEGIN_OBJECT</CODE>}
	classification.
<P>
	@return	true if the Parameter is an object aggregate.
*/
public boolean Is_Object ()
{return (_Classification_ == OBJECT);}

/**	Tests if the Parameter is classified as beginning an object aggregate.
<P>
	The {@link #BEGIN_OBJECT <CODE>BEGIN_OBJECT</CODE>} classification
	is identical to the {@link #OBJECT <CODE>OBJECT</CODE>}
	classification.
<P>
	@return	true if the Parameter begins an object aggregate.
*/
public boolean Is_Begin_Object ()
{return (_Classification_ == BEGIN_OBJECT);}

/**	Tests if the Parameter is classified as ending an object aggregate list.
<P>
	@return	true if the Parameter ends an object aggregate.
	@see	#Is_End_Aggregate()
*/
public boolean Is_End_Object ()
{return (_Classification_ == END_OBJECT);}

/**	Tests if the Parameter is classified as a group aggregate.
<P>
	The {@link #GROUP <CODE>GROUP</CODE>} classification is
	identical to the {@link #OBJECT <CODE>BEGIN_GROUP</CODE>}
	classification.
<P>
	@return	true if the Parameter is a group aggregate.
*/
public boolean Is_Group ()
{return (_Classification_ == GROUP);}

/**	Tests if the Parameter is classified as beginning a group aggregate.
<P>
	The {@link #BEGIN_GROUP <CODE>BEGIN_GROUP</CODE>} classification
	is identical to the {@link #GROUP <CODE>GROUP</CODE>}
	classification.
<P>
	@return	true if the Parameter begins a group aggregate.
*/
public boolean Is_Begin_Group ()
{return (_Classification_ == BEGIN_GROUP);}

/**	Tests if the Parameter is classified as ending a group aggregate list.
<P>
	@return	true if the Parameter ends a group aggregate.
	@see	#Is_End_Aggregate()
*/
public boolean Is_End_Group ()
{return (_Classification_ == END_GROUP);}

/**	Tests if the Parameter is classified as <CODE>UNKNOWN</CODE>
	(an empty Parameter).
<P>
	@return	true if the Parameter is an <CODE>UNKNOWN</CODE> classification.
	@see	#UNKNOWN
*/
public boolean Is_Unknown ()
{return (_Classification_ == UNKNOWN);}


//	Static versions:

/**	Tests for the token classification.
<P>
	@param	class_code	The classification code int.
	@return	true if the class_code is for a token.
	@see	#TOKEN
*/
public static boolean Is_Token (int class_code)
{return ((class_code & TOKEN) != 0);}

/**	Tests for an <CODE>END</CODE> classification.
<P>
	@param	class_code	The classification code int.
	@return	true if the class_code is for an <CODE>END</CODE> classification.
	@see	#Is_End()
	@see	#END_AGGREGATE
	@see	#END_PVL
*/
public static boolean Is_End (int class_code)
{return ((class_code & END) != 0);}

/**	Tests for the <CODE>END_PVL</CODE> classification.
<P>
	@param	class_code	The classification code int.
	@return	true if the class_code is the <CODE>END_PVL</CODE> classification.
	@see	#END_PVL
*/
public static boolean Is_End_PVL (int class_code)
{return (class_code == END_PVL);}

/**	Tests for the assignment classification.
<P>
	@param	class_code	The classification code int.
	@return	true if the class_code is for an assignment.
	@see	#ASSIGNMENT
*/
public static boolean Is_Assignment (int class_code)
{return (class_code == ASSIGNMENT);}

/**	Tests for an aggregate classification.
<P>
	@param	class_code	The classification code int.
	@return	true if the class_code is for an aggregate classification.
	@see	#Is_Aggregate()
*/
public static boolean Is_Aggregate (int class_code)
{return ((class_code & AGGREGATE) != 0);}

/**	Tests for an aggregate beginning classification.
<P>
	@param	class_code	The classification code int.
	@return	true if the class_code is for an aggregate beginning.
	@see	#Is_Begin_Aggregate()
*/
public static boolean Is_Begin_Aggregate (int class_code)
{return (((class_code & AGGREGATE) != 0) &&
	((class_code & END) == 0));}

/**	Tests for an aggregate ending classification.
<P>
	@param	class_code	The classification code int.
	@return	true if the class_code is for the end of an aggregate list.
	@see	#Is_End_Aggregate()
	@see	#END_AGGREGATE
*/
public static boolean Is_End_Aggregate (int class_code)
{return (((class_code & AGGREGATE) != 0) &&
	((class_code & END) != 0));}

/**	Tests for the object aggregate classification.
<P>
	@param	class_code	The classification code int.
	@return	true if the class_code is for an object aggregate.
	@see	#Is_Object()
*/
public static boolean Is_Object (int class_code)
{return (class_code == OBJECT);}

/**	Tests fpr the begin object aggregate classification.
<P>
	@param	class_code	The classification code int.
	@return	true if the class_code is for a begin object aggregate.
	@see	#Is_Begin_Object()
*/
public static boolean Is_Begin_Object (int class_code)
{return (class_code == BEGIN_OBJECT);}

/**	Tests for the end object aggregate list classification.
<P>
	@param	class_code	The classification code int.
	@return	true if the class_code is for the end of an object aggregate list.
	@see	#Is_End_Object()
*/
public static boolean Is_End_Object (int class_code)
{return (class_code == END_OBJECT);}

/**	Tests for the group aggregate classification.
<P>
	@param	class_code	The classification code int.
	@return	true if the class_code is for a group aggregate.
	@see	#Is_Group()
*/
public static boolean Is_Group (int class_code)
{return (class_code == GROUP);}

/**	Tests for the begin group aggregate classification.
<P>
	@param	class_code	The classification code int.
	@return	true if the class_code is for a begin group aggregate.
	@see	#Is_Begin_Group()
*/
public static boolean Is_Begin_Group (int class_code)
{return (class_code == BEGIN_GROUP);}

/**	Tests for the end group aggregate list classification.
<P>
	@param	class_code	The classification code int.
	@return	true if the class_code is for the end of a group aggregate list.
	@see	#Is_End_Group()
*/
public static boolean Is_End_Group (int class_code)
{return (class_code == END_GROUP);}

/**	Tests for the <CODE>UNKNOWN</CODE> classification.
<P>
	@param	class_code	The classification code int.
	@return	true if the class_code is the <CODE>UNKNOWN</CODE> classification.
	@see	#UNKNOWN
*/
public static boolean Is_Unknown (int class_code)
{return (class_code == UNKNOWN);}

/*------------------------------------------------------------------------------
	Comments:
*/
/**	Gets the comments String.
<P>
	@return	The String of comments associated with the Parameter.
		This will be null if the Parameter has no comments.
*/
public String Comments ()
{return _Comments_;}

/**	Sets the comments String.
<P>
	@param	comments	The String to become the Parameter comments.
		This may be null to remove all comments.
	@return	This Parameter.
*/
public Parameter Comments
	(
	String		comments
	)
{
_Comments_ = comments;
return this;
}

/*------------------------------------------------------------------------------
	Parent:
*/
/**	Gets the Parameter that is this Parameter's parent.
<P>
	A parent Parameter is always an aggregate that contains this
	Parameter in its aggregate list.
<P>
	@return	The parent aggregate Parameter that contains this Parameter,
		or null if this Parameter has no parent.
*/
public Parameter Parent ()
{return (Parameter)getParent ();}

/*------------------------------------------------------------------------------
	Warning:
*/
/**	Gets the current warning status.
<P>
	When conditions are encountered that are unusual enough to warrant
	attention, but not an error condition that would prevent successful
	processing which would cause an exception to be thrown, a warning
	condition is registered. The warning is in the form of a
	PVL_Exception that was not thrown. The current warning status is
	either the {@link #First_Warning(boolean)
	<CODE>First_Warning</CODE>} or the {@link #Last_Warning(boolean)
	<CODE>Last_Warning</CODE>} since a {@link #Reset_Warning()
	<CODE>Reset_Warning</CODE>}.
<P>
	@return	The current warning status as a PVL_Exception object,
		or null if no warning condition is registered.
	@see	PVL_Exception
	@see	#First_Warning(boolean)
	@see	#Last_Warning(boolean)
	@see	#Reset_Warning()
*/
public PVL_Exception Warning ()
{
if (_Use_First_Warning_)
	return _First_Warning_;
return _Last_Warning_;
}

/**	Clears any warning status so that the <CODE>Warning</CODE> method
	will return null until the next warning condition occurs.
<P>
	@return	This Parameter.
	@see	#Warning()
*/
public Parameter Reset_Warning ()
{
_First_Warning_ = null;
_Last_Warning_ = null;
return this;
}

/**	Enables or disables returning the first warning that occurs as the
	current warning status.
<P>
	The first warning is one that occurs when the current warning
	status is null.
<P>
	@param	first	true to enable returning the first warning status;
		false to return the last warning that occurred as the current
		warning status.
	@return	This Parameter.
	@see	#Warning()
	@see	#First_Warning()
	@see	#Reset_Warning()
*/
public Parameter First_Warning
	(
	boolean		first
	)
{
_Use_First_Warning_ = first;
return this;
}

/**	Returns the first warning since the last <CODE>Reset_Warning</CODE>.
<P>
	@return	The first warning status as a PVL_Exception object,
		or null if no warning condition is registered.
	@see	PVL_Exception
	@see	#Warning()
	@see	#First_Warning(boolean)
	@see	#Reset_Warning()
*/
public PVL_Exception First_Warning ()
{return _First_Warning_;}

/**	Enables or disables returning the last warning that occurs as the
	current warning status.
<P>
	The last warning is the most recent one regarless of any previous
	warning conditions that may have occured without an intervening
	<CODE>Reset_Warning</CODE>.
<P>
	@param	last	true to enable returning the last warning status;
		false to return the first warning condition that occurred as
		the current warning status.
	@return	This Parameter.
	@see	#Warning()
	@see	#Last_Warning()
	@see	#Reset_Warning()
*/
public Parameter Last_Warning
	(
	boolean		last
	)
{
_Use_First_Warning_ = ! last;
return this;
}

/**	Returns the last warning since a <CODE>Reset_Warning</CODE>.
<P>
	@return	The last warning status as a PVL_Exception object,
		or null if no warning condition is registered.
	@see	PVL_Exception
	@see	#Warning()
	@see	#Last_Warning(boolean)
	@see	#Reset_Warning()
*/
public PVL_Exception Last_Warning ()
{return _Last_Warning_;}

/*==============================================================================
	General Methods
*/
/**	Makes a deep copy of the Parameter.
<P>
	@return	An Object of class Parameter that is a copy of this Parameter.
	@see	#Parameter(Parameter)
	@see	Object#clone()
	@see	Cloneable
*/
public Object clone ()
{
//	Make a deep copy.
try {return new Parameter (this);}
catch (PVL_Exception execption) {return null;}
}

/**	Tests if a Parameter matches this Parameter using the specified
	criteria.
<P>
	@param	parameter	The Parameter to be compared to this Parameter.
	@param	criteria	The Selector providing the comparison criteria.
	@return	true if there is a criteria match; false otherwise.
	@see	Selector
	@see	Selection
*/
public boolean Match
	(
	Parameter	parameter,
	Selector	criteria
	)
{
if ((DEBUG & DEBUG_MATCH) != 0)
	System.out.println
		(">>> Parameter.Match: " + Name () + " == " + parameter);
boolean
	match = criteria.Parameters_Match (this, parameter);
if ((DEBUG & DEBUG_MATCH) != 0)
	System.out.println
		("<<< Parameter.Match: " + match);
return match;
}

/**	Tests if a Parameter matches this Parameter using the specified
	criteria.
<P>
	If the Parameter is an Aggregate matching is done recursively on all
	Parameters of the Aggregate. The match succeeds if all Parameters match.
<P>
	<B>N.B.</B>: To avoid unnecessary redundant recursive comparisons of
	Aggregate parameter values if the {@link Selector#Value(boolean)
	Value} criteria is included in the selection the {@link
	Selector#Parameter_Values_Match(Parameter, Parameter) data
	comparison} is only done for non-Aggregate Parameters.
<P>
	@param	parameter	The Parameter to be compared to this Parameter.
	@param	criteria	The Selector providing the comparison criteria.
	@return	true if there is a criteria match; false otherwise.
	@see	Selector
	@see	Selection
*/
public boolean Match_Depth
	(
	Parameter	parameter,
	Selector	criteria
	)
{
boolean
	matches,
	match_data = criteria.Value ();
if (match_data)
	//	Defer Data_Match to non-Array values.
	criteria.Value (false);

boolean
	end[] = {false};
matches = match_depth (parameter, criteria, match_data, end);

if (match_data)
		criteria.Value (true);
return matches;
}


private boolean match_depth
	(
	Parameter	parameter,
	Selector	criteria,
	boolean		match_data,
	boolean[]	end
	)
{
if ((DEBUG & DEBUG_MATCH) != 0)
	System.out.println
		(">>> Parameter.match_depth:\n"
		+"         this - " + this.Name () + '\n'
		+"    parameter - " + parameter.Name () + '\n'
		+"    " + criteria + '\n'
		+"    match_data - " + match_data);
if (parameter == null ||
	! criteria.Parameters_Match (this, parameter))
	{
	if ((DEBUG & DEBUG_MATCH) != 0)
		System.out.println
			("<<< Parameter.match_depth: false");
	return false;
	}

boolean
	matches = true;
if (     this.Is_Aggregate () &&
	parameter.Is_Aggregate ())
	{
	if ((DEBUG & DEBUG_MATCH) != 0)
		System.out.println
			("    Comparing Aggregate Lists ...");
	Iterator
		these_parameters,
		those_parameters;
	if (criteria.Specific ())
		{
		//	Compare the entire lists.
		if (     this.List_Size () !=
			parameter.List_Size ())
			{
			if ((DEBUG & DEBUG_MATCH) != 0)
				System.out.println
					("    this.List_Size " + this.List_Size ()
						+ " != parameter.List_Size " + parameter.List_Size () + '\n'
					+"<<< Parameter.match_depth: false");
			return false;
			}
		try
			{
			Vector
				this_list =      this.List (),
				that_list = parameter.List ();
			if (this_list == null &&
				that_list == null)
				//	Empty lists.
				{
				if ((DEBUG & DEBUG_MATCH) != 0)
					System.out.println
						("    Empty lists.\n"
						+"<<< Parameter.match_depth: true");
				return true;
				}
			these_parameters = this_list.iterator ();
			those_parameters = that_list.iterator ();
			}
		catch (PVL_Exception exception)
			{
			//	Shouldn't happen since they're both valid aggregates.
			if ((DEBUG & DEBUG_MATCH) != 0)
				System.out.println
					("    PVL_Exception!\n"
					+"<<< Parameter.match_depth: false");
			return false;
			}
		}
	else
		{
		/*	END_XXX parameters will end the lists.

			END_PVL parameters will be specially noticed and the end flag
			will be set so that no further comparisons will be done up the
			recursion line. 
		*/
		these_parameters =      this.iterator ();
		those_parameters = parameter.iterator ();
		}

	while (these_parameters.hasNext () &&
		   those_parameters.hasNext ())
		{
		if (! ((Parameter)these_parameters.next ()).match_depth
			  ((Parameter)those_parameters.next (),
			  	criteria, match_data, end))
			{
			matches = false;
			break;
			}
		if (end[0])
			//	END_PVL in sub-parameter.
			break;
		}
	if (! end[0] &&
		! criteria.Specific ())
		{
		if (these_parameters.hasNext () ||
			those_parameters.hasNext ())
			//	The lists are not the same size.
			{
			if ((DEBUG & DEBUG_MATCH) != 0)
				System.out.println
					("    The lists are not the same size");
			matches = false;
			}

		//	Test for END_PVL in either list.
		try
			{
			these_parameters.next ();
			those_parameters.next ();
			}
		catch (NoSuchElementException exception)
			{
			if (exception.getMessage ().startsWith
				(Parameter.Classification_Name (Parameter.END_PVL) + "\n"))
				//	END_PVL in list.
				end[0] = true;
			}
		}
	}
else if (match_data)
	{
	if ((DEBUG & DEBUG_MATCH) != 0)
		System.out.println
			("    Comparing Assignment Values ...");
	if (this.Data () == null)
		{
		if ((DEBUG & DEBUG_MATCH) != 0)
			System.out.println
				("    this Data is null\n"
				+"    parameter Data is "
					+ ((parameter.Data () != null) ? "not " : "") + "null\n"
				+"<<< Parameter.match_depth: " + (parameter.Data () == null));
		return (parameter.Data () == null);
		}
	if (parameter.Data () == null)
		{
		if ((DEBUG & DEBUG_MATCH) != 0)
			System.out.println
				("    this Data is not null\n"
				+"    parameter Data is null\n"
				+"<<< Parameter.match_depth: false");
		return false;
		}
	try {matches = this.Value ().Match_Depth (parameter.Value (), criteria);}
	catch (PVL_Exception exception)
		{
		//	Shouldn't happen since they're both valid assignments.
		matches = false;
		}
	}
if ((DEBUG & DEBUG_MATCH) != 0)
	System.out.println
		("<<< Parameter.match_depth: " + matches);
return matches;
}

/**	Tests if an Object is equal to this Parameter.
<P>
	An Object is equal to this Parameter when it is also a Parameter
	object and all of its contents are equal to this Parameter's contents.
	A Selection object with <CODE>PARAMETER_MATCH</CODE> criteria is
	used with the <CODE>Match_Depth</CODE> method.
<P>
	@param	object	The Object to be compared to this Parameter.
	@return	true if the object equals this Parameter, false otherwise.
	@see	#Match_Depth(Parameter, Selector)
	@see	Selection
	@see	Selector#PARAMETER_MATCH
*/
public boolean equals
	(
	Object		object
	)
{
if (! Is_Parameter (object))
	return false;
return Match_Depth
	((Parameter)object, new Selection (Selector.PARAMETER_MATCH));
}

/**	Tests if a Parameter is equivalent to this Parameter, ignoring
	certain cases where field values are not equal.
<P>
	Unlike <CODE>equals</CODE>, for a Parameter to be equivalent to this
	Parameter the contents need only be logically equivalent, rather
	than containing the identical contents. For example, shortening the
	parameter list of an Aggregate by setting to
	<CODE>END_AGGREGATE</CODE> the classification of the first entry
	that is different from the entries of another aggregate will result
	in the two aggregates being logically equivalent. A Selection object
	with <CODE>PARAMETER_MATCH</CODE> but <CODE>Specific (false)</CODE>
	criteria is used with the <CODE>Match_Depth</CODE> method.
<P>
	@param	parameter	The Parameter to be compared to this Parameter.
	@return	true if the Parameter is logically equivalent to this
		Parameter, false otherwise.
	@see	#Match_Depth(Parameter, Selector)
	@see	Selection
	@see	Selector#PARAMETER_MATCH
	@see	Selection#Specific(boolean)
*/
public boolean equalsIgnoreCase
	(
	Parameter	parameter
	)
{
return Match_Depth
	(parameter, new Selection (Selector.PARAMETER_MATCH).Specific (false));
}

/*------------------------------------------------------------------------------
	Write:
*/
/**	Writes the Parameter in PVL syntax using a PVL Lister.
<P>
	@param	lister	The Lister to write the Parameter.
	@return	The total number of bytes written.
	@throws	PVL_Exception
		<DL>
		<DT><CODE>BAD_ARGUMENT</CODE>
			<DD>If an element in an aggregate list is not a Parameter.
		</DL>
	@throws	IOException	From the Lister's <CODE>Write</CODE> method.
	@see	Lister#Write(Parameter)
*/
public int Write
	(
	Lister			lister
	)
	throws PVL_Exception, IOException
{
if ((DEBUG & DEBUG_WRITE) != 0)
	System.out.println (">>> Parameter.Write");
int
	total = lister.Write (this);
Warning (lister.Warning ());
if ((DEBUG & DEBUG_WRITE) != 0)
	System.out.println ("<<< Parameter Write: total written = " + total);
return total;
}

/**	Writes the Parameter in PVL syntax to an OutputStream.
<P>
	A Lister is created with the output stream as its list
	destination using specified indent level and syntax strictness.
<P>
	@param	output	The OutputStream to receive what is written.
		If output is null <CODE>System.out</CODE> is used.
	@param	level	The indent level at which to start the listing.
		If the level is negative, indenting is disabled.
	@param	strict	Use strict PVL syntax if true; otherwise use an
		easier to read format.
	@return	The total number of bytes written.
	@throws	PVL_Exception
		<DL>
		<DT><CODE>BAD_ARGUMENT</CODE>
			<DD>If an element in an aggregate list is not a Parameter.
		</DL>
	@throws	IOException	From the Lister's <CODE>Write</CODE> method.
	@see	Lister#Write(Parameter)
*/
public int Write
	(
	OutputStream	output,
	int				level,
	boolean			strict
	)
	throws PVL_Exception, IOException
{
return Write
	(new Lister (output)
		.Indent_Level (level)
		.Indent_Arrays ((level < 0) ? false : true)
		.Strict (strict));
}

/**	Writes the Parameter in PVL syntax to <CODE>System.out</CODE>
	starting at level 0, not strict.
<P>
	@return	The total number of bytes written.
	@throws	PVL_Exception
		<DL>
		<DT><CODE>BAD_ARGUMENT</CODE>
			<DD>If an element in an aggregate list is not a Parameter.
		</DL>
	@throws	IOException	From the OutputStream <CODE>write</CODE> method.
	@see	#Write(OutputStream, int, boolean)
*/
public int Write ()
	throws PVL_Exception, IOException
{return Write (System.out, 0, false);}

/**	Writes the Parameter in PVL syntax to <CODE>System.out</CODE>
	with or without indenting, not strict.
<P>
	@param	indent	Enable indenting starting at level 0 if true;
		disable indenting otherwise.
	@return	The total number of bytes written.
	@throws	PVL_Exception
		<DL>
		<DT><CODE>BAD_ARGUMENT</CODE>
			<DD>If an element in an aggregate list is not a Parameter.
		</DL>
	@throws	IOException	From the OutputStream <CODE>write</CODE> method.
	@see	#Write(OutputStream, int, boolean)
*/
public int Write
	(
	boolean			indent
	)
	throws PVL_Exception, IOException
{return Write (System.out, (indent ? 0 : -1), false);}

/**	Writes the Parameter in PVL syntax to output <CODE>System.out</CODE>
	with indenting and strict PVL enabled or disabled.
<P>
	@param	indent	Enable indenting starting at level 0 if true;
		disable indenting otherwise.
	@param	strict	Use strict PVL syntax if true; otherwise use an
		easier to read format.
	@return	The total number of bytes written.
	@throws	PVL_Exception
		<DL>
		<DT><CODE>BAD_ARGUMENT</CODE>
			<DD>If an element in an aggregate list is not a Parameter.
		</DL>
	@throws	IOException	From the OutputStream <CODE>write</CODE> method.
	@see	#Write(OutputStream, int, boolean)
*/
public int Write
	(
	boolean			indent,
	boolean			strict
	)
	throws PVL_Exception, IOException
{return Write (System.out, (indent ? 0 : -1), strict);}

/**	Writes the Parameter in PVL syntax to the output starting at level
	0, not strict.
<P>
	@param	output	The OutputStream to receive what is written.
	@return	The total number of bytes written.
	@throws	PVL_Exception
		<DL>
		<DT><CODE>BAD_ARGUMENT</CODE>
			<DD>If an element in an aggregate list is not a Parameter.
		</DL>
	@throws	IOException	From the OutputStream <CODE>write</CODE> method.
	@see	#Write(OutputStream, int, boolean)
*/
public int Write
	(
	OutputStream	output
	)
	throws PVL_Exception, IOException
{return Write (output, 0, false);}

/**	Writes the Parameter in PVL syntax to the output with or without
	indenting, not strict.
<P>
	@param	output	The OutputStream to receive what is written.
	@param	indent	Enable indenting starting at level 0 if true;
		disable indenting otherwise.
	@return	The total number of bytes written.
	@throws	PVL_Exception
		<DL>
		<DT><CODE>BAD_ARGUMENT</CODE>
			<DD>If an element in an aggregate list is not a Parameter.
		</DL>
	@throws	IOException	From the OutputStream <CODE>write</CODE> method.
	@see	#Write(OutputStream, int, boolean)
*/
public int Write
	(
	OutputStream	output,
	boolean			indent
	)
	throws PVL_Exception, IOException
{return Write (output, (indent ? 0 : -1), false);}

/**	Writes the Parameter in PVL syntax to the output with indenting and
	strict PVL enabled or disabled.
<P>
	@param	output	The OutputStream to receive what is written.
	@param	indent	Enable indenting starting at level 0 if true;
		disable indenting otherwise.
	@param	strict	Use strict PVL syntax if true; otherwise use an
		easier to read format.
	@return	The total number of bytes written.
	@throws	PVL_Exception
		<DL>
		<DT><CODE>BAD_ARGUMENT</CODE>
			<DD>If an element in an aggregate list is not a Parameter.
		</DL>
	@throws	IOException	From the OutputStream <CODE>write</CODE> method.
	@see	#Write(OutputStream, int, boolean)
*/
public int Write
	(
	OutputStream	output,
	boolean			indent,
	boolean			strict
	)
	throws PVL_Exception, IOException
{return Write (output, (indent ? 0 : -1), strict);}

/*------------------------------------------------------------------------------
	Iterators:
*/
/**	Provides an Iterator for the aggregate list positioned so
	that the next element returned will be at the specified
	index of the list.
<P>
	This Iterator does not descend into the lists of aggregate
	Parameters in the current list. The {@link DefaultMutableTreeNode
	DefaultMutableTreeNode} base class provdes various Enumeration
	objects that will traverse the hierarchy in different orders.
<P>
	In addition to implementing the Iterator interface, the
	implementation for a Parameter treats <CODE>END</CODE> Parameters
	as the end of the list. The <CODE>hasNext</CODE> method will return
	false when the next Parameter in the list has an <CODE>END</CODE>
	classification. To distinguish this case from the usual case when
	the end of the list has been reached, the <CODE>next</CODE> method
	will include this information when it throws NoSuchElementException
	at the end of the list. If the end of the list is as result of an
	<CODE>END</CODE> Parameter, then the <CODE>getMessage</CODE> method
	for the exception will contain the {@link #Classification_Name()
	Classification_Name} as the first line (String terminated by a
	new-line ('\n') character) of the message String. Otherwise the
	message will only contain the ID of the Parameter class module (in
	all cases a non-null message String will be provided). For example:
<P><PRE>
Iterator list = aggregate.iterator ();
while (list.hasNext ())
&nbsp;&nbsp;&nbsp;&nbsp;{
&nbsp;&nbsp;&nbsp;&nbsp;Parameter parameter = (Parameter)list.next ();
&nbsp;&nbsp;&nbsp;&nbsp;// Do something with the parameter.
&nbsp;&nbsp;&nbsp;&nbsp;}
try {list.next ();}
catch (NoSuchElementException exception)
&nbsp;&nbsp;&nbsp;&nbsp;{
&nbsp;&nbsp;&nbsp;&nbsp;if (exception.getMessage ().startsWith
&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;(Parameter.Classification_Name (Parameter.END)))
&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;{
&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;if (exception.getMessage ().startsWith
&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;(Parameter.Classification_Name (Parameter.END_PVL) + '\n'))
&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;// The END_PVL Parameter was encountered.
&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;else
&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;// An END_AGGREGATE was encountered.
&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;}
&nbsp;&nbsp;&nbsp;&nbsp;}
</PRE><P>
	A starting index beyond an <CODE>END</CODE> Parameter may be
	specified to skip over it.
<P>
	For non-aggregate Parameters an "empty" Iterator will be provided.
	This will act the same as an Iterator on an empty list.
<P>
	@param	index	The starting list element.
	@return	An Iterator object.
	@see	Iterator
	@see	#Classification_Name()
	@see	DefaultMutableTreeNode
*/
public Iterator iterator
	(
	int			index
	)
{
if (Is_Aggregate ())
	return new Parameter_Iterator (index);
return EMPTY_LIST_ITERATOR;
}

/**	Provides an Iterator starting at the first element of the list.
<P>
	@return	An Iterator object.
	@see	#iterator(int)
*/
public Iterator iterator ()
{return iterator (0);}

/**	Provides a ListIterator for the aggregate list positioned so
	that the next element returned will be at the specified
	index of the list.
<P>
	The functionality of the Parameter Iterator is provided (which
	it extends) with the ListIterator interface implemented.
<P>
	As with the Parameter Iterator, <CODE>END</CODE> Parameters are
	treated as the end of the list. During reverse iteration
	(<CODE>hasPrevious</CODE> and <CODE>previous</CODE>) <CODE>END</CODE>
	Parameters mark an artificial beginning of the list, when the
	start index is positioned beyond one. In this case the same special
	message is provided with the NoSuchElementException object as for
	the <CODE>next</CODE> method.
<P>
	<B>Note</B>: When the Parameter is an aggregate, but has no list,
	an empty one is provided so the <CODE>add</CODE> method will be
	able to insert new Parameters into the list.
<P>
	@param	index	The starting list element.
	@return	A ListIterator object.
	@see	#iterator()
	@see	ListIterator
*/
public ListIterator listIterator
	(
	int			index
	)
{
if (Is_Aggregate ())
	return new Parameter_ListIterator (index);
return EMPTY_LIST_ITERATOR;
}

/**	Provides a ListIterator starting at the first element of the list.
<P>
	@return	A ListIterator object.
	@see	#listIterator(int)
*/
public ListIterator listIterator ()
{return listIterator (0);}

/*.............................................................................
*/
private class Parameter_Iterator
	implements Iterator
{
ListIterator
	list_iterator;
Parameter
	last = null;
String
	next_end = null,
	previous_end = null;


private Parameter_Iterator ()
{this (0);}


private Parameter_Iterator
	(
	int			index
	)
{
if (children == null)
	children = new Vector ();	//	May need to add to this aggregate.
list_iterator = children.listIterator (index);
}


public boolean hasNext ()
{
if (list_iterator.hasNext () && next_end == null)
	{
	//	Don't iterate over an END Parameter.
	Parameter
		parameter = (Parameter)children.get (list_iterator.nextIndex ());
	if (! parameter.Is_End ())
		return true;
	next_end = parameter.Classification_Name ();
	}
return false;
}


public Object next ()
{
if (hasNext ())
	{
	previous_end = null;
	return (last = (Parameter)list_iterator.next ());
	}
if (next_end != null)
	/*	Start the exception message with the END_XXX classification name.

		>>> WARNING <<< This is the only way to tell that the
		NoSuchElementException was thrown due to an END_XXX Parameter
		and which specific classification.
	*/
	throw new NoSuchElementException (next_end + "\n" + ID);
throw new NoSuchElementException (ID);
}


public void remove ()
{
if (last != null)
	{
	list_iterator.remove ();
	last.parent = null;
	last = null;
	return;
	}
throw new IllegalStateException (ID);
}
}	//	End of Parameter_Iterator class


private class Parameter_ListIterator
	extends Parameter_Iterator
	implements ListIterator
{
private Parameter_ListIterator
	(
	int			index
	)
{super (index);}


public boolean hasPrevious ()
{
if (list_iterator.hasPrevious ())
	{
	//	Don't iterate over an END Parameter.
	Parameter
		parameter = (Parameter)children.get (list_iterator.previousIndex ());
	if (! parameter.Is_End ())
		return true;
	previous_end = parameter.Classification_Name ();
	}
return false;
}


public Object previous ()
{
if (hasPrevious ())
	{
	next_end = null;
	return (last = (Parameter)list_iterator.previous ());
	}
if (previous_end != null)
	throw new NoSuchElementException (previous_end + "\n" + ID);
throw new NoSuchElementException (ID);
}


public int nextIndex ()
{return list_iterator.nextIndex ();}


public int previousIndex ()
{return list_iterator.previousIndex ();}


public void set
	(
	Object		object
	)
{
if (! Is_Parameter (object))
	throw new ClassCastException
		(ID + "\nThe Object to set is not a Parameter.");
if (last != null)
	{
	list_iterator.set (object);
	((Parameter)object).parent = last.parent;
	last.parent = null;
	last = (Parameter)object;
	return;
	}
throw new IllegalStateException (ID);
}


public void add
	(
	Object		object
	)
{
if (! Is_Parameter (object))
	throw new ClassCastException
		(ID + "\nThe Object to set is not a Parameter.");
list_iterator.add (object);
((Parameter)object).parent = Parameter.this;
last = null;
}
}	//	End of Parameter_ListIterator class


static final ListIterator
	EMPTY_LIST_ITERATOR = new ListIterator ()
{
public boolean hasNext ()
{return false;}

public int nextIndex ()
{return 0;}

public Object next ()
{throw new NoSuchElementException (ID);}

public boolean hasPrevious ()
{return false;}

public int previousIndex ()
{return -1;}

public Object previous ()
{throw new NoSuchElementException (ID);}

public void remove ()
{throw new UnsupportedOperationException ();}

public void set (Object object)
{throw new UnsupportedOperationException ();}

public void add (Object object)
{throw new UnsupportedOperationException ();}
};	//	End of Null_Iterator class

/*------------------------------------------------------------------------------
	Find:
*/
/**	Searches for a Parameter that matches a test Parameter according
	the Selector criteria, optionally after some specific Parameter
	object has been located first.
<P>
	A search is made of the Parameters contained in this aggregate,
	and, recursively, any aggregates it contains. The search is
	depth-wise: each aggregate Parameter is searched when it is
	encountered.
<P>
	The search must first find the last_parameter (having the same
	object reference) before proceeding with the search for the
	test_parameter beginning with the Parameter following the
	last_parameter. If the last_parameter is null, then the search for
	the test_parameter begins with the first Parameter contained in
	this aggregate.
<P>
	The search compares each Parameter encountered against the
	test_parameter using the Selector criteria's
	<CODE>Parameters_Match</CODE> method. The first Parameter that
	matches the test_parameter is returned. If the last_parameter or
	a match for the test_parameter is not found, null is returned. If the
	last_parameter could not be found a
	<CODE>PVL_Exception.NO_LAST_LOCATION</CODE> warning status is
	registered with this Parameter. If this Parameter in which to
	<CODE>Find</CODE> is not an aggregate, null will be returned and a
	<CODE>PVL_Exception.ILLEGAL_SYNTAX</CODE> warning status will be
	set.
<P>
	The test_parameter may be any parameter, pre-existing or created ad
	hoc as a template for the Selector criteria. A Selector offers
	numerous criteria that may be conditionally applied when comparing
	the test_parameter with prospective mathing parameters. <B>Note</B>:
	The PIRL.PVL package provides a Selection class that implements the
	Selector interface.
<P>
	In addition, when the Selector criteria enables {@link
	Selector#Name(boolean) Name} matching the search engine will
	provide pathname tracking. This allows the name of the
	test_parameter to be a full (absolute) or partial (relative)
	pathname to a Parameter, of the form:
<P><BLOCKQUOTE>
	[ <B>/</B> ] <B><I>Name</I></B> [ <B>/<I>Name</I></B> [...] ]
</BLOCKQUOTE><P>
	The name of the test_parameter may be a simple name; i.e. it contains
	no {@link #Path_Delimiter() <CODE>Path_Delimiter</CODE>} characters.
	In this case a Parameter will be selected when its name matches the
	test_parameter name entirely. A pathname beginning with a
	<CODE>Path_Delimiter</CODE> character is "absolute" in that a
	Parameter with a Name that matches the first segment of the pathname
	must be found in this Parameter's aggregate list - the root of the
	<CODE>Find</CODE> - before the next segment of the pathname will
	qualify for a match of a Parameter Name in the aggregate list of the
	first Parameter matched; etc. <B>Note</B>: The first name of an
	absolute pathname is not the name of this Parameter, it is the
	name of the Parameter to find in the root aggregate list. All Parameters
	in the pathname sequence must, of course, be aggregates except the
	last Parameter (which may be of any classification). This is the same
	as file pathnames in Unix. A pathname that does not begin with a
	<CODE>Path_Delimiter</CODE> is "relative" in that the first Name is
	searched for like a simple name and then the remainder of the path is
	searched for relative to the location of the first Parameter found.
	The remainder of the pathname after the first name is treated like an
	absolute pathname. 
<P>
	<B>N.B.</B>: The current {@link #Warning() warning status} is
	{@link #Reset_Warning() reset} before the search begins.
<P>
	@param	test_parameter	A Parameter object to use for testing
		Parameters in this aggregate for a match. If null, or its
		Name is null, null is immediately returned.
	@param	criteria		A Selector object providing the methods
		to determine if Parameters match.
	@param	last_parameter	A Parameter object to be found within the
		aggregate before search comparisons begin. If null, begin the
		search at the first element of the aggregate list.
	@return	The Parameter object that was found, or null if not found.
		If the Parameter in which to <CODE>Find</CODE> is not an
		aggregate, null will be returned and a
		<CODE>PVL_Exception.ILLEGAL_SYNTAX</CODE> warning status will
		be set. If the last_parameter could not be found a
		<CODE>PVL_Exception.NO_LAST_LOCATION</CODE> warning status is
		registered with this Parameter.
	@see	Selector
	@see	Selection#Parameters_Match(Parameter, Parameter)
	@see	#Path_Name()
*/
public Parameter Find
	(
	Parameter	test_parameter,
	Selector	criteria,
	Parameter	last_parameter
	)
{
if (! find_check (test_parameter))
	return null;

//	Non-null when searching for last_parameter.
Parameter[]
	Last_Parameter = {last_parameter};

/*	User specified END parameter flag.
	Used the same way as in the Write method.
*/
int[]
	end = {0};

if ((DEBUG & DEBUG_FIND) != 0)
	{
	System.out.println
		(
		">>> Parameter Find:\n"
		+ "  criteria = " + Integer.toString (criteria.Criteria (), 2) + "\n"
		+ "    Name           = " + criteria.Name () + "\n"
		+ "    Classification = " + criteria.Classification () + "\n"
		+ "    Comments       = " + criteria.Comments () + "\n"
		+ "    Value          = " + criteria.Value () + "\n"
		+ "    Data           = " + criteria.Data () + "\n"
		+ "    Type           = " + criteria.Type () + "\n"
		+ "    Base           = " + criteria.Base () + "\n"
		+ "    Units          = " + criteria.Units () + "\n"
		+ "    Specific       = " + criteria.Specific () + "\n"
		+ "    Pattern_Match  = " + criteria.Pattern_Match () + "\n"
		+ "    AND            = " + criteria.And ()
		);
	if (last_parameter != null)
		System.out.println
			("  last_parameter is \"" + last_parameter.Path_Name () + "\"");
	}
test_parameter = find (test_parameter, criteria, Last_Parameter, end);

if (test_parameter == null &&
	Last_Parameter[0] != null &&
	Warning () == null)
	Warning
		(
		PVL_Exception.NO_LAST_LOCATION,
		"Can't Find the last Parameter named \""
			+ last_parameter.Name () + "\"."
		);
return test_parameter;
}

/**	Finds the Parameter with the specified name, which may be a simple
	name or a pathname.
<P>
	The name string matching is case sensitive.
<P>
	@param	name	The name or pathname String to search for. If null,
		the empty String will be used.
	@return	The Parameter object that was found, or null if not found.
	@see	#Find(String, boolean, Parameter)
*/
public Parameter Find
	(
	String		name
	)
{return Find (name, true, null);}

/**	Finds the Parameter with the specified name, which may be a simple
	name or a pathname, beginning after the specified Parameter object
	within the aggregate.
<P>
	The name string matching is case sensitive.
<P>
	@param	name	The name or pathname String to search for. If null,
		the empty String will be used.
	@param	last_parameter	A Parameter object to be found within the
		aggregate before search comparisons begin. If null, begin the
		search at the first element of the aggregate list.
	@return	The Parameter object that was found, or null if not found.
	@see	#Find(String, boolean, Parameter)
*/
public Parameter Find
	(
	String		name,
	Parameter	last_parameter
	)
{return Find (name, true, last_parameter);}

/**	Finds the Parameter with the specified name, which may be a simple
	name or a pathname.
<P>
	@param	name	The name or pathname String to search for. If null,
		the empty String will be used.
	@param	case_sensitive	true for the case sensitive name matching;
		false for case insensitive name matching.
	@return	The Parameter object that was found, or null if not found.
	@see	#Find(String, boolean, Parameter)
*/
public Parameter Find
	(
	String		name,
	boolean		case_sensitive
	)
{return Find (name, case_sensitive, null);}

/**	Finds the Parameter with the specified name, which may be a simple
	name or a pathname, beginning after the specified Parameter object
	within the aggregate.
<P>
	@param	name	The name or pathname String to search for. If null,
		the empty String will be used.
	@param	case_sensitive	true for the case sensitive name matching;
		false for case insensitive name matching.
	@param	last_parameter	A Parameter object to be found within the
		aggregate before search comparisons begin. If null, begin the
		search at the first element of the aggregate list.
	@return	The Parameter object that was found, or null if not found.
	@see	#Find(Parameter, Selector, Parameter)
	@see	Selection#Name(boolean)
	@see	Selection#Specific(boolean)
*/
public Parameter Find
	(
	String		name,
	boolean		case_sensitive,
	Parameter	last_parameter
	)
{
return Find
		(
		new Parameter (name),
		new Selection ().Name (true).Specific (case_sensitive),
		last_parameter
		);
}

/**	Finds the Parameter having the specified classification.
<P>
	The classification matching is exact. For a general classification
	match - e.g. a <CODE>GROUP</CODE> will match with an
	<CODE>OBJECT</CODE> - use:
<P>
<PRE>Find (new Parameter ().Classification (classification),
&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;new Selection ().Classification (true).Specific (false));
</PRE><P>
	@param	classification	The classification code to search for.
	@return	The Parameter object that was found, or null if not found.
	@see	#Find(Parameter, Selector, Parameter)
*/
public Parameter Find
	(
	int			classification
	)
{return Find (classification, null);}

/**	Finds the Parameter having the specified classification, beginning
	after the specified Parameter object within the aggregate.
<P>
	@param	classification	The classification code to search for.
	@param	last_parameter	A Parameter object to be found within the
		aggregate before search comparisons begin. If null, begin the
		search at the first element of the aggregate list.
	@return	The Parameter object that was found, or null if not found.
	@see	#Find(int)
	@see	#Find(Parameter, Selector, Parameter)
*/
public Parameter Find
	(
	int			classification,
	Parameter	last_parameter
	)
{
return Find
	(
	new Parameter ().Classification (classification),
	new Selection ().Classification (true),
	last_parameter
	);
}

/**	Finds the Parameter having the specified pathname and classification.
<P>
	Parameter pathname matching is specific (case sensitive) but
	classification matching is non-specific (any Aggregate type).
<P>
	@param	name	The name or pathname String to search for. If null, the
		empty String will be used.
	@param	classification	The classification code to search for.
	@return	The Parameter object that was found, or null if not found.
	@see	#Find(String, int, Parameter)
*/
public Parameter Find
	(
	String		name,
	int			classification
	)
{return Find (name, classification, null);}

/**	Finds the Parameter having the specified pathname and classification,
	beginning after the specified Parameter object within the
	aggregate.
<P>
	Parameter pathname matching is specific (case sensitive) but
	classification matching is non-specific (any Aggregate type).
<P>
	@param	name	The name or pathname String to search for. If null, the
		empty String will be used.
	@param	classification	The classification code to search for.
	@param	last_parameter	A Parameter object to be found within the
		aggregate before search comparisons begin. If null, begin the
		search at the first element of the aggregate list.
	@return	The Parameter object that was found, or null if not found.
	@see	#Find(String)
	@see	#Find(int)
	@see	#Find(Parameter, Selector, Parameter)
*/
public Parameter Find
	(
	String		name,
	int			classification,
	Parameter	last_parameter
	)
{
Selector
	criteria = new Selection ()
				.Name (true)
				.Specific (true);
Parameter
	test_parameter = new Parameter (name).Classification (classification),
	selected_parameter;

while ((selected_parameter = Find (test_parameter, criteria, last_parameter))
			!= null)
	{
	if (criteria
			.Specific (false)
			.Classifications_Match (selected_parameter, test_parameter))
		break;
	last_parameter = selected_parameter;
	criteria.Specific (false);
	}
return selected_parameter;
}

/**	Finds the Parameter that completely matches a test Parameter.
<P>
	A complete match requires an exact match of the name (which may be
	a pathname), classification, comments, and Value when the Parameter
	is an assignment. The Value match requires that the data and its
	type (as well as the numeric base for integers), and any units
	String all be exactly equal. <B>Note</B>: Parameter contents, not
	Parameter object references, are compared.
<P>
	@param	test_parameter	The Parameter to use when testing for a
		match.
	@return	The Parameter object that was found, or null if not found.
	@see	#Find(Parameter, Selector, Parameter)
*/
public Parameter Find
	(
	Parameter	test_parameter
	)
{
return Find
	(
	test_parameter,
	new Selection ().Criteria (Selector.PARAMETER_MATCH),
	null
	);
}

/**	Finds the Parameter that completely matches a test Parameter,
	beginning after the specified Parameter object within the
	aggregate.
<P>
	@param	test_parameter	The Parameter to use when testing for a
		match.
	@param	last_parameter	A Parameter object to be found within the
		aggregate before search comparisons begin. If null, begin the
		search at the first element of the aggregate list.
	@return	The Parameter object that was found, or null if not found.
	@see	#Find(Parameter)
	@see	#Find(Parameter, Selector, Parameter)
*/
public Parameter Find
	(
	Parameter	test_parameter,
	Parameter	last_parameter
	)
{
return Find
	(
	test_parameter,
	new Selection ().Criteria (Selector.PARAMETER_MATCH),
	last_parameter
	);
}

/**	Finds the Parameter that matches a test Parameter according the
	Selector criteria.
<P>
	@param	test_parameter	The Parameter to use when testing for a
		match.
	@param	criteria		A Selector object providing the methods
		to determine if Parameters match.
	@return	The Parameter object that was found, or null if not found.
	@see	#Find(Parameter, Selector, Parameter)
*/
public Parameter Find
	(
	Parameter	test_parameter,
	Selector	criteria
	)
{return Find (test_parameter, criteria, null);}


private boolean find_check
	(
	Parameter	test_parameter
	)
{
//	So last parameter not found will be noticed.
Reset_Warning ();

if (! Is_Begin_Aggregate ())
	{
	Warning
		(
		PVL_Exception.ILLEGAL_SYNTAX,
		"Can't Find in the " + Classification_Name ()
		+ " Parameter named \"" + Name () + "\"."
		);
	return false;
	}

//	Guarantee that the test_parameter passed in will not be null.
if (test_parameter == null ||
	test_parameter.Name () == null ||
	children == null)
	return false;
return true;
}


private Parameter find
	(
	Parameter	Test_Parameter,
	Selector	Criteria,
	Parameter[]	Last_Parameter,
	int[]		end
	)
{
/*	An array is used for Last_Parameter to provide a pass-back value
	when the Last_Parameter[0] has been located.
*/
if ((DEBUG & DEBUG_FIND) != 0)
	System.out.println
		(">>> Parameter find: In \"" + Path_Name () + "\"");

Parameter
	parameter,
	found_parameter = null;
String
	selection_name = Test_Parameter.Name (),
	path = null,
	name = null;
boolean
	absolute_pathname = false;

if (Criteria.Name ())
	{
	/*	Pathname tracking.

		Find a Parameter with a name on the Test_Parameter's pathname.
		The name of the Test_Parameter is a pathname of the form:

			[/]Name[/Name[...]]

		The name of the Test_Parameter may be a simple name; i.e. it
		contains no _Path_Delimiter_ characters. A Parameter will be
		selected when its name matches the Test_Parameter name.

		A pathname beginning with a _Path_Delimiter_ is "absolute" in
		that a Parameter with the first Name must be found in the
		Parameter list of the root (top) Parameter before the next Name
		will qualify for a match in a Parameter aggregate in this list;
		etc. All Parameters in the pathname sequence must be aggregates
		except the last Parameter (which may be of any classification).
		This is the same as file pathnames in Unix.

		A pathname that does not begin with a _Path_Delimiter_ is
		"relative" in that the first Name is searched for like a simple
		name and then the remainder of the path is searched for
		relative to the location of the first Parameter found. The
		remainder of the path after the first name is an absolute
		pathname. 

		Note: The Test_Parameter Name is only null when just looking
		for the Last_Parameter while pathname tracking and this
		aggregate is off the potential path to the name.
	*/

	if ((name = selection_name) != null)
		{
		if (name.length () > 0 &&
			name.charAt (0) == _Path_Delimiter_)
			{
			//	Absolute pathname.
			name = name.substring (1);	//	Get the relative pathname,
			absolute_pathname = true;	//	but root it here.
			}
		/*
			Separate the name to find in the Parameter list of this
			aggregate from the remainder of the path, if any, to be
			passed on to any aggregate Parameters encountered in the
			aggregate list. The name is used in comparisons with the
			Name of each Parameter in the current aggregate list. The
			path, if any, is always an absolute pathname; i.e. it
			begins with the _Path_Delimiter_ that separated it from the
			leading name. If there is no path remaining then path will
			be null.
		*/
		int index = name.indexOf (_Path_Delimiter_);
		if (index >= 0)
			{
			//	There's another delimiter in the name.
			path = name.substring (index);		//	The remainder of the path.
			name = name.substring (0, index);	//	The first name.
			}
		}
	//	Set the Test_Parameter to the local name.
	Test_Parameter.set_name (name);

	if ((DEBUG & DEBUG_FIND) != 0)
		System.out.println
			(
			"Parameter find:\n"
			+ "  selection_name = " + selection_name + "\n"
			+ "  name = " + name + "\n"
			+ "  path = " + path + "\n"
			+ "  absolute_pathname = " + absolute_pathname
			);
	}
else
	//	For restoring the name after searching aggregates.
	name = selection_name;

/*	Search the Parameters in this aggregate.

	Note: The algorithm of this function searches depth-wise; i.e. each
	aggregate encountered in the Parameter list is searched
	(conditionally) when it is encountered.
*/
ListIterator
	parameters = children.listIterator ();
while (parameters.hasNext ())
	{
	if (end[0] < 0)
		break;	//	Abort due to END_PVL

	parameter = (Parameter)parameters.next ();
	if ((DEBUG & DEBUG_FIND) != 0) System.out.println
		("--- parameter \"" + parameter.Path_Name () + "\"");

	if (Last_Parameter[0] == null)
		{
		/*	Compare the Parameter with the Test_Parameter using the Criteria.

			Note that the Test_Parameter comparison only occurs after the
			Last_Parameter has been found (or isn't being used).

			Note that the Test_Parameter is that_parameter for the Criteria
			Selection. This is important when pattern matching is used,
			because the regex pattern must be in that_parameter.
		*/
		if (Criteria.Parameters_Match (parameter, Test_Parameter))
			{
			if ((DEBUG & DEBUG_FIND) != 0) System.out.println
				("Potential match");
			found_parameter = parameter;
			if ((Criteria.Parameters_Match () & Selector.NAME) != 0 &&
				path != null)
				{
				/*
					A Name match was selected. But the name has an additional
					path component to match. So this match is insufficient; the
					entire path need to be matched.
				*/
				if ((DEBUG & DEBUG_FIND) != 0) System.out.println
					("    More path: " + path);
				found_parameter = null;
				}
			else
				{
				if ((DEBUG & DEBUG_FIND) != 0) System.out.println
					("    MATCH");
				break;	//	Found.
				}
			}
		}
	else
		{
		//	Just searching for the Last_Parameter.
		if (parameter == Last_Parameter[0])
			Last_Parameter[0] = null;	//	Found it.
		if ((DEBUG & DEBUG_FIND) != 0) System.out.println
			("Last_Parameter " +
				((Last_Parameter[0] == null) ? "======= found" : "not found"));
		}

	if (parameter.Is_End ())
		{
		/*	User specified end of the Parameter list.

			Note that matching against this END Parameter is allowed,
			but pathname tracking beyond this point is not.
		*/
		if (parameter.Is_End_PVL ())
			/*	User specified end of all Parameters.

				Hit the abort button. This Parameter effectively marks
				the end of all Parameters regardless of what may exist
				at higer levels. 
			*/
			end[0] = -1;
		else
			end[0] = 1;
		break;
		}

	Aggregate_Search:
	if (parameter.Is_Begin_Aggregate () &&
		parameter.List_Size () != 0)
		{
		if ((DEBUG & DEBUG_FIND) != 0) System.out.println
			("Aggregate search ---");
		/*	Aggregate search.

			The_Parameter is an aggregate and a decision needs to be
			made on whether to extend the search into it's Parameter
			list and, if so, how to conduct the search there.

			There are two possible cases for searching into an
			aggregate Parameter list: a) pathname tracking mode and b)
			all others. The inclusion of pathname tracking when
			searching for a Parameter name complicates the logic, which
			would ordinarily be to always extend the search into
			aggregates. Only pathname tracking must be evaluated
			specially.
		*/
		if (Criteria.Name ())
			{
			/*	Pathname tracking.

				If the search in this aggregate fails and the
				Test_Parameter specified a relative pathname, then the
				search will be retried with the full Test_Parameter
				Name. In this case the state of the Last_Parameter
				search will need to be restored to its state before
				searching in the aggregate.
			*/
			String
				pathname = path;
			Parameter
				last_parameter = Last_Parameter[0];

			if (name != null)
				{
				/*	We've been passed a name to find so we're on the path.
					Is the aggregate Parameter also on the path?
				*/
				if (! Criteria.Names_Match (parameter, Test_Parameter))
					//	The aggregate is not on the path.
					pathname = null;
				}
			if ((DEBUG & DEBUG_FIND) != 0) System.out.println
				("  Tracking pathname \"" + pathname + "\"\n"
				+ "  Looking for Last_Parameter? " + (last_parameter != null)); 

			if (Last_Parameter[0] != null ||
				pathname != null)
				{
				/*	Search this aggregate for the absolute pathname.

					This search is always done when still looking for
					the Last_Parameter or when the aggregate is still
					on the path.

					This search doesn't happen if the Last_Parameter
					has already been found and this aggregate is off
					the path; i.e if we're in the Test_Parameter criteria
					phase and there's no pathname to find, then this
					search won't happen. In this case the aggregate
					search will be done below instead.
				*/
				Test_Parameter.set_name (pathname);
				if ((DEBUG & DEBUG_FIND) != 0)
					System.out.println
						("\n  Searching for pathname \"" + pathname
						+ "\" in \"" + parameter.Path_Name () + "\"");
				found_parameter =
					parameter.find
						(
						Test_Parameter,
						Criteria,
						Last_Parameter,
						end
						);
				if (found_parameter != null)
					break;	//	Found a selection.
				Test_Parameter.set_name (name);
				if ((DEBUG & DEBUG_FIND) != 0)
					System.out.println
						("\n  Resuming search for selection_name \"" + selection_name
						+ "\" in \"" + Path_Name () + "\"");

				if (Last_Parameter[0] == null)
					{
					//	The Last_Parameter has been found.
					if ((DEBUG & DEBUG_FIND) != 0)
						System.out.println
							("  Last_Parameter was found.");
					if (name == null)
						{
						//	We're not on the path here.
						if (Criteria.And () ||
							Criteria.Parameter_Criteria () == Selector.NAME)
							/*
								A pathname match is required but it can't be
								found in the current Parameter list.
							*/
							break;
						/*
							There's no selection_name so there's no reason to
							retry the search of the aggregate.
						*/
						break Aggregate_Search;
						}
					if (absolute_pathname)
						/*
							The search will only be repeated when the selection
							name is a relative path and the Last_Parameter has
							been found.
						*/
						break Aggregate_Search;
					else
						/*	Restore the Last_Parameter.

							Note: For a relative pathname the search will be
							retried using the full selection_name, so the state
							of the Last_Parameter from before the aggregate
							search needs to remain in effect.
						*/
						Last_Parameter[0] = last_parameter;
					}
				else
					/*
						It's no use retrying in any case since the
						Last_Parameter couldn't be found.
					*/
					break Aggregate_Search;
				}	//	End of pathname search in the aggregate.
			}	//	End of pathname tracking.

		if (! absolute_pathname)
			{
			//	Search the aggregate using the full selection_name.
			Test_Parameter.set_name (selection_name);
			if ((DEBUG & DEBUG_FIND) != 0)
				System.out.println
					("\n  Searching for selection_name \"" + selection_name
					+ "\" in \"" + parameter.Path_Name () + "\"");
			found_parameter =
				parameter.find
					(
					Test_Parameter,
					Criteria,
					Last_Parameter,
					end
					);
			if (found_parameter != null)
				break;	//	Found a criteria.
			Test_Parameter.set_name (name);
			if ((DEBUG & DEBUG_FIND) != 0)
				System.out.println
					("\n  Resuming search for selection_name \"" + selection_name
					+ "\" in \"" + Path_Name () + "\"");
			}

		}	//	End of aggregate search.
	}	//	End of Parameter list.

//	Restore the full selection name.
Test_Parameter.set_name (selection_name);

if ((DEBUG & DEBUG_FIND) != 0)
	System.out.println
		("<<< Parameter find: found_parameter is " + (found_parameter != null));
return found_parameter;
}

/*==============================================================================
	Methods that override base class methods:
*/
/*	Errors are reported as a Warning.

	It would be better if there were a way to tell the caller directly
	about problems. Unfortunately, the flawed design of the base class
	methods does not provide any means to safely report back to the
	caller, not even a function return value.
*/
/**	Overrides the base class method to ensure Parameter integrity.
<P>
	@param	object	Since the <CODE>userObject</CODE> of a Parameter is
		it's name, only String objects are allowed for the argument.
		Anything else is ignored and a
		<CODE>PVL_Exception.BAD_ARGUMENT</CODE> warning status is set.
	@see	DefaultMutableTreeNode#setUserObject(Object)
*/
public void setUserObject
	(
	Object		object
	)
{
if (object instanceof String)
	userObject = object;
else
	Warning
		(
		PVL_Exception.BAD_ARGUMENT,
		"Can't setUserObject of " + Classification_Name ()
			+ " Parameter named \"" + Name () + "\"\n"
		+ "  to " + object.getClass ().getName () + " object.\n"
		+ "  Use the Name (String) method."
		);
}

/**	Overrides the base class method to ensure Parameter integrity.
<P>
	@param	allows	Since the ability of a Parameter to have children
		is totally determined by its classification (only aggregates
		may have a children list), this argument is ignored and a
		<CODE>PVL_Exception.ILLEGAL_SYNTAX</CODE> warning status is
		set.
	@see	DefaultMutableTreeNode#setAllowsChildren(boolean)
*/
public void setAllowsChildren
	(
	boolean		allows
	)
{
Warning
	(
	PVL_Exception.ILLEGAL_SYNTAX,
	"Can't setAllowsChildren of " + Classification_Name ()
		+ " Parameter named \"" + Name () + "\"\n"
	+ "  This is determined by the Parameter's Classification."
	);
}

/**	Invokes the Parameter's <CODE>Add</CODE> method.
<P>
	@param	parameter	A MutableTreeNode that must be a Parameter object.
	@throws	IllegalArgumentException	If the parameter argument is not
		a Parameter object, or the <CODE>Add</CODE> method threw a
		PVL_Exception (in which case it's message is used for the
		exception message).
	@see	#Add(Parameter)
	@see	DefaultMutableTreeNode#add(MutableTreeNode)
*/
public void add
	(
	MutableTreeNode	parameter
	)
	throws IllegalArgumentException
{
if (! Is_Parameter (parameter))
	throw new IllegalArgumentException
		(
		ID + "\n"
		+ "The argument to add is not a Parameter."
		);
try {Add ((Parameter)parameter);}
catch (PVL_Exception exception)
	{throw new IllegalArgumentException (exception.getMessage ());}
}

/**	Invokes the Parameter's <CODE>Insert</CODE> method.
<P>
	@param	parameter	A MutableTreeNode that must be a Parameter object.
	@param	index		The index in the aggregate list where the Parameter
		is to be inserted.
	@throws	IllegalArgumentException	If the parameter argument is not
		a Parameter object, or the <CODE>Insert</CODE> method threw a
		PVL_Exception (in which case it's message is used for the
		exception message).
	@see	#Insert(Parameter, int)
	@see	DefaultMutableTreeNode#insert(MutableTreeNode, int)
*/
public void insert
	(
	MutableTreeNode	parameter,
	int				index
	)
	throws IllegalArgumentException
{
if (! Is_Parameter (parameter))
	throw new IllegalArgumentException
		(
		ID + "\n"
		+ "The argument to insert is not a Parameter."
		);
try {Insert ((Parameter)parameter, index);}
catch (PVL_Exception exception)
	{throw new IllegalArgumentException (exception.getMessage ());}
}

/**	Invokes the Parameter's <CODE>Remove</CODE> method.
<P>
	@param	index	The index of the element in the aggregate list to remove.
	@see	#Remove(int)
	@see	DefaultMutableTreeNode#remove(int)
*/
public void remove
	(
	int			index
	)
{Remove (index);}

/**	Invokes the Parameter's <CODE>Remove</CODE> method.
<P>
	@param	parameter	A MutableTreeNode that must be a Parameter object.
	@throws	IllegalArgumentException	If the parameter argument is not
		a Parameter object.
	@see	#Remove(Parameter)
	@see	DefaultMutableTreeNode#remove(MutableTreeNode)
*/
public void remove
	(
	MutableTreeNode	parameter
	)
	throws IllegalArgumentException
{
if (! Is_Parameter (parameter))
	throw new IllegalArgumentException
		(
		ID + "\n"
		+ "The argument to remove is not a Parameter."
		);
Remove ((Parameter)parameter);
}

/*------------------------------------------------------------------------------
*/
private void Warning
	(
	String		description,
	String		explanation
	)
{
Warning
	(
	new PVL_Exception
		(
		ID,
		description,
		explanation
		)
	);
}


private void Warning
	(
	PVL_Exception	warning
	)
{
if (warning != null)
	{
	_Last_Warning_ = warning;
	if (_First_Warning_ == null)
		_First_Warning_ = _Last_Warning_;
	}
}


}	//	End of class
