/**
 * @brief Low level backlight driver for ATI X1600 graphics card.
 *
 * This module contains the low level driver to control a LCD backlight
 * connected to the ATI X1600 graphics card. This card is used in the
 * new MacBook Pros from Apple.
 
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * version 2 as published by the Free Software Foundation
 * (http://www.gnu.org/licenses/gpl.html)
 *
 * @file    src/driver_backlight_x1600.c
 * @author  Matthias Grimm <matthias.grimm@users.sourceforge.net>
 * @author  Stefan Bruda <bruda@cs.ubishops.ca> 
 * @author  Nicolas Boichat <nicolas@boichat.ch>
 * @author  Julien Blache <jb@jblache.org>
 */

#ifdef HAVE_CONFIG_H
#  include <config.h>
#endif

#include <stdlib.h>
#include <sys/mman.h>

#include <pci/pci.h>

#include <stdio.h>
#include <stdlib.h>
#include <fcntl.h>
#include <errno.h>
#include <string.h>
#include <unistd.h>

#include "gettext_macros.h"
#include "class_backlight.h"
#include "driver_backlight_x1600.h"

/**
 * @brief Address in memory of the ATI X1600 graphics card
 *
 */
static long address;

/**
 * @brief Length of memory segment the ATI X1600 graphics card uses
 */
static long length;

/**
 * @brief Exit function of the driver - cleans up all ressources
 *
 * This function will be called when the work is done and the driver
 * should be unloaded. It frees all allocated ressources.
 */
void
driver_backlight_x1600_exit ()
{
}

/**
 * @brief Get the current brightness level from the device
 *
 * <b>This function is part of the public interface of the driver.</b>
 *
 * It returns the current brightness level of the LCD backlight.
 *
 * @return  current brightness level or -1 on error
 */
int
x1600bl_get_brightness ()
{
	int fd, state, rc = -1;
	char *memory;

	if ((fd = open("/dev/mem", O_RDWR)) >= 0) {
		memory = mmap(NULL, length, PROT_READ|PROT_WRITE, MAP_SHARED, fd, address);
		if (memory != MAP_FAILED) {
			OUTREG(0x4dc, 0x00000005);
			state = INREG(0x7ae4);
			OUTREG(0x7ae4, state);
			rc = INREG(0x7af8) >> 8;  /* backlight value */
        	munmap(memory, length);
		}
		close(fd);
		rc = rc < X1600_BRIGHTNESS_MIN ? 0 : rc - (X1600_BRIGHTNESS_MIN - 1);
	}
	return rc;
}

/**
 * @brief Get the maximum brightness level the device supports
 *
 * <b>This function is part of the public interface of the driver.</b>
 *
 * It returns the maximum brightness level the LCD backlight
 * device supports. The number of steps is hardware dependent and
 * 256 for the ATi X1600 graphics card.
 *
 * @return  maximum supported brightness level of the device.
 */
int
x1600bl_get_brightness_max ()
{
	return X1600_BRIGHTNESS_MAX - X1600_BRIGHTNESS_MIN + 1;
}

/**
 * @brief Change the current brightness level.
 *
 * <b>This function is part of the public interface of the driver.</b>
 *
 * This function sets a new brightness level. The given level must be
 * valid for this device, that means it must be lower than the level
 * pmubl_get_brightness_max() returns. No further check is done on that.
 *
 * @param  val   new brightness level
 */
void
x1600bl_set_brightness (int val)
{
	int fd, state;
	char *memory;

	if (val >= 0) {
		if ((fd = open("/dev/mem", O_RDWR)) >= 0) {
			memory = mmap(NULL, length, PROT_READ|PROT_WRITE, MAP_SHARED, fd, address);
			if (memory != MAP_FAILED) {
				OUTREG(0x4dc, 0x00000005);
				state = INREG(0x7ae4);
				OUTREG(0x7ae4, state);
				val = val ? val + (X1600_BRIGHTNESS_MIN - 1) : 0;
				OUTREG(0x7af8, 0x00000001 | ((val & 0xFF) << 8));
        		munmap(memory, length);
			}
			close (fd);
		}
	}
}

static struct driver_backlight driver_backlight_x1600 = {
	.name               = N_("ATI X1600 Backlight Driver"),
	.get_brightness     = x1600bl_get_brightness,
	.get_brightness_max = x1600bl_get_brightness_max,
	.set_brightness     = x1600bl_set_brightness,
	.driver_exit        = driver_backlight_x1600_exit,
};

/**
 * @brief Constructor of a X1600 backlight driver object
 *
 * This function probes for a ATI X1600 graphics card on the
 * PCI bus and if one is found, saves the memory address and
 * the seqment length. The other functions of this driver rely
 * on this data to change the LCD brightness.
 *
 * @return  Initializes backlight driver object or NULL
 */
struct driver_backlight *
driver_backlight_x1600_init ()
{
	struct pci_access *pacc = pci_alloc();
	struct pci_dev *dev;

	address = 0;
	length = 0;

	pci_init(pacc);
	pci_scan_bus(pacc);

	for(dev=pacc->devices; dev; dev=dev->next) {  /* Iterate over all devices */
		pci_fill_info(dev, PCI_FILL_IDENT | PCI_FILL_BASES);
		if ((dev->vendor_id == PCI_ID_VENDOR_ATI) &&
			(dev->device_id == PCI_ID_PRODUCT_X1600)) {
			address = dev->base_addr[2];
			length = dev->size[2];
		}
	}
	pci_cleanup(pacc);

	if (!address)
		return NULL;

	return &driver_backlight_x1600;
}

