/** @file
	Parser: http support functions.

	Copyright (c) 2001-2017 Art. Lebedev Studio (http://www.artlebedev.com)
	Author: Alexandr Petrosian <paf@design.ru> (http://paf.design.ru)
 */

#include "pa_http.h"
#include "pa_common.h"
#include "pa_charsets.h"
#include "pa_request_charsets.h"
#include "pa_request.h"
#include "pa_vfile.h"
#include "pa_random.h"

volatile const char * IDENT_PA_HTTP_C="$Id: pa_http.C,v 1.79 2017/02/07 22:00:43 moko Exp $" IDENT_PA_HTTP_H; 

#ifdef _MSC_VER
#include <windows.h>
#else
#define closesocket close
#endif

// defines

#define HTTP_METHOD_NAME	"method"
#define HTTP_FORM_NAME	"form"
#define HTTP_BODY_NAME	"body"
#define HTTP_TIMEOUT_NAME	"timeout"
#define HTTP_HEADERS_NAME	"headers"
#define HTTP_FORM_ENCTYPE_NAME	"enctype"
#define HTTP_ANY_STATUS_NAME	"any-status"
#define HTTP_OMIT_POST_CHARSET_NAME	"omit-post-charset"	// ^file::load[...;http://...;$.method[post]] by default adds charset to content-type

#define HTTP_USER "user"
#define HTTP_PASSWORD "password"

#define HTTP_USER_AGENT "user-agent"
#define DEFAULT_USER_AGENT "parser3"

#ifndef INADDR_NONE
#define INADDR_NONE ((ulong) -1)
#endif

#undef CRLF
#define CRLF "\r\n"

// helpers

bool ResponseHeaders::add_header(const char *line){
	const char *value=strchr(line, ':');

	if(value && value != line){ // we need only headers, not the response code
		Header header(str_upper(line, value-line), String::Body(value+1).trim(String::TRIM_BOTH, " \t\n\r"));

		if(header.name == String::Body(HTTP_CONTENT_TYPE_UPPER) && content_type.is_empty())
			content_type=header.value;

		if(header.name == String::Body("CONTENT-LENGTH") && content_length==0)
			content_length=pa_atoul(header.value.cstr(), 10);

		headers+=header;

		return true;
	}
	return false;
}

class Cookies_table_template_columns: public ArrayString {
public:
	Cookies_table_template_columns() {
		*this+=new String("name");
		*this+=new String("value");
		*this+=new String("expires");
		*this+=new String("max-age");
		*this+=new String("domain");
		*this+=new String("path");
		*this+=new String("httponly");
		*this+=new String("secure");
	}
};


static bool set_addr(struct sockaddr_in *addr, const char* host, const short port){
	memset(addr, 0, sizeof(*addr)); 
	addr->sin_family=AF_INET;
	addr->sin_port=htons(port); 
	if(host) {
		struct hostent *hostIP=gethostbyname(host);
		if(hostIP && hostIP->h_addrtype == AF_INET){
			memcpy(&addr->sin_addr, hostIP->h_addr, hostIP->h_length);
			return true;
		}
	}
	return false;
}

class HTTP_response {
public:
	char *buf;
	size_t length;
	size_t buf_size;
	size_t body_offset;

	ResponseHeaders headers;
	String &url;

	HTTP_response(String& aurl) : buf(NULL), length(0), buf_size(0), body_offset(0), url(aurl){}

	void resize(size_t size){
		buf_size=size;
		buf=(char *)pa_realloc(buf, size + 1);
	}

	bool read(int sock, size_t size){
		if(length+size>buf_size)
			resize(buf_size*2 + size);
		ssize_t received_size=recv(sock, buf + length, size, 0);
		if(received_size==0)
			return false;
		if(received_size<0) {
			if(int no=pa_socks_errno())
				throw Exception("http.timeout", &url, "error receiving response body: %s (%d)", pa_socks_strerr(no), no);
			return false;
		}
		length+=received_size;
		buf[length]='\0';
		return true;
	}

	size_t status_size(){
		char *headers=strchr(buf, '\n');
		if(!headers)
			return false;

		return headers-buf;
	}

	const char *status_code(char *status_line, int &result){
		char* status_start = strchr(status_line, ' ');

		if(!(status_start++))
			return status_line;

		char* status_end=strchr(status_start, ' ');

		if(!status_end)
			return status_line;

		if(status_end==status_start)
			return status_line;

		const char *result_str=pa_strdup(status_start, status_end-status_start);
		result=atoi(result_str);
		return result_str;
	}

	bool body_start(){
		char *p=buf;
		while((p=strchr(p, '\n'))) {
			if(p[1]=='\r' && p[2]=='\n'){  // \r\n\r\n
				*p='\0';
				body_offset=p-buf+3;
				return true;
			}
			if(p[1]=='\n') { // \n\n
				*p='\0';
				body_offset=p-buf+2;
				return true;
			}
			p++;
		}
		return false;
	}

	void parse_headers(){
		const String header_block(buf, String::L_TAINTED);
		
		ArrayString aheaders;
		header_block.split(aheaders, 0, "\n");

		Array_iterator<const String*> i(aheaders);
		i.next(); // skipping status
		for(;i.has_next();){
			const char *line=i.next()->cstr();
			if(!headers.add_header(line))
				throw Exception("http.response", &url, "bad response from host - bad header \"%s\"", line);
		}
	}

};

enum HTTP_response_state {
	HTTP_STATUS_CODE,
	HTTP_HEADERS,
	HTTP_BODY
};

static int http_read_response(HTTP_response& response, int sock, bool fail_on_status_ne_200) {
	HTTP_response_state state=HTTP_STATUS_CODE;
	int result=0;

	size_t chunk_size=0x400*16;
	response.resize(2*chunk_size);

	while(response.read(sock, chunk_size)){
		switch(state){
			case HTTP_STATUS_CODE: {
				size_t status_size=response.status_size();
				if(!status_size)
					break;

				const char *status=response.status_code(pa_strdup(response.buf, status_size), result);

				if(!result || fail_on_status_ne_200 && result!=200)
					throw Exception("http.status", status ? new String(status) : &String::Empty, "invalid HTTP response status");

				state=HTTP_HEADERS;
			}

			case HTTP_HEADERS: {
				if(!response.body_start())
					break;

				response.parse_headers();

				size_t content_length=check_file_size(response.headers.content_length, response.url);
				if(content_length>0 && (content_length + response.body_offset) > response.length){
					response.resize(content_length + response.body_offset + 0x400*64);
				}

				state=HTTP_BODY;
				break;
			}

			case HTTP_BODY: {
				chunk_size=0x400*64;
				break;
			}
		}
	}

	if(state==HTTP_STATUS_CODE)
		throw Exception("http.response", &response.url, "bad response from host - no status found (size=%u)", response.length);

	if(state==HTTP_HEADERS){
		response.parse_headers();
		response.body_offset=response.length;
	}

	return result;
}

/* ********************** request *************************** */

#if defined(SIGALRM) && defined(HAVE_SIGSETJMP) && defined(HAVE_SIGLONGJMP)
#	define PA_USE_ALARM
#endif

#ifdef PA_USE_ALARM
static sigjmp_buf timeout_env;
static void timeout_handler(int /*sig*/){
	siglongjmp(timeout_env, 1); 
}
#endif

static int http_request(HTTP_response& response, const char* host, short port, const char* request, size_t request_size, int timeout_secs, bool fail_on_status_ne_200) {
	if(!host)
		throw Exception("http.host", 0, "zero hostname");  //never

	volatile // to prevent makeing it register variable, because it will be clobbered by longjmp [thanks gcc warning]
		int sock=-1;
#ifdef PA_USE_ALARM
	signal(SIGALRM, timeout_handler); 
#endif
#ifdef PA_USE_ALARM
	if(sigsetjmp(timeout_env, 1)) {
		// stupid gcc [2.95.4] generated bad code
		// which failed to handle sigsetjmp+throw: crashed inside of pre-throw code.
		// rewritten simplier [athough duplicating closesocket code]
		if(sock>=0) 
			closesocket(sock); 
		throw Exception("http.timeout", 0, "timeout occured while retrieving document"); 
		return 0; // never
	} else {
		alarm(timeout_secs); 
#endif
		try {
			int result;
			struct sockaddr_in dest;
		
			if(!set_addr(&dest, host, port))
				throw Exception("http.host", 0, "can not resolve hostname \"%s\"", host); 
			
			if((sock=socket(AF_INET, SOCK_STREAM, IPPROTO_TCP/*0*/))<0) {
				int no=pa_socks_errno();
				throw Exception("http.connect", 0, "can not make socket: %s (%d)", pa_socks_strerr(no), no); 
			}

			// To enable SO_DONTLINGER (that is, disable SO_LINGER) 
			// l_onoff should be set to zero and setsockopt should be called
			linger dont_linger={0,0};
			setsockopt(sock, SOL_SOCKET, SO_LINGER, (const char *)&dont_linger, sizeof(dont_linger));

#ifdef WIN32
// SO_*TIMEO can be defined in .h but not implemlemented in protocol,
// failing subsequently with Option not supported by protocol (99) message
// could not suppress that, so leaving this only for win32
			int timeout_ms=timeout_secs*1000;
			setsockopt(sock, SOL_SOCKET, SO_SNDTIMEO, (const char*)&timeout_ms, sizeof(timeout_ms));
			setsockopt(sock, SOL_SOCKET, SO_RCVTIMEO, (const char*)&timeout_ms, sizeof(timeout_ms));
#endif

			if(connect(sock, (struct sockaddr *)&dest, sizeof(dest))) {
				int no=pa_socks_errno();
				throw Exception("http.connect", 0, "can not connect to host \"%s\": %s (%d)", host, pa_socks_strerr(no), no);
			}

			if(send(sock, request, request_size, 0)!=(ssize_t)request_size) {
				int no=pa_socks_errno();
				throw Exception("http.timeout", 0, "error sending request: %s (%d)", pa_socks_strerr(no), no);
			}

			result=http_read_response(response, sock, fail_on_status_ne_200);
			closesocket(sock);
#ifdef PA_USE_ALARM
			alarm(0);
#endif
			return result;
		} catch(...) {
#ifdef PA_USE_ALARM
			alarm(0);
#endif
			if(sock>=0)
				closesocket(sock);
			rethrow;
		}
#ifdef PA_USE_ALARM
	}
#endif
}

#ifndef DOXYGEN
struct Http_pass_header_info {
	Request_charsets* charsets;
	String* request;
	bool* user_agent_specified;
	bool* content_type_specified;
	bool* content_type_url_encoded;
};
#endif

char *pa_http_safe_header_name(const char *name) {
	char *result=pa_strdup(name);
	char *n=result;
	if(!pa_isalpha((unsigned char)*n))
		*n++ = '_';
	for(; *n; ++n) {
		if (!pa_isalnum((unsigned char)*n) && *n != '-' && *n != '_')
			*n = '_';
	}
	return result;
}

static void http_pass_header(HashStringValue::key_type aname, 
				HashStringValue::value_type avalue, 
				Http_pass_header_info *info) {

	const char* name_cstr=aname.cstr();

	if(strcasecmp(name_cstr, HTTP_CONTENT_LENGTH)==0)
		return;

	String name=String(pa_http_safe_header_name(capitalize(name_cstr)), String::L_AS_IS);
	String value=attributed_meaning_to_string(*avalue, String::L_HTTP_HEADER, true);

	*info->request << name << ": " << value << CRLF;
	
	if(strcasecmp(name_cstr, HTTP_USER_AGENT)==0)
		*info->user_agent_specified=true;
	if(strcasecmp(name_cstr, HTTP_CONTENT_TYPE)==0){
		*info->content_type_specified=true;
		*info->content_type_url_encoded=pa_strncasecmp(value.cstr(), HTTP_CONTENT_TYPE_FORM_URLENCODED)==0;
	}
}

static void http_pass_cookie(HashStringValue::key_type name, 
				HashStringValue::value_type value, 
				Http_pass_header_info *info) {
	
	*info->request << String(name, String::L_HTTP_COOKIE) << "="
		<< attributed_meaning_to_string(*value, String::L_HTTP_COOKIE, true)
		<< "; "; 

}

static const String* basic_authorization_field(const char* user, const char* pass) {
	if(!user&& !pass)
		return 0;

	String combined;  
	if(user)
		combined<<user;
	combined<<":";
	if(pass)
		combined<<pass;
	
	String* result=new String("Basic ");
	*result<<pa_base64_encode(combined.cstr(), combined.length());
	return result;
}

static void form_string_value2string(HashStringValue::key_type key, const String& value, String& result) {
	result << String(key, String::L_URI) << "=" << String(value, String::L_URI) << "&";
}

#ifndef DOXYGEN
struct Form_table_value2string_info {
	HashStringValue::key_type key;
	String& result;

	Form_table_value2string_info(HashStringValue::key_type akey, String& aresult): 
		key(akey), result(aresult) {}
};
#endif
static void form_table_value2string(Table::element_type row, Form_table_value2string_info* info) {
	form_string_value2string(info->key, *row->get(0), info->result);
}

static void form_value2string(HashStringValue::key_type key, HashStringValue::value_type value, String* result) {
	if(const String* svalue=value->get_string())
		form_string_value2string(key, *svalue, *result);
	else if(Table* tvalue=value->get_table()) {
		Form_table_value2string_info info(key, *result);
		tvalue->for_each(form_table_value2string, &info);
	} else
		throw Exception(PARSER_RUNTIME, new String(key, String::L_TAINTED),
			"is %s, " HTTP_FORM_NAME " option value can be string or table only (file is allowed for $." HTTP_METHOD_NAME "[POST] + $." HTTP_FORM_ENCTYPE_NAME "[" HTTP_CONTENT_TYPE_MULTIPART_FORMDATA "])", value->type());
}

const char* pa_form2string(HashStringValue& form, Request_charsets& charsets) {
	String string;
	form.for_each<String*>(form_value2string, &string);
	return string.untaint_and_transcode_cstr(String::L_URI, &charsets);
}

struct FormPart {
	Request* r;
	const char* boundary;
	String* string;
	Form_table_value2string_info* info;

	struct BinaryBlock{
		const char* ptr;
		size_t length;

		BinaryBlock(String* astring, Request* r): ptr(astring->untaint_and_transcode_cstr(String::L_AS_IS, &r->charsets)), length(strlen(ptr)){}
		BinaryBlock(const char* aptr, size_t alength): ptr(aptr), length(alength){}
	};

	Array<BinaryBlock> blocks;

	FormPart(Request* ar, const char* aboundary): r(ar), boundary(aboundary), string(new String()){}

	const char *post(size_t &length){
		if(blocks.count()){
			blocks+=BinaryBlock(string, r);

			length=0;
			for(size_t i=0; i<blocks.count(); i++)
				length+=blocks[i].length;

			char *result=(char *)pa_malloc_atomic(length);
			char *ptr=result;

			for(size_t i=0; i<blocks.count(); i++){
				memcpy(ptr, blocks[i].ptr, blocks[i].length);
				ptr+=blocks[i].length;
			}

			return result;
		} else {
			BinaryBlock result(string, r);
			length=result.length;
			return result.ptr;
		}
	}
};

static void form_part_boundary_header(FormPart& part, String::Body name, const char* file_name=0) {
	*part.string << "--" << part.boundary << CRLF CONTENT_DISPOSITION_CAPITALIZED ": form-data; name=\"" << name << "\"";
	if(file_name){
		if(strcmp(file_name, NONAME_DAT)!=0)
			*part.string << "; filename=\"" << file_name << "\"";
		*part.string << CRLF HTTP_CONTENT_TYPE_CAPITALIZED ": " << part.r->mime_type_of(file_name);
	}
	*part.string << CRLF CRLF;
}

static void form_string_value2part(HashStringValue::key_type key, const String& value, FormPart& part) {
	form_part_boundary_header(part, key);
	*part.string << value << CRLF;
}

static void form_file_value2part(HashStringValue::key_type key, VFile& vfile, FormPart& part) {
	form_part_boundary_header(part, key, vfile.fields().get(name_name)->as_string().cstr());
	part.blocks+=FormPart::BinaryBlock(part.string, part.r);
	part.blocks+=FormPart::BinaryBlock(vfile.value_ptr(), vfile.value_size());
	part.string=new String();
	*part.string << CRLF;
}

static void form_table_value2part(Table::element_type row, FormPart* part) {
	form_string_value2part(part->info->key, *row->get(0), *part);
}

static void form_value2part(HashStringValue::key_type key, HashStringValue::value_type value, FormPart& part) {
	if(const String* svalue=value->get_string())
		form_string_value2part(key, *svalue, part);
	else if(Table* tvalue=value->get_table()) {
		Form_table_value2string_info info(key, *part.string);
		part.info = &info;
		tvalue->for_each(form_table_value2part, &part);
	} else if(VFile* vfile=static_cast<VFile *>(value->as("file"))){
		form_file_value2part(key, *vfile, part);
	} else
		throw Exception(PARSER_RUNTIME, new String(key, String::L_TAINTED), "is %s, " HTTP_FORM_NAME " option value can be string, table or file only", value->type());
}

const char* pa_form2string_multipart(HashStringValue& form, Request& r, const char* boundary, size_t& post_size){
	FormPart formpart(&r, boundary);
	form.for_each<FormPart&>(form_value2part, formpart);
	*formpart.string << "--" << boundary << "--";
	// @todo: return binary blocks here to save memory in pa_internal_file_read_http
	return formpart.post(post_size);
}

// Set-Cookie: name=value; Domain=docs.foo.com; Path=/accounts; Expires=Wed, 13-Jan-2021 22:23:01 GMT; Secure; HttpOnly
static ArrayString* parse_cookie(Request& r, const String& cookie) {
	char *current=pa_strdup(cookie.cstr());
	
	const String* name=0;
	const String* value=&String::Empty;
	const String* expires=&String::Empty;
	const String* max_age=&String::Empty;
	const String* path=&String::Empty;
	const String* domain=&String::Empty;
	const String* httponly=&String::Empty;
	const String* secure=&String::Empty;

	bool first_pair=true;

	do {
		if(char *meaning=search_stop(current, ';'))
			if(char *attribute=search_stop(meaning, '=')) {
				const String* sname=new String(unescape_chars(attribute, strlen(attribute), &r.charsets.source(), true/*don't convert '"' to space*/), String::L_TAINTED);
				const String* smeaning=0;
				if(meaning)
					smeaning=new String(unescape_chars(meaning, strlen(meaning), &r.charsets.source(), true/*don't convert '"' to space*/), String::L_TAINTED);

				if(first_pair) {
					// name + value
					name=sname;
					value=smeaning;
					first_pair=false;
				} else {
					const String& slower=sname->change_case(r.charsets.source(), String::CC_LOWER);

					if(slower == "expires")
						expires=smeaning;
					else if(slower == "max-age")
						max_age=smeaning;
					else if(slower == "domain")
						domain=smeaning;
					else if(slower == "path")
						path=smeaning;
					else if(slower == "httponly")
						httponly=new String("1", String::L_CLEAN);
					else if(slower == "secure")
						secure=new String("1", String::L_CLEAN);
					else {
						// todo@ ?
					}
				}
			}
	} while(current);

	if(!name)
		return 0;

	ArrayString* result=new ArrayString(8);
	*result+=name;
	*result+=value;
	*result+=expires;
	*result+=max_age;
	*result+=domain;
	*result+=path;
	*result+=httponly;
	*result+=secure;

	return result;
}

Table* parse_cookies(Request& r, Table *cookies){
	Table& result=*new Table(new Cookies_table_template_columns);

	for(Array_iterator<Table::element_type> i(*cookies); i.has_next(); )
		if(ArrayString* row=parse_cookie(r, *i.next()->get(0)))
			result+=row;

	return &result;
}

void tables_update(HashStringValue& tables, const String::Body name, const String& value){
	Table *table;
	if(Value *valready=tables.get(name)) {
		// second+ appearence
		table=valready->get_table();
	} else {
		// first appearence
		Table::columns_type columns=new ArrayString(1);
		*columns+=new String("value");
		table=new Table(columns);
		tables.put(name, new VTable(table));
	}
	// this string becomes next row
	ArrayString& row=*new ArrayString(1);
	row+=&value;
	*table+=&row;
}

/// @todo build .cookies field. use ^file.tables.SET-COOKIES.menu{ for now
File_read_http_result pa_internal_file_read_http(Request& r, const String& file_spec, bool as_text, HashStringValue *options, bool transcode_text_result) {
	File_read_http_result result;
	char host[MAX_STRING];
	const char *idna_host;
	const char* uri; 
	short port=80;
	const char* method="GET";
	bool method_is_get=true;
	HashStringValue* form=0;
	int timeout_secs=2;
	bool fail_on_status_ne_200=true;
	bool omit_post_charset=false;
	Value* vheaders=0;
	Value* vcookies=0;
	Value* vbody=0;
	Charset* asked_remote_charset=0;
	Charset* real_remote_charset=0;
	const char* user_cstr=0;
	const char* password_cstr=0;
	const char* encode=0;
	bool multipart=false;

	if(options) {
		int valid_options=pa_get_valid_file_options_count(*options);

		if(Value* vmethod=options->get(HTTP_METHOD_NAME)) {
			valid_options++;
			method=vmethod->as_string().change_case(r.charsets.source(), String::CC_UPPER).cstr();
			method_is_get=strcmp(method, "GET")==0;
		}
		if(Value* vencode=options->get(HTTP_FORM_ENCTYPE_NAME)) {
			valid_options++;
			encode=vencode->as_string().cstr();
		}
		if(Value* vform=options->get(HTTP_FORM_NAME)) {
			valid_options++;
			form=vform->get_hash(); 
		} 
		if(vbody=options->get(HTTP_BODY_NAME)) {
			valid_options++;
		} 
		if(Value* vtimeout=options->get(HTTP_TIMEOUT_NAME)) {
			valid_options++;
			timeout_secs=vtimeout->as_int(); 
		} 
		if(vheaders=options->get(HTTP_HEADERS_NAME)) {
			valid_options++;
		} 
		if(vcookies=options->get(HTTP_COOKIES_NAME)) {
			valid_options++;
		} 
		if(Value* vany_status=options->get(HTTP_ANY_STATUS_NAME)) {
			valid_options++;
			fail_on_status_ne_200=!vany_status->as_bool(); 
		}
		if(Value* vomit_post_charset=options->get(HTTP_OMIT_POST_CHARSET_NAME)){
			valid_options++;
			omit_post_charset=vomit_post_charset->as_bool();
		}
		if(Value* vcharset_name=options->get(PA_CHARSET_NAME)) {
			asked_remote_charset=&pa_charsets.get(vcharset_name->as_string());
		} 
		if(Value* vresponse_charset_name=options->get(PA_RESPONSE_CHARSET_NAME)) {
			valid_options++;
			real_remote_charset=&pa_charsets.get(vresponse_charset_name->as_string());
		} 
		if(Value* vuser=options->get(HTTP_USER)) {
			valid_options++;
			user_cstr=vuser->as_string().cstr();
		} 
		if(Value* vpassword=options->get(HTTP_PASSWORD)) {
			valid_options++;
			password_cstr=vpassword->as_string().cstr();
		}

		if(valid_options!=options->count())
			throw Exception(PARSER_RUNTIME, 0, CALLED_WITH_INVALID_OPTION);
	}
	if(!asked_remote_charset) // defaulting to $request:charset
		asked_remote_charset=&(r.charsets).source();

	if(encode){
		if(method_is_get)
			throw Exception(PARSER_RUNTIME, 0, "you can not use $." HTTP_FORM_ENCTYPE_NAME " option with method GET");

		multipart=strcasecmp(encode, HTTP_CONTENT_TYPE_MULTIPART_FORMDATA)==0;

		if(!multipart && strcasecmp(encode, HTTP_CONTENT_TYPE_FORM_URLENCODED)!=0)
			throw Exception(PARSER_RUNTIME, 0, "$." HTTP_FORM_ENCTYPE_NAME " option value can be " HTTP_CONTENT_TYPE_FORM_URLENCODED " or " HTTP_CONTENT_TYPE_MULTIPART_FORMDATA " only");
	}

	if(vbody){
		if(method_is_get)
			throw Exception(PARSER_RUNTIME, 0, "you can not use $." HTTP_BODY_NAME " option with method GET");

		if(form)
			throw Exception(PARSER_RUNTIME, 0, "you can not use options $." HTTP_BODY_NAME " and $." HTTP_FORM_NAME " together");
	}

	//preparing request
	String& connect_string=*new String(file_spec);

	const char* request;
	size_t request_size;
	{
		// influence URLencoding of tainted pieces to String::L_URI lang
		Temp_client_charset temp(r.charsets, *asked_remote_charset);

		const char* connect_string_cstr=connect_string.untaint_and_transcode_cstr(String::L_URI, &(r.charsets));

		const char* current=connect_string_cstr;
		if(strncmp(current, "http://", 7)!=0)
			throw Exception(PARSER_RUNTIME, &connect_string, "does not start with http://"); //never
		current+=7;

		strncpy(host, current, sizeof(host)-1);  host[sizeof(host)-1]=0;
		char* host_uri=lsplit(host, '/');
		uri=host_uri?current+(host_uri-1-host):"/";
		char* port_cstr=lsplit(host, ':');
		
		if (port_cstr){
			char* error_pos=0;
			port=(short)strtol(port_cstr, &error_pos, 10);
			if(port==0 || *error_pos)
				throw Exception(PARSER_RUNTIME, &connect_string, "invalid port number '%s'", port_cstr);
		}

		idna_host=pa_idna_encode(host, r.charsets.source());

		// making request head
		String head;
		head << method << " " << uri;
		if(method_is_get && form)
			head << (strchr(uri, '?')!=0?"&":"?") << pa_form2string(*form, r.charsets);

		head <<" HTTP/1.0" CRLF "Host: "<< idna_host;
		if (port != 80)
			head << ":" << port_cstr;
		head << CRLF;

		char* boundary= multipart ? get_uuid_boundary() : 0;

		String user_headers;
		bool user_agent_specified=false;
		bool content_type_specified=false;
		bool content_type_url_encoded=false;
		if(vheaders && !vheaders->is_string()) { // allow empty
			if(HashStringValue *headers=vheaders->get_hash()) {
				Http_pass_header_info info={
					&(r.charsets),
					&user_headers,
					&user_agent_specified,
					&content_type_specified,
					&content_type_url_encoded};
				headers->for_each<Http_pass_header_info*>(http_pass_header, &info); 
			} else
				throw Exception(PARSER_RUNTIME, 0, "headers param must be hash"); 
		};

		const char* request_body=0;
		size_t post_size=0;
		if(form && !method_is_get) {
			head << "Content-Type: " << (multipart ? HTTP_CONTENT_TYPE_MULTIPART_FORMDATA : HTTP_CONTENT_TYPE_FORM_URLENCODED);

			if(!omit_post_charset)
				head << "; charset=" << asked_remote_charset->NAME_CSTR();

			if(multipart) {
				head << "; boundary=" << boundary;
				request_body=pa_form2string_multipart(*form, r/*charsets & mime_type needed*/, boundary, post_size/*correct post_size returned here*/);
			} else {
				request_body=pa_form2string(*form, r.charsets);
				post_size=strlen(request_body);
			}
			head << CRLF;
		} else if(vbody) {
			// $.body was specified
			if(content_type_url_encoded){
				// transcode + url-encode
				request_body=vbody->as_string().untaint_and_transcode_cstr(String::L_URI, &(r.charsets));
			} else {
				// content-type != application/x-www-form-urlencoded -> transcode only, don't url-encode!
				const String &sbody=vbody->as_string();
				request_body=Charset::transcode(String::C(sbody.cstr(), sbody.length()), r.charsets.source(), *asked_remote_charset).str;
			}
			post_size=strlen(request_body);
		}

		// http://www.ietf.org/rfc/rfc2617.txt
		if(const String* authorization_field_value=basic_authorization_field(user_cstr, password_cstr))
			head << "Authorization: " << *authorization_field_value << CRLF;

		head << user_headers;

		if(!user_agent_specified) // defaulting
			head << "User-Agent: " DEFAULT_USER_AGENT CRLF;

		if(form && !method_is_get && content_type_specified) // POST + form + content-type was specified
			throw Exception(PARSER_RUNTIME, 0, "$.content-type can't be specified with method POST"); 

		if(vcookies && !vcookies->is_string()){ // allow empty
			if(HashStringValue* cookies=vcookies->get_hash()) {
				head << "Cookie: ";
				Http_pass_header_info info={&(r.charsets), &head, 0, 0, 0};
				cookies->for_each<Http_pass_header_info*>(http_pass_cookie, &info); 
				head << CRLF;
			} else
				throw Exception(PARSER_RUNTIME, 0, "cookies param must be hash");
		}

		if(request_body)
			head << "Content-Length: " << format(post_size, "%u") << CRLF;
		
		head << CRLF;
		
		const char *request_head=head.untaint_and_transcode_cstr(String::L_URI, &(r.charsets));

		if(request_body){
			size_t head_size = strlen(request_head);
			request_size=post_size + head_size;
			char *ptr=(char *)pa_malloc_atomic(request_size);
			memcpy(ptr, request_head, head_size);
			memcpy(ptr+head_size, request_body, post_size);
			request=ptr;
		} else {
			request_size=strlen(request_head);
			request=request_head;
		}
	}
	

	HTTP_response response(connect_string);

	// sending request
	int status_code=http_request(response, idna_host, port, request, request_size, timeout_secs, fail_on_status_ne_200);

	// processing results
	char* raw_body=response.buf + response.body_offset;
	size_t raw_body_size=response.length - response.body_offset;

	result.headers=new HashStringValue;
	VHash* vtables=new VHash;
	result.headers->put("tables", vtables);

	if (!real_remote_charset && !response.headers.content_type.is_empty())
		real_remote_charset=detect_charset(response.headers.content_type.cstr());

	if(as_text)
		real_remote_charset=pa_charsets.checkBOM(raw_body, raw_body_size, real_remote_charset);

	if (!real_remote_charset)
		real_remote_charset=asked_remote_charset; // never null

	for(Array_iterator<ResponseHeaders::Header> i(response.headers.headers); i.has_next(); ){
		ResponseHeaders::Header header=i.next();

		header.transcode(*real_remote_charset, r.charsets.source());

		String &header_value=*new String(header.value, String::L_TAINTED);

		tables_update(vtables->hash(), header.name, header_value);
		result.headers->put(header.name, new VString(header_value));
	}

	// filling $.cookies
	if(Value *vcookies=vtables->hash().get("SET-COOKIE"))
		result.headers->put(HTTP_COOKIES_NAME, new VTable(parse_cookies(r, vcookies->get_table())));

	// output response
	String::C real_body=String::C(raw_body, raw_body_size);

	if(as_text && transcode_text_result && raw_body_size) { // raw_body_size must be checked because transcode returns CONST string in case length==0, which contradicts hacking few lines below
		real_body=Charset::transcode(real_body, *real_remote_charset, r.charsets.source());
	}

	result.str=const_cast<char *>(real_body.str); // hacking a little
	result.length=real_body.length;

	if(as_text && result.length)
		fix_line_breaks(result.str, result.length);

	result.headers->put(file_status_name, new VInt(status_code));

	return result;
}
